<?php
// super_admin_dashboard.php
ob_start();
session_start();

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once 'includes/functions.php';
require_once 'includes/db_config.php';

// Session timeout
$timeout = 1800; // 30 minutes
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity']) > $timeout) {
    session_unset();
    session_destroy();
    redirect("login.php?timeout=1");
}
$_SESSION['last_activity'] = time();

// Restrict to super_admin role
if (!is_logged_in() || $_SESSION['user_role'] !== 'super_admin') {
    redirect("login.php");
}

// Generate CSRF token
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Handle admin approval/rejection
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && isset($_POST['user_id']) && isset($_POST['csrf_token']) && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
    $user_id = sanitize_input($_POST['user_id']);
    $action = sanitize_input($_POST['action']);

    if ($action === 'approve_admin' || $action === 'reject_admin') {
        $new_role = $action === 'approve_admin' ? 'admin' : 'rejected';
        $sql = "UPDATE users SET role = ? WHERE id = ? AND role = 'pending'";
        $stmt = mysqli_prepare($link, $sql);
        if ($stmt) {
            mysqli_stmt_bind_param($stmt, "si", $new_role, $user_id);
            mysqli_stmt_execute($stmt);
            $affected = mysqli_stmt_affected_rows($stmt);
            mysqli_stmt_close($stmt);

            if ($affected > 0) {
                $message = $action === 'approve_admin' ? "Admin approved successfully." : "Admin rejected.";
                // Send email notification
                $sql = "SELECT email, full_name FROM users WHERE id = ?";
                $stmt = mysqli_prepare($link, $sql);
                if ($stmt) {
                    mysqli_stmt_bind_param($stmt, "i", $user_id);
                    mysqli_stmt_execute($stmt);
                    $result = mysqli_stmt_get_result($stmt);
                    if ($user = mysqli_fetch_assoc($result)) {
                        $subject = $action === 'approve_admin' ? "Admin Approval" : "Admin Application Status";
                        $body = $action === 'approve_admin' ?
                            "Dear {$user['full_name']},<br>Your admin application has been approved. You can now access the admin dashboard." :
                            "Dear {$user['full_name']},<br>Your admin application has been rejected. Please contact support for more details.";
                        $email_result = send_mail($user['email'], $subject, $body);
                        if ($email_result !== true) {
                            error_log("Failed to send email to {$user['email']}: $email_result");
                        }
                    }
                    mysqli_free_result($result);
                    mysqli_stmt_close($stmt);
                }
            } else {
                $error = "Failed to update admin status.";
            }
        } else {
            $error = "Database error: " . mysqli_error($link);
            error_log("Failed to prepare statement in super_admin_dashboard.php: " . mysqli_error($link));
        }
    }
}

// Handle token purchase approval/rejection
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['purchase_action']) && isset($_POST['purchase_id']) && isset($_POST['csrf_token']) && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
    $purchase_id = sanitize_input($_POST['purchase_id']);
    $action = sanitize_input($_POST['purchase_action']);
    $currency_symbol = get_setting('currency_symbol', $link) ?? '₦';

    if ($action === 'approve_purchase' || $action === 'reject_purchase') {
        $new_status = $action === 'approve_purchase' ? 'approved' : 'rejected';
        $sql = "UPDATE token_purchases SET status = ?, approved_by = ?, approved_at = NOW() WHERE id = ? AND status = 'pending'";
        $stmt = mysqli_prepare($link, $sql);
        if ($stmt) {
            mysqli_stmt_bind_param($stmt, "sii", $new_status, $_SESSION['user_id'], $purchase_id);
            mysqli_stmt_execute($stmt);
            $affected = mysqli_stmt_affected_rows($stmt);
            mysqli_stmt_close($stmt);

            if ($affected > 0 && $new_status === 'approved') {
                // Credit tokens
                $sql = "INSERT INTO token_balances (user_id, balance, last_updated) SELECT user_id, quantity, NOW() FROM token_purchases WHERE id = ? ON DUPLICATE KEY UPDATE balance = balance + (SELECT quantity FROM token_purchases WHERE id = ?), last_updated = NOW()";
                $stmt = mysqli_prepare($link, $sql);
                if ($stmt) {
                    mysqli_stmt_bind_param($stmt, "ii", $purchase_id, $purchase_id);
                    mysqli_stmt_execute($stmt);
                    mysqli_stmt_close($stmt);

                    // Send email notification
                    $sql = "SELECT u.email, u.full_name, tp.quantity, tp.amount, tp.user_id FROM token_purchases tp JOIN users u ON tp.user_id = u.id WHERE tp.id = ?";
                    $stmt = mysqli_prepare($link, $sql);
                    if ($stmt) {
                        mysqli_stmt_bind_param($stmt, "i", $purchase_id);
                        mysqli_stmt_execute($stmt);
                        $result = mysqli_stmt_get_result($stmt);
                        if ($purchase = mysqli_fetch_assoc($result)) {
                            $subject = "Token Purchase Approved";
                            $body = "Dear {$purchase['full_name']},<br>Your purchase of {$purchase['quantity']} tokens for {$currency_symbol}" . number_format($purchase['amount'], 2) . " has been approved. Your new token balance is " . get_token_balance($link, $purchase['user_id']) . ".";
                            $email_result = send_mail($purchase['email'], $subject, $body);
                            if ($email_result !== true) {
                                error_log("Failed to send token purchase email to {$purchase['email']}: $email_result");
                            }
                        }
                        mysqli_free_result($result);
                        mysqli_stmt_close($stmt);
                    }
                    $message = "Token purchase approved and tokens credited.";
                } else {
                    $error = "Error crediting tokens: " . mysqli_error($link);
                    error_log("Failed to prepare token balance statement: " . mysqli_error($link) . " | Query: $sql");
                }
            } elseif ($affected > 0) {
                // Send rejection email
                $sql = "SELECT u.email, u.full_name, tp.quantity, tp.amount FROM token_purchases tp JOIN users u ON tp.user_id = u.id WHERE tp.id = ?";
                $stmt = mysqli_prepare($link, $sql);
                if ($stmt) {
                    mysqli_stmt_bind_param($stmt, "i", $purchase_id);
                    mysqli_stmt_execute($stmt);
                    $result = mysqli_stmt_get_result($stmt);
                    if ($purchase = mysqli_fetch_assoc($result)) {
                        $subject = "Token Purchase Rejected";
                        $body = "Dear {$purchase['full_name']},<br>Your purchase request for {$purchase['quantity']} tokens has been rejected. Please contact support for details.";
                        $email_result = send_mail($purchase['email'], $subject, $body);
                        if ($email_result !== true) {
                            error_log("Failed to send token rejection email to {$purchase['email']}: $email_result");
                        }
                    }
                    mysqli_free_result($result);
                    mysqli_stmt_close($stmt);
                }
                $message = "Token purchase rejected.";
            } else {
                $error = "Failed to update purchase status.";
                error_log("Failed to update token purchase status: no rows affected | Query: $sql | Purchase ID: $purchase_id");
            }
        } else {
            $error = "Database error: " . mysqli_error($link);
            error_log("Failed to prepare token purchase statement: " . mysqli_error($link) . " | Query: $sql");
        }
    }
    // Regenerate CSRF token
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
} elseif ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['purchase_action'])) {
    $error = "Invalid CSRF token.";
    error_log("CSRF token validation failed in super_admin_dashboard.php for purchase action");
}

// Fetch metrics
$currency_symbol = get_setting('currency_symbol', $link) ?? '₦';
$total_admins = get_total_users_by_role($link, 'admin');
$total_pending_admins = get_total_users_by_role($link, 'pending');
$sql = "SELECT COUNT(*) AS total FROM token_purchases WHERE status = 'pending'";
$result = mysqli_query($link, $sql);
$total_pending_purchases = $result ? mysqli_fetch_assoc($result)['total'] ?? 0 : 0;
if ($result) {
    mysqli_free_result($result);
}

// Calculate total tokens sold (sum of approved purchases by admin users)
$sql_tokens_sold = "SELECT COALESCE(SUM(tp.quantity), 0) AS total
                    FROM token_purchases tp
                    JOIN users u ON tp.user_id = u.id
                    WHERE tp.status = 'approved' AND u.role = 'admin'";
$result = mysqli_query($link, $sql_tokens_sold);
$total_tokens_sold = $result ? mysqli_fetch_assoc($result)['total'] ?? 0 : 0;
if ($result) {
    mysqli_free_result($result);
} else {
    error_log("Failed to fetch tokens sold: " . mysqli_error($link));
}

// Calculate total tokens used (from token_usage_logs)
$sql_tokens_used = "SELECT COALESCE(SUM(tokens_used), 0) AS total FROM token_usage_logs";
$result = mysqli_query($link, $sql_tokens_used);
$total_tokens_used = $result ? mysqli_fetch_assoc($result)['total'] ?? 0 : 0;
if ($result) {
    mysqli_free_result($result);
} else {
    error_log("Failed to fetch tokens used: " . mysqli_error($link));
}

// Calculate token income (sum of amounts from approved purchases by admin users)
$sql_token_income = "SELECT COALESCE(SUM(tp.amount), 0) AS total
                     FROM token_purchases tp
                     JOIN users u ON tp.user_id = u.id
                     WHERE tp.status = 'approved' AND u.role = 'admin'";
$result = mysqli_query($link, $sql_token_income);
$token_income = $result ? mysqli_fetch_assoc($result)['total'] ?? 0 : 0;
if ($result) {
    mysqli_free_result($result);
} else {
    error_log("Failed to fetch token income: " . mysqli_error($link));
}

ob_end_clean();
?>

<?php require_once 'includes/header.php'; ?>

<script>
// Set dynamic page title
document.title = 'Super Admin Dashboard - <?php echo htmlspecialchars(get_setting('system_name', $link) ?? 'Club Portal'); ?>';
</script>

<style>
.main-content {
    padding: 20px;
}
.card {
    background-color: #ffffff;
    border: none;
    border-radius: 15px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    transition: transform 0.3s ease, box-shadow 0.3s ease;
    opacity: 0;
    animation: fadeIn 0.5s ease-out forwards;
}
.card:hover {
    transform: scale(1.05);
    box-shadow: 0 8px 20px rgba(0, 0, 0, 0.2);
}
.card-icon {
    font-size: 2.5rem;
    color: #ffffff;
    padding: 15px;
    border-radius: 50%;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    transition: transform 0.3s ease;
}
.card:hover .card-icon {
    transform: rotate(10deg);
}
.card-title {
    font-size: 1.2rem;
    font-weight: bold;
    color: #333;
}
.card-text {
    font-size: 1.5rem;
    font-weight: bold;
    color: #333;
}
@keyframes fadeIn {
    from {
        opacity: 0;
        transform: translateY(10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}
/* Staggered animation delays for cards */
.card:nth-child(1) { animation-delay: 0.1s; }
.card:nth-child(2) { animation-delay: 0.2s; }
.card:nth-child(3) { animation-delay: 0.3s; }
.card:nth-child(4) { animation-delay: 0.4s; }
.card:nth-child(5) { animation-delay: 0.5s; }
.card:nth-child(6) { animation-delay: 0.6s; }
/* Icon background colors */
.card-total-admins .card-icon {
    background-color: #4b5563;
}
.card-pending-admins .card-icon {
    background-color: #f59e0b;
}
.card-pending-purchases .card-icon {
    background-color: #14b8a6;
}
.card-tokens-sold .card-icon {
    background-color: #22c55e;
}
.card-tokens-used .card-icon {
    background-color: #ef4444;
}
.card-token-income .card-icon {
    background-color: #06b6d4;
}
</style>

<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Super Admin Dashboard</h2>
            <hr>
            <?php if (isset($message)): ?>
                <div class="alert alert-success"><?php echo htmlspecialchars($message, ENT_QUOTES, 'UTF-8'); ?></div>
            <?php endif; ?>
            <?php if (isset($error)): ?>
                <div class="alert alert-danger"><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></div>
            <?php endif; ?>

            <!-- Metrics Overview -->
            <div class="row mb-4">
                <div class="col-md-4 mb-3">
                    <div class="card card-total-admins">
                        <div class="card-body text-center">
                            <i class="fas fa-users card-icon"></i>
                            <h5 class="card-title mt-3">Total Admins</h5>
                            <p class="card-text"><?php echo htmlspecialchars($total_admins, ENT_QUOTES, 'UTF-8'); ?></p>
                        </div>
                    </div>
                </div>
                <div class="col-md-4 mb-3">
                    <div class="card card-pending-admins">
                        <div class="card-body text-center">
                            <i class="fas fa-user-clock card-icon"></i>
                            <h5 class="card-title mt-3">Pending Admins</h5>
                            <p class="card-text"><?php echo htmlspecialchars($total_pending_admins, ENT_QUOTES, 'UTF-8'); ?></p>
                        </div>
                    </div>
                </div>
                <div class="col-md-4 mb-3">
                    <div class="card card-pending-purchases">
                        <div class="card-body text-center">
                            <i class="fas fa-coins card-icon"></i>
                            <h5 class="card-title mt-3">Pending Token Purchases</h5>
                            <p class="card-text"><?php echo htmlspecialchars($total_pending_purchases, ENT_QUOTES, 'UTF-8'); ?></p>
                        </div>
                    </div>
                </div>
                <div class="col-md-4 mb-3">
                    <div class="card card-tokens-sold">
                        <div class="card-body text-center">
                            <i class="fas fa-coins card-icon"></i>
                            <h5 class="card-title mt-3">Tokens Sold</h5>
                            <p class="card-text"><?php echo htmlspecialchars($total_tokens_sold, ENT_QUOTES, 'UTF-8'); ?></p>
                        </div>
                    </div>
                </div>
                <div class="col-md-4 mb-3">
                    <div class="card card-tokens-used">
                        <div class="card-body text-center">
                            <i class="fas fa-hand-holding-usd card-icon"></i>
                            <h5 class="card-title mt-3">Tokens Used</h5>
                            <p class="card-text"><?php echo htmlspecialchars($total_tokens_used, ENT_QUOTES, 'UTF-8'); ?></p>
                        </div>
                    </div>
                </div>
                <div class="col-md-4 mb-3">
                    <div class="card card-token-income">
                        <div class="card-body text-center">
                            <i class="fas fa-money-bill-wave card-icon"></i>
                            <h5 class="card-title mt-3">Token Income</h5>
                            <p class="card-text"><?php echo htmlspecialchars(format_currency($token_income, $currency_symbol), ENT_QUOTES, 'UTF-8'); ?></p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Approve Admins Section -->
            <div class="card mb-4">
                <div class="card-header">Approve New Admins</div>
                <div class="card-body">
                    <table class="table table-striped">
                        <thead>
                            <tr>
                                <th>Name</th>
                                <th>Email</th>
                                <th>Created At</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            $sql = "SELECT id, full_name, email, created_at FROM users WHERE role = 'pending' ORDER BY created_at DESC";
                            $result = mysqli_query($link, $sql);
                            $no_pending = true;
                            if ($result) {
                                while ($row = mysqli_fetch_assoc($result)) {
                                    $no_pending = false;
                                    echo "<tr>";
                                    echo "<td>" . htmlspecialchars($row['full_name'], ENT_QUOTES, 'UTF-8') . "</td>";
                                    echo "<td>" . htmlspecialchars($row['email'], ENT_QUOTES, 'UTF-8') . "</td>";
                                    echo "<td>" . htmlspecialchars($row['created_at'], ENT_QUOTES, 'UTF-8') . "</td>";
                                    echo "<td>
                                        <form method='POST' class='d-inline'>
                                            <input type='hidden' name='csrf_token' value='" . htmlspecialchars($_SESSION['csrf_token'], ENT_QUOTES, 'UTF-8') . "'>
                                            <input type='hidden' name='user_id' value='" . htmlspecialchars($row['id'], ENT_QUOTES, 'UTF-8') . "'>
                                            <button type='submit' name='action' value='approve_admin' class='btn btn-sm btn-success'>Approve</button>
                                            <button type='submit' name='action' value='reject_admin' class='btn btn-sm btn-danger'>Reject</button>
                                        </form>
                                    </td>";
                                    echo "</tr>";
                                }
                                mysqli_free_result($result);
                            } else {
                                error_log("Failed to fetch pending admins: " . mysqli_error($link));
                            }
                            ?>
                        </tbody>
                    </table>
                    <?php if ($no_pending): ?>
                        <p class="text-muted">No pending admin approvals.</p>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Manage Token Purchases Section -->
            <div class="card">
                <div class="card-header">Manage Token Purchases</div>
                <div class="card-body">
                    <table class="table table-striped">
                        <thead>
                            <tr>
                                <th>User</th>
                                <th>Quantity</th>
                                <th>Amount (<?php echo htmlspecialchars($currency_symbol, ENT_QUOTES, 'UTF-8'); ?>)</th>
                                <th>Receipt</th>
                                <th>Status</th>
                                <th>Date</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            $sql = "SELECT tp.id, u.full_name, tp.quantity, tp.amount, tp.receipt_proof, tp.status, tp.created_at 
                                    FROM token_purchases tp JOIN users u ON tp.user_id = u.id 
                                    WHERE tp.status = 'pending' ORDER BY tp.created_at DESC";
                            $result = mysqli_query($link, $sql);
                            $no_purchases = true;
                            if ($result) {
                                while ($row = mysqli_fetch_assoc($result)) {
                                    $no_purchases = false;
                                    echo "<tr>";
                                    echo "<td>" . htmlspecialchars($row['full_name'], ENT_QUOTES, 'UTF-8') . "</td>";
                                    echo "<td>" . htmlspecialchars($row['quantity'], ENT_QUOTES, 'UTF-8') . "</td>";
                                    echo "<td>" . htmlspecialchars(number_format($row['amount'], 2), ENT_QUOTES, 'UTF-8') . "</td>";
                                    echo "<td>" . ($row['receipt_proof'] ? "<a href='" . htmlspecialchars($row['receipt_proof'], ENT_QUOTES, 'UTF-8') . "' target='_blank'>View</a>" : "N/A") . "</td>";
                                    echo "<td>" . htmlspecialchars(ucfirst($row['status']), ENT_QUOTES, 'UTF-8') . "</td>";
                                    echo "<td>" . htmlspecialchars($row['created_at'], ENT_QUOTES, 'UTF-8') . "</td>";
                                    echo "<td>
                                        <form method='POST' class='d-inline'>
                                            <input type='hidden' name='csrf_token' value='" . htmlspecialchars($_SESSION['csrf_token'], ENT_QUOTES, 'UTF-8') . "'>
                                            <input type='hidden' name='purchase_id' value='" . htmlspecialchars($row['id'], ENT_QUOTES, 'UTF-8') . "'>
                                            <button type='submit' name='purchase_action' value='approve_purchase' class='btn btn-sm btn-success'>Approve</button>
                                            <button type='submit' name='purchase_action' value='reject_purchase' class='btn btn-sm btn-danger'>Reject</button>
                                        </form>
                                    </td>";
                                    echo "</tr>";
                                }
                                mysqli_free_result($result);
                            } else {
                                error_log("Failed to fetch token purchases: " . mysqli_error($link));
                                $error = "Error fetching token purchases: " . mysqli_error($link);
                            }
                            ?>
                        </tbody>
                    </table>
                    <?php if ($no_purchases): ?>
                        <p class="text-muted">No pending token purchases.</p>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Quick Actions -->
            <div class="card mb-4">
                <div class="card-header">Quick Actions</div>
                <div class="card-body">
                    <div class="d-flex flex-wrap gap-2">
                        <a href="admin_settings.php" class="btn btn-outline-primary"><i class="fas fa-cogs me-2"></i>System Settings</a>
                        <a href="add_admin.php" class="btn btn-outline-primary"><i class="fas fa-user-plus me-2"></i>Add Admin</a>
                        <a href="financial_reports.php" class="btn btn-outline-primary"><i class="fas fa-chart-bar me-2"></i>Financial Reports</a>
                        <a href="token_settings.php" class="btn btn-outline-primary"><i class="fas fa-coins me-2"></i>Token Settings</a>
                        <a href="token_store.php" class="btn btn-outline-primary"><i class="fas fa-store me-2"></i>Token Store</a>
                    </div>
                </div>
            </div>
        </main>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function () {
    // Debug dropdown initialization
    console.log('Initializing dropdowns in super_admin_dashboard.php');
    const dropdownElements = document.querySelectorAll('.dropdown-toggle');
    console.log('Found dropdown elements:', dropdownElements.length);
    dropdownElements.forEach((dropdown, index) => {
        console.log(`Initializing dropdown ${index}:`, dropdown);
        try {
            new bootstrap.Dropdown(dropdown, {
                popperConfig: {
                    strategy: 'fixed'
                }
            });
        } catch (e) {
            console.error('Error initializing dropdown:', e);
        }
    });
});
</script>

<?php require_once 'includes/footer.php'; ?>