<?php
// record_benefits.php
ob_start();
session_start();

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Enable mysqli exceptions
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

// Clear OPCache
if (function_exists('opcache_reset')) {
    opcache_reset();
}

require_once 'includes/functions.php';
require_once 'includes/db_config.php';

// Session timeout
$timeout = 1800; // 30 minutes
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity']) > $timeout) {
    session_unset();
    session_destroy();
    redirect("login.php?timeout=1");
}
$_SESSION['last_activity'] = time();

// Generate CSRF token
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Check permission
if (!is_logged_in() || !user_has_permission($link, 'view_benefits')) {
    redirect("login.php");
    exit;
}

$message = '';
$message_type = '';
$currency_symbol = get_setting('currency_symbol', $link) ?? '₦';
$token_cost = (int)get_setting('tokens_per_benefit', $link) ?: 1;

// Fetch system available tokens
$system_available_tokens = 0;
try {
    $sql_available_tokens = "SELECT COALESCE(SUM(balance), 0) AS total FROM token_balances";
    $result = mysqli_query($link, $sql_available_tokens);
    if ($result) {
        $system_available_tokens = mysqli_fetch_assoc($result)['total'] ?? 0;
        mysqli_free_result($result);
    }
} catch (Exception $e) {
    error_log("Failed to fetch system available tokens: " . $e->getMessage());
}

// Handle submission
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['log_benefit']) && isset($_POST['csrf_token']) && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
    $user_ids = $_POST['user_ids'] ?? [];
    $amount = floatval($_POST['amount'] ?? 0);
    $benefit_type_id = intval($_POST['benefit_type_id'] ?? 0);
    $description = sanitize_input($_POST['description'] ?? '');
    $deduct_per_member = (int)get_setting('deduct_per_member', $link) ?? 0;

    // Dynamic token calculation
    $num_logs = $deduct_per_member ? count($user_ids) : 1;
    $required_tokens = $token_cost * $num_logs;
    $current_balance = get_token_balance($link, $_SESSION['user_id']);
    if ($current_balance < $required_tokens) {
        $message = "Insufficient tokens. You need $required_tokens tokens, but you have $current_balance. <a href='token_store.php'>Please purchase more tokens</a>.";
        $message_type = 'danger';
    } else {
        if (!empty($user_ids) && $amount > 0 && $benefit_type_id > 0) {
            $benefit_type_name = '';
            try {
                $sql_check_type = "SELECT type_name FROM benefit_types WHERE id = ? AND deleted_at IS NULL";
                $stmt_check = mysqli_prepare($link, $sql_check_type);
                mysqli_stmt_bind_param($stmt_check, "i", $benefit_type_id);
                mysqli_stmt_execute($stmt_check);
                $result_check = mysqli_stmt_get_result($stmt_check);
                if ($row_check = mysqli_fetch_assoc($result_check)) {
                    $benefit_type_name = $row_check['type_name'];
                }
                mysqli_stmt_close($stmt_check);
            } catch (Exception $e) {
                error_log("Benefit type check failed: " . $e->getMessage());
            }
            if ($benefit_type_name) {
                // *** BEGIN TRANSACTION BLOCK ***
                mysqli_begin_transaction($link);
                $transaction_success = true;

                // Deduct tokens (dynamic amount)
                $tokens_used = deduct_tokens($link, $_SESSION['user_id'], 'benefit', $required_tokens);
                if ($tokens_used !== false) {
                    try {
                        $sql = "INSERT INTO member_benefits (user_id, member_id, benefit_type_id, benefit_type, amount, benefit_date, benefit_status, created_at) 
                                VALUES (?, ?, ?, ?, ?, CURDATE(), 'Approved', NOW())";
                        $stmt = mysqli_prepare($link, $sql);
                        foreach ($user_ids as $member_id) {
                            $member_id = intval($member_id);
                            $user_id = $_SESSION['user_id'];
                            mysqli_stmt_bind_param($stmt, "iiisd", $user_id, $member_id, $benefit_type_id, $benefit_type_name, $amount);
                            if (!mysqli_stmt_execute($stmt)) {
                                throw new Exception("Benefit insert failed for member $member_id: " . mysqli_stmt_error($stmt));
                            }
                        }
                        mysqli_stmt_close($stmt);
                    } catch (Exception $e) {
                        $transaction_success = false;
                        $message = "Error logging benefit: " . $e->getMessage();
                        $message_type = 'danger';
                        error_log("Benefit insert failed: " . $e->getMessage());
                    }
                } else {
                    $transaction_success = false;
                    $message = "Failed to deduct tokens. <a href='token_store.php'>Please purchase more</a>.";
                    $message_type = 'danger';
                }

                if ($transaction_success) {
                    mysqli_commit($link);
                    $message = "Benefit logged successfully! Used $tokens_used tokens.";
                    $message_type = 'success';
                } else {
                    mysqli_rollback($link);
                    if (empty($message)) {
                        $message = "Transaction failed. No benefit was logged and no tokens were deducted.";
                        $message_type = 'danger';
                    }
                }
            } else {
                $message = "Invalid benefit type.";
                $message_type = 'danger';
            }
        } else {
            $message = "Invalid input.";
            $message_type = 'danger';
        }
    }
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
} elseif ($_SERVER["REQUEST_METHOD"] == "POST") {
    $message = "Invalid CSRF token.";
    $message_type = 'danger';
}

// Fetch data
$members = [];
$sql_members = "SELECT id, full_name FROM users WHERE role = 'member' AND (deleted_at IS NULL OR deleted_at > NOW()) ORDER BY full_name ASC";
$result_members = mysqli_query($link, $sql_members);
if ($result_members) {
    while ($row = mysqli_fetch_assoc($result_members)) {
        $members[] = $row;
    }
    mysqli_free_result($result_members);
}

$benefit_types = [];
$sql_benefit_types = "SELECT id, type_name FROM benefit_types WHERE deleted_at IS NULL ORDER BY type_name ASC";
$result_benefit_types = mysqli_query($link, $sql_benefit_types);
if ($result_benefit_types) {
    while ($row = mysqli_fetch_assoc($result_benefit_types)) {
        $benefit_types[] = $row;
    }
    mysqli_free_result($result_benefit_types);
}
?>

<?php require_once 'includes/header.php'; ?>

<script>
document.title = 'Record Benefits - <?php echo htmlspecialchars(get_setting('system_name', $link) ?? 'Club Portal'); ?>';
</script>

<style>
.main-content {
    padding: 20px;
}
</style>

<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Record Benefits</h2>

            <?php if ($message): ?>
                <div class="alert alert-<?php echo htmlspecialchars($message_type); ?>" role="alert">
                    <?php echo $message; ?>
                </div>
            <?php endif; ?>

            <div class="card mb-4">
                <div class="card-header">Log New Benefit</div>
                <div class="card-body">
                    <p>System Available Tokens: <?php echo htmlspecialchars($system_available_tokens); ?> tokens</p>
                    <form action="record_benefits.php" method="post">
                        <input type="hidden" name="log_benefit" value="1">
                        <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">
                        <div class="mb-3">
                            <label for="user_ids" class="form-label">Select Member(s)</label>
                            <select name="user_ids[]" id="user_ids" class="form-select" multiple size="10" required>
                                <?php foreach ($members as $member): ?>
                                    <option value="<?php echo htmlspecialchars($member['id']); ?>">
                                        <?php echo htmlspecialchars($member['full_name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <small class="form-text text-muted">Hold Ctrl to select multiple. (<?php echo $token_cost; ?> token per log)</small>
                        </div>
                        <div class="mb-3">
                            <label for="benefit_type_id" class="form-label">Benefit Type</label>
                            <select name="benefit_type_id" id="benefit_type_id" class="form-select" required>
                                <option value="">Select</option>
                                <?php foreach ($benefit_types as $type): ?>
                                    <option value="<?php echo htmlspecialchars($type['id']); ?>">
                                        <?php echo htmlspecialchars($type['type_name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label for="amount" class="form-label">Amount (<?php echo htmlspecialchars($currency_symbol); ?>)</label>
                            <input type="number" name="amount" id="amount" class="form-control" step="0.01" min="0.01" required>
                        </div>
                        <div class="mb-3">
                            <label for="description" class="form-label">Description (Optional)</label>
                            <textarea name="description" id="description" class="form-control" rows="3"></textarea>
                        </div>
                        <button type="submit" class="btn btn-primary">Log Benefit</button>
                    </form>
                </div>
            </div>
        </main>
    </div>
</div>

<?php require_once 'includes/footer.php'; ?>