<?php
require_once 'includes/functions.php';
require_once 'includes/db_config.php';

// Check if user is logged in and is an admin
if (!is_logged_in() || !is_admin()) {
    redirect("login.php");
}

$message = '';
$admin_id = $_SESSION['user_id'];

// Handle payment/donation logging
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['log_payment'])) {
    $user_ids = $_POST['user_ids']; // Array of selected member IDs
    $amount = floatval($_POST['amount']);
    $payment_type = sanitize_input($_POST['payment_type']);
    $description = sanitize_input($_POST['description']);
    $status = 'Paid'; // Manual payments are approved instantly

    // Determine payment_cycle based on payment_type
    $payment_cycle = null;
    $event_date = null;
    if ($payment_type !== 'donation') { // Donations are non-recurring, so no payment_cycle
        $sql_type = "SELECT recurrence_interval FROM payment_types WHERE type_name = ? AND deleted_at IS NULL";
        if ($stmt_type = mysqli_prepare($link, $sql_type)) {
            mysqli_stmt_bind_param($stmt_type, "s", $payment_type);
            mysqli_stmt_execute($stmt_type);
            $result_type = mysqli_stmt_get_result($stmt_type);
            if ($row_type = mysqli_fetch_assoc($result_type)) {
                $recurrence_interval = $row_type['recurrence_interval'];
                if ($recurrence_interval === 'monthly') {
                    $payment_cycle = date('Y-m'); // e.g., '2025-09'
                } elseif ($recurrence_interval === 'weekly') {
                    $payment_cycle = date('Y-m-d', strtotime('monday this week')); // Start of the week
                } elseif ($recurrence_interval === 'yearly') {
                    $payment_cycle = date('Y'); // e.g., '2025'
                }
            } else {
                $message = "Error: Payment type '$payment_type' not found.";
            }
            mysqli_stmt_close($stmt_type);
        } else {
            $message = "Error preparing payment type query: " . mysqli_error($link);
        }
    }

    if (!empty($user_ids) && $amount > 0 && empty($message)) {
        $sql = "INSERT INTO payments (user_id, payment_type, amount, description, payment_status, payment_cycle, event_date, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, NOW())";
        if ($stmt = mysqli_prepare($link, $sql)) {
            foreach ($user_ids as $user_id) {
                $user_id = intval($user_id);
                mysqli_stmt_bind_param($stmt, "isdssss", $user_id, $payment_type, $amount, $description, $status, $payment_cycle, $event_date);
                mysqli_stmt_execute($stmt);
            }
            mysqli_stmt_close($stmt);
            $message = "Payment logged successfully for all selected members!";
        } else {
            $message = "Error preparing payment insert statement: " . mysqli_error($link);
        }
    } else {
        $message = "Invalid input. Please check the amount, select at least one member, or verify payment type.";
    }
}

// Fetch all members for the dropdown
$sql_members = "SELECT id, full_name FROM users WHERE role = 'member' AND (deleted_at IS NULL OR deleted_at > NOW()) ORDER BY full_name ASC";
$result_members = mysqli_query($link, $sql_members);
$members = [];
if ($result_members) {
    while ($row = mysqli_fetch_assoc($result_members)) {
        $members[] = $row;
    }
}

// Fetch all payment types for the dropdown
$sql_payment_types = "SELECT type_name, recurrence_interval FROM payment_types WHERE deleted_at IS NULL ORDER BY type_name ASC";
$result_payment_types = mysqli_query($link, $sql_payment_types);
$payment_types = [];
if ($result_payment_types) {
    while ($row = mysqli_fetch_assoc($result_payment_types)) {
        $payment_types[] = $row;
    }
}

// Fetch all payments for display
$sql_payments = "SELECT p.*, u.full_name FROM payments p JOIN users u ON p.user_id = u.id ORDER BY p.created_at DESC";
$result_payments = mysqli_query($link, $sql_payments);
$payment_list = [];
if ($result_payments) {
    while ($row = mysqli_fetch_assoc($result_payments)) {
        $payment_list[] = $row;
    }
}
?>

<?php require_once 'includes/header.php'; ?>
<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Payment & Donation Management</h2>

            <?php if ($message): ?>
                <div class="alert alert-<?php echo strpos($message, 'Error') !== false ? 'danger' : 'success'; ?>">
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>

            <div class="card mb-4">
                <div class="card-header">Log a Manual Payment</div>
                <div class="card-body">
                    <form action="payment_management.php" method="post">
                        <input type="hidden" name="log_payment" value="1">
                        <div class="form-group mb-3">
                            <label for="user_ids">Select Member(s)</label>
                            <select name="user_ids[]" id="user_ids" class="form-control" multiple size="10" required>
                                <?php foreach ($members as $member): ?>
                                    <option value="<?php echo $member['id']; ?>"><?php echo htmlspecialchars($member['full_name']); ?></option>
                                <?php endforeach; ?>
                            </select>
                            <small class="form-text text-muted">Hold down the Ctrl (Windows) or Cmd (Mac) key to select multiple members.</small>
                        </div>
                        <div class="form-group mb-3">
                            <label for="amount">Amount (<?php echo htmlspecialchars(get_setting('currency_symbol', $link) ?? '₦'); ?>)</label>
                            <input type="number" name="amount" id="amount" class="form-control" step="0.01" min="0.01" required>
                        </div>
                        <div class="form-group mb-3">
                            <label for="payment_type">Payment Type</label>
                            <select name="payment_type" id="payment_type" class="form-control" required>
                                <?php foreach ($payment_types as $type): ?>
                                    <option value="<?php echo htmlspecialchars($type['type_name']); ?>">
                                        <?php echo htmlspecialchars($type['type_name']); ?>
                                        <?php echo $type['recurrence_interval'] ? ' (' . ucfirst($type['recurrence_interval']) . ')' : ''; ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group mb-3">
                            <label for="description">Description</label>
                            <textarea name="description" id="description" class="form-control" rows="3"></textarea>
                        </div>
                        <button type="submit" class="btn btn-primary">Log Payment</button>
                    </form>
                </div>
            </div>

            <div class="card">
                <div class="card-header">Payment History Log</div>
                <div class="card-body">
                    <table class="table table-striped table-hover">
                        <thead>
                            <tr>
                                <th>Member</th>
                                <th>Amount</th>
                                <th>Type</th>
                                <th>Payment Cycle</th>
                                <th>Event Date</th>
                                <th>Description</th>
                                <th>Date</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($payment_list)): ?>
                                <tr><td colspan="8" class="text-center">No payments have been logged yet.</td></tr>
                            <?php else: ?>
                                <?php foreach ($payment_list as $payment): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($payment['full_name']); ?></td>
                                        <td><?php echo htmlspecialchars(get_setting('currency_symbol', $link) ?? '₦') . ' ' . htmlspecialchars(number_format($payment['amount'], 2)); ?></td>
                                        <td><?php echo htmlspecialchars(ucfirst($payment['payment_type'])); ?></td>
                                        <td><?php echo htmlspecialchars($payment['payment_cycle'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($payment['event_date'] ? date('Y-m-d', strtotime($payment['event_date'])) : 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($payment['description'] ?? 'N/A'); ?></td>
                                        <td><?php echo date('M j, Y', strtotime($payment['created_at'])); ?></td>
                                        <td>
                                            <a href="edit_payment.php?id=<?php echo $payment['id']; ?>" class="btn btn-sm btn-secondary">Edit</a>
                                            <a href="delete_payment.php?id=<?php echo $payment['id']; ?>" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure you want to delete this payment?');">Delete</a>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </main>
    </div>
</div>
<?php require_once 'includes/footer.php'; ?>