<?php
// manage_token_purchases.php
ob_start();
if (session_status() !== PHP_SESSION_ACTIVE) {
    session_start();
}

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Clear OPCache
if (function_exists('opcache_reset')) {
    opcache_reset();
}

require_once 'includes/functions.php';
require_once 'includes/db_config.php';

// Session timeout
$timeout = 1800; // 30 minutes
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity']) > $timeout) {
    session_unset();
    session_destroy();
    redirect("login.php?timeout=1");
}
$_SESSION['last_activity'] = time();

// Restrict to super_admin
if (!is_logged_in() || $_SESSION['user_role'] !== 'super_admin') {
    redirect("login.php");
}

// Generate CSRF token
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

$message = '';
$message_type = '';
$purchases = [];

// Check database connection
if (mysqli_connect_error()) {
    $message = "Database connection failed: " . mysqli_connect_error();
    $message_type = 'danger';
    error_log("Database connection failed in manage_token_purchases.php: " . mysqli_connect_error());
} else {
    // Check if token_purchases table exists
    $check_table = mysqli_query($link, "SHOW TABLES LIKE 'token_purchases'");
    if (mysqli_num_rows($check_table) == 0) {
        $message = "Error: Token purchases table does not exist.";
        $message_type = 'danger';
        error_log("Token purchases table missing in manage_token_purchases.php");
    } else {
        // Handle approval/rejection
        if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['action']) && isset($_POST['csrf_token']) && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
            $purchase_id = intval($_POST['purchase_id'] ?? 0);
            $action = sanitize_input($_POST['action']);
            $user_id = intval($_POST['user_id'] ?? 0);
            $quantity = intval($_POST['quantity'] ?? 0);

            if ($action === 'approve') {
                // Update purchase status
                $sql = "UPDATE token_purchases SET status = 'approved' WHERE id = ?";
                $stmt = mysqli_prepare($link, $sql);
                if ($stmt) {
                    mysqli_stmt_bind_param($stmt, "i", $purchase_id);
                    $success = mysqli_stmt_execute($stmt);
                    mysqli_stmt_close($stmt);

                    if ($success) {
                        // Update token balance
                        $balance_success = false;
                        $sql = "INSERT INTO token_balances (user_id, balance) VALUES (?, ?) ON DUPLICATE KEY UPDATE balance = balance + ?";
                        $stmt = mysqli_prepare($link, $sql);
                        if ($stmt) {
                            mysqli_stmt_bind_param($stmt, "iii", $user_id, $quantity, $quantity);
                            $balance_success = mysqli_stmt_execute($stmt);
                            mysqli_stmt_close($stmt);
                        }

                        if ($balance_success) {
                            // NEW: Auto-log as payment (income) in the payments table
                            // First, fetch the monetary amount from token_purchases
                            $monetary_amount = 0;
                            $sql_amount = "SELECT amount FROM token_purchases WHERE id = ?";
                            $stmt_amount = mysqli_prepare($link, $sql_amount);
                            if ($stmt_amount) {
                                mysqli_stmt_bind_param($stmt_amount, "i", $purchase_id);
                                mysqli_stmt_execute($stmt_amount);
                                $result_amount = mysqli_stmt_get_result($stmt_amount);
                                if ($row_amount = mysqli_fetch_assoc($result_amount)) {
                                    $monetary_amount = floatval($row_amount['amount']);
                                }
                                mysqli_stmt_close($stmt_amount);
                            }

                            if ($monetary_amount > 0) {
                                // Determine payment_type: Use 'Token Purchase' or insert if not exists
                                $payment_type = 'Token Purchase';
                                $sql_type_check = "SELECT type_name FROM payment_types WHERE type_name = ? AND deleted_at IS NULL";
                                $stmt_type_check = mysqli_prepare($link, $sql_type_check);
                                $type_exists = false;
                                if ($stmt_type_check) {
                                    mysqli_stmt_bind_param($stmt_type_check, "s", $payment_type);
                                    mysqli_stmt_execute($stmt_type_check);
                                    mysqli_stmt_store_result($stmt_type_check);
                                    $type_exists = mysqli_stmt_num_rows($stmt_type_check) > 0;
                                    mysqli_stmt_close($stmt_type_check);
                                }

                                if (!$type_exists) {
                                    // Insert new payment type (assume category 'income' or similar; adjust if needed)
                                    $sql_insert_type = "INSERT INTO payment_types (type_name, payment_category, created_at) VALUES (?, 'income', NOW())";
                                    $stmt_insert_type = mysqli_prepare($link, $sql_insert_type);
                                    if ($stmt_insert_type) {
                                        mysqli_stmt_bind_param($stmt_insert_type, "s", $payment_type);
                                        mysqli_stmt_execute($stmt_insert_type);
                                        mysqli_stmt_close($stmt_insert_type);
                                    }
                                }

                                // Log as payment (no obligation_id, status 'Paid')
                                $description = "Auto-logged token purchase of $quantity tokens";
                                $obligation_id = 0; // No obligation
                                $status = 'Paid';
                                $sql_payment = "INSERT INTO payments (user_id, payment_type, amount, description, payment_status, obligation_id, payment_date, created_at)
                                                VALUES (?, ?, ?, ?, ?, ?, NOW(), NOW())";
                                $stmt_payment = mysqli_prepare($link, $sql_payment);
                                if ($stmt_payment) {
                                    mysqli_stmt_bind_param($stmt_payment, "isdssi", $user_id, $payment_type, $monetary_amount, $description, $status, $obligation_id);
                                    $payment_success = mysqli_stmt_execute($stmt_payment);
                                    mysqli_stmt_close($stmt_payment);
                                    if ($payment_success) {
                                        error_log("Auto-logged token purchase as payment for purchase_id $purchase_id");
                                    } else {
                                        error_log("Failed to auto-log payment for token purchase_id $purchase_id: " . mysqli_error($link));
                                    }
                                } else {
                                    error_log("Prepare failed for auto-log payment: " . mysqli_error($link));
                                }
                            } else {
                                error_log("No monetary amount found for token purchase_id $purchase_id");
                            }

                            $message = "Purchase approved, tokens added, and revenue logged automatically.";
                            $message_type = 'success';
                            // Notify user
                            $sql = "SELECT email FROM users WHERE id = ?";
                            $stmt = mysqli_prepare($link, $sql);
                            mysqli_stmt_bind_param($stmt, "i", $user_id);
                            mysqli_stmt_execute($stmt);
                            $result = mysqli_stmt_get_result($stmt);
                            if ($row = mysqli_fetch_assoc($result)) {
                                $subject = "Token Purchase Approved";
                                $body = "Your token purchase of $quantity tokens has been approved and added to your balance.";
                                if (!send_mail($row['email'], $subject, $body)) {
                                    error_log("Failed to send approval email in manage_token_purchases.php");
                                }
                            }
                            mysqli_stmt_close($stmt);
                        } else {
                            $message = "Error updating token balance: " . mysqli_error($link);
                            $message_type = 'danger';
                            error_log("Error updating token balance in manage_token_purchases.php: " . mysqli_error($link));
                        }
                    } else {
                        $message = "Error approving purchase: " . mysqli_error($link);
                        $message_type = 'danger';
                        error_log("Error approving purchase in manage_token_purchases.php: " . mysqli_error($link));
                    }
                } else {
                    $message = "Error preparing approval: " . mysqli_error($link);
                    $message_type = 'danger';
                    error_log("Error preparing approval in manage_token_purchases.php: " . mysqli_error($link));
                }
            } elseif ($action === 'reject') {
                $sql = "UPDATE token_purchases SET status = 'rejected' WHERE id = ?";
                $stmt = mysqli_prepare($link, $sql);
                if ($stmt) {
                    mysqli_stmt_bind_param($stmt, "i", $purchase_id);
                    $success = mysqli_stmt_execute($stmt);
                    mysqli_stmt_close($stmt);
                    if ($success) {
                        $message = "Purchase rejected.";
                        $message_type = 'success';
                        // Notify user
                        $sql = "SELECT email FROM users WHERE id = ?";
                        $stmt = mysqli_prepare($link, $sql);
                        mysqli_stmt_bind_param($stmt, "i", $user_id);
                        mysqli_stmt_execute($stmt);
                        $result = mysqli_stmt_get_result($stmt);
                        if ($row = mysqli_fetch_assoc($result)) {
                            $subject = "Token Purchase Rejected";
                            $body = "Your token purchase request has been rejected.";
                            if (!send_mail($row['email'], $subject, $body)) {
                                error_log("Failed to send rejection email in manage_token_purchases.php");
                            }
                        }
                        mysqli_stmt_close($stmt);
                    } else {
                        $message = "Error rejecting purchase: " . mysqli_error($link);
                        $message_type = 'danger';
                        error_log("Error rejecting purchase in manage_token_purchases.php: " . mysqli_error($link));
                    }
                } else {
                    $message = "Error preparing rejection: " . mysqli_error($link);
                    $message_type = 'danger';
                    error_log("Error preparing rejection in manage_token_purchases.php: " . mysqli_error($link));
                }
            }
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        } elseif ($_SERVER["REQUEST_METHOD"] == "POST") {
            $message = "Invalid CSRF token.";
            $message_type = 'danger';
            error_log("CSRF token validation failed in manage_token_purchases.php");
        }

        // Fetch pending purchases
        $sql = "SELECT tp.id, tp.user_id, u.email, tp.package_id, tp.quantity, tp.amount, tp.status, tp.created_at, tp.package_name
                FROM token_purchases tp
                JOIN users u ON tp.user_id = u.id
                WHERE tp.status = 'pending'";
        if ($result = mysqli_query($link, $sql)) {
            while ($row = mysqli_fetch_assoc($result)) {
                $purchases[] = $row;
            }
            mysqli_free_result($result);
        } else {
            $message = "Error fetching purchases: " . mysqli_error($link);
            $message_type = 'danger';
            error_log("Error fetching purchases in manage_token_purchases.php: " . mysqli_error($link));
        }
    }
}

ob_end_clean();
?>

<?php require_once 'includes/header.php'; ?>

<script>
document.title = 'Manage Token Purchases - <?php echo htmlspecialchars(get_setting('system_name', $link) ?? 'Club Portal'); ?>';
</script>

<style>
.main-content {
    padding: 20px;
}
</style>

<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Manage Token Purchase Requests</h2>

            <?php if ($message): ?>
                <div class="alert alert-<?php echo htmlspecialchars($message_type); ?>" role="alert">
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>

            <div class="card">
                <div class="card-header">Pending Token Purchases</div>
                <div class="card-body">
                    <?php if (empty($purchases)): ?>
                        <p class="text-muted">No pending purchase requests.</p>
                    <?php else: ?>
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>User Email</th>
                                    <th>Package Name</th>
                                    <th>Quantity</th>
                                    <th>Amount</th>
                                    <th>Date</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($purchases as $purchase): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($purchase['email']); ?></td>
                                        <td><?php echo htmlspecialchars($purchase['package_name']); ?></td>
                                        <td><?php echo htmlspecialchars($purchase['quantity']); ?></td>
                                        <td><?php echo htmlspecialchars(format_currency($purchase['amount'], get_setting('currency_symbol', $link) ?? '₦')); ?></td>
                                        <td><?php echo htmlspecialchars($purchase['created_at']); ?></td>
                                        <td>
                                            <form action="manage_token_purchases.php" method="post" style="display:inline;">
                                                <input type="hidden" name="action" value="approve">
                                                <input type="hidden" name="purchase_id" value="<?php echo htmlspecialchars($purchase['id']); ?>">
                                                <input type="hidden" name="user_id" value="<?php echo htmlspecialchars($purchase['user_id']); ?>">
                                                <input type="hidden" name="quantity" value="<?php echo htmlspecialchars($purchase['quantity']); ?>">
                                                <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">
                                                <button type="submit" class="btn btn-success btn-sm">Approve</button>
                                            </form>
                                            <form action="manage_token_purchases.php" method="post" style="display:inline;">
                                                <input type="hidden" name="action" value="reject">
                                                <input type="hidden" name="purchase_id" value="<?php echo htmlspecialchars($purchase['id']); ?>">
                                                <input type="hidden" name="user_id" value="<?php echo htmlspecialchars($purchase['user_id']); ?>">
                                                <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">
                                                <button type="submit" class="btn btn-danger btn-sm">Reject</button>
                                            </form>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>
</div>

<?php require_once 'includes/footer.php'; ?>