<?php
// generate_obligations.php
// Generates payment obligations for members based on payment types, recurrence intervals, and due dates

require_once 'includes/db_config.php';
require_once 'includes/functions.php';

// Get current timestamp and year for comparison
$current_date = date('Y-m-d H:i:s');
$current_year = date('Y');

// Check if required columns exist in payment_obligations
$columns_query = "SHOW COLUMNS FROM payment_obligations";
$columns_result = mysqli_query($link, $columns_query);
$columns = [];
if ($columns_result) {
    while ($row = mysqli_fetch_assoc($columns_result)) {
        $columns[] = $row['Field'];
    }
}
$has_due_date = in_array('due_date', $columns);
$has_required_amount = in_array('required_amount', $columns);
$has_deleted_at = in_array('deleted_at', $columns);

// Fetch active members
$members = [];
$sql_members = "SELECT id FROM users WHERE role = 'member' AND (deleted_at IS NULL OR deleted_at > NOW())";
$result_members = mysqli_query($link, $sql_members);
if ($result_members) {
    while ($row = mysqli_fetch_assoc($result_members)) {
        $members[] = $row['id'];
    }
} else {
    error_log("Failed to fetch members: " . mysqli_error($link));
    exit;
}

// Fetch active payment types
$payment_types = [];
$sql_payment_types = "SELECT id, type_name, required_amount, payment_category, recurrence_interval, event_date 
                     FROM payment_types 
                     WHERE deleted_at IS NULL";
$result_payment_types = mysqli_query($link, $sql_payment_types);
if ($result_payment_types) {
    while ($row = mysqli_fetch_assoc($result_payment_types)) {
        $payment_types[] = $row;
    }
} else {
    error_log("Failed to fetch payment types: " . mysqli_error($link));
    exit;
}

// Process each payment type
$obligations_created = 0;
foreach ($payment_types as $payment_type) {
    $type_name = $payment_type['type_name'];
    $required_amount = $payment_type['required_amount'] ?? 0.00;
    $payment_category = $payment_type['payment_category'];
    $recurrence_interval = $payment_type['recurrence_interval'];
    $event_date = $payment_type['event_date'];

    foreach ($members as $user_id) {
        // Check for any existing obligation (active or soft-deleted)
        $select_fields = 'id, payment_cycle';
        if ($has_required_amount) {
            $select_fields .= ', required_amount';
        }
        if ($has_due_date) {
            $select_fields .= ', due_date';
        }
        // Include soft-deleted obligations in the check
        $where_clause = '';
        $sql_check = "SELECT $select_fields 
                      FROM payment_obligations 
                      WHERE user_id = ? AND payment_type = ? AND payment_cycle = ?$where_clause 
                      ORDER BY " . ($has_due_date ? 'due_date' : 'id') . " DESC LIMIT 1";
        
        if ($stmt_check = mysqli_prepare($link, $sql_check)) {
            mysqli_stmt_bind_param($stmt_check, "isi", $user_id, $type_name, $current_year);
            if (!mysqli_stmt_execute($stmt_check)) {
                error_log("Failed to check obligation for user_id $user_id, payment_type $type_name: " . mysqli_stmt_error($stmt_check));
                mysqli_stmt_close($stmt_check);
                continue;
            }
            $result = mysqli_stmt_get_result($stmt_check);
            $existing_obligation = mysqli_fetch_assoc($result);
            mysqli_stmt_close($stmt_check);

            $create_new_obligation = false;
            $new_due_date = null;

            if ($payment_category === 'recurring' && $recurrence_interval !== 'none' && $has_due_date) {
                if (!$existing_obligation) {
                    $create_new_obligation = true;
                    $new_due_date = calculate_due_date($recurrence_interval, $current_date);
                } elseif ($existing_obligation['due_date'] <= $current_date) {
                    $create_new_obligation = true;
                    $new_due_date = calculate_due_date($recurrence_interval, $existing_obligation['due_date']);
                }
            } elseif ($payment_category === 'event' && $event_date && $has_due_date) {
                if (!$existing_obligation && $event_date >= $current_date) {
                    $create_new_obligation = true;
                    $new_due_date = $event_date;
                }
            } elseif ($payment_category === 'freewill') {
                if (!$existing_obligation) {
                    $create_new_obligation = true;
                    $new_due_date = null;
                }
            }

            if ($create_new_obligation) {
                $insert_fields = 'user_id, payment_type, payment_cycle, created_at';
                $insert_values = '?, ?, ?, NOW()';
                $bind_types = 'isi';
                $bind_params = [&$user_id, &$type_name, &$current_year];
                if ($has_required_amount) {
                    $insert_fields .= ', required_amount';
                    $insert_values .= ', ?';
                    $bind_types .= 'd';
                    $bind_params[] = &$required_amount;
                }
                if ($has_due_date) {
                    $insert_fields .= ', due_date';
                    $insert_values .= ', ?';
                    $bind_types .= 's';
                    $bind_params[] = &$new_due_date;
                }
                $sql_insert = "INSERT INTO payment_obligations ($insert_fields) VALUES ($insert_values)";
                if ($stmt_insert = mysqli_prepare($link, $sql_insert)) {
                    mysqli_stmt_bind_param($stmt_insert, $bind_types, ...$bind_params);
                    if (mysqli_stmt_execute($stmt_insert)) {
                        $obligations_created++;
                    } else {
                        error_log("Failed to create obligation for user_id $user_id, payment_type $type_name, cycle $current_year: " . mysqli_stmt_error($stmt_insert));
                    }
                    mysqli_stmt_close($stmt_insert);
                } else {
                    error_log("Failed to prepare insert query for user_id $user_id, payment_type $type_name: " . mysqli_error($link));
                }
            }
        } else {
            error_log("Failed to prepare check query for user_id $user_id, payment_type $type_name: " . mysqli_error($link));
        }
    }
}

// Helper function to calculate next due date based on recurrence interval
function calculate_due_date($interval, $base_date) {
    $base = new DateTime($base_date);
    switch ($interval) {
        case 'weekly':
            $base->modify('+1 week');
            break;
        case 'monthly':
            $base->modify('+1 month');
            break;
        case 'yearly':
            $base->modify('+1 year');
            break;
        default:
            $base->modify('+1 month'); // Default to monthly
    }
    return $base->format('Y-m-d H:i:s');
}

// Do not close the database connection here to allow parent script to continue using it
// mysqli_close($link);
?>