$(function() {
    let fieldConfig = {};

    // Initializes draggable functionality for form fields
    $(".draggable-field").draggable({
        helper: "clone",
        revert: "invalid",
        connectToSortable: "#form-canvas"
    });

    // Initializes sortable functionality for the form canvas
    $("#form-canvas").sortable({
        placeholder: "ui-state-highlight",
        forcePlaceholderSize: true,
        stop: function(event, ui) {
            // Check if the item is a new draggable field from the palette
            if (ui.item.hasClass("draggable-field")) {
                const fieldType = ui.item.data("type");
                const fieldId = `field_${Date.now()}`;
                const fieldName = `${fieldType}_${Date.now()}`;
                const fieldLabel = ui.item.text();

                // Create the new field element with a consistent structure
                const newField = createFieldElement(fieldType, {
                    id: fieldId,
                    name: fieldName,
                    label: fieldLabel
                });

                ui.item.replaceWith(newField);
                
                // Add default config for the new field
                fieldConfig[fieldId] = {
                    required: false,
                    options: [],
                    conditional_logic: []
                };

                updateFormStructure();
            }
        }
    });

    // Creates a new form field HTML element
    function createFieldElement(type, data = {}) {
        const fieldId = data.id || `field_${Date.now()}`;
        const labelText = data.label || `${type.charAt(0).toUpperCase() + type.slice(1)} Label`;
        const fieldName = data.name || `${type}_${Date.now()}`;
        const requiredIndicator = data.required ? '<span class="required-indicator">*</span>' : '';

        const fieldHtml = `
            <div class="form-field form-field-container" data-type="${type}" data-id="${fieldId}">
                <div class="field-header">
                    <span class="field-label">${labelText} ${requiredIndicator}</span>
                    <div class="field-actions">
                        <button type="button" class="btn btn-sm btn-secondary edit-field">Edit</button>
                        <button type="button" class="btn btn-sm btn-danger remove-field">Remove</button>
                    </div>
                </div>
                <div class="field-preview">
                    <p class="field-type">${type}</p>
                </div>
                <input type="hidden" class="field-name" value="${fieldName}">
            </div>
        `;
        return $(fieldHtml);
    }
    
    // Updates the hidden input with the current form structure
    function updateFormStructure() {
        const formStructure = { fields: [] };
        $("#form-canvas > .form-field").each(function(fieldIndex) {
            const fieldElement = $(this);
            const fieldId = fieldElement.data("id");
            const fieldType = fieldElement.data("type");
            const fieldName = fieldElement.find(".field-name").val();
            const fieldLabel = fieldElement.find(".field-label").text().replace('*', '').trim();
            
            const fieldData = {
                id: fieldId,
                type: fieldType,
                name: fieldName,
                label: fieldLabel,
                order: fieldIndex
            };
            
            // Merge with fieldConfig data (required, options, etc.)
            if (fieldConfig[fieldId]) {
                Object.assign(fieldData, fieldConfig[fieldId]);
            }

            formStructure.fields.push(fieldData);
        });

        // Set the value of the hidden input
        $("#form-data-input").val(JSON.stringify(formStructure));
    }

    // Populates the field edit modal with data
    function populateFieldModal(fieldData) {
        $("#field-label-edit").val(fieldData.label);
        $("#field-name-edit").val(fieldData.name);
        $("#field-required-edit").prop("checked", fieldData.required);
        
        const type = fieldData.type;
        if (type === 'select' || type === 'radio' || type === 'checkbox') {
            $("#field-options-group").show();
            const options = fieldData.options || [];
            $("#field-options-edit").val(options.map(opt => opt.label).join('\n'));
        } else {
            $("#field-options-group").hide();
            $("#field-options-edit").val('');
        }
        
        // Populate conditional logic rules
        const rules = fieldData.conditional_logic || [];
        $("#conditional-rules").empty();
        rules.forEach(rule => { addConditionalRule(rule.field_id, rule.operator, rule.value); });
        updateConditionalFieldOptions();
        
        $("#field-edit-modal").fadeIn();
    }

    // Adds a new conditional rule row to the modal
    function addConditionalRule(fieldId = '', operator = 'equals', value = '') {
        const ruleHtml = `<div class="conditional-rule-row"><select class="form-control conditional-field-select"></select><select class="form-control conditional-operator-select"><option value="equals">Equals</option><option value="not_equals">Not Equals</option></select><input type="text" class="form-control conditional-value-input" value="${value}"><button type="button" class="btn btn-sm btn-danger remove-conditional-rule">Remove</button></div>`;
        const ruleRow = $(ruleHtml);
        ruleRow.find('.conditional-field-select').val(fieldId);
        ruleRow.find('.conditional-operator-select').val(operator);
        $("#conditional-rules").append(ruleRow);
        updateConditionalFieldOptions();
    }
    
    // Updates the list of fields for conditional logic dropdowns
    function updateConditionalFieldOptions() {
        const allFields = [];
        $("#form-canvas > .form-field").each(function() {
            allFields.push({
                id: $(this).data("id"),
                label: $(this).find(".field-label").text().replace('*', '').trim()
            });
        });

        $(".conditional-field-select").each(function() {
            const select = $(this);
            const selectedValue = select.val();
            select.empty();
            select.append('<option value="">- Select Field -</option>');
            allFields.forEach(field => {
                if (field.id !== $(`[data-id="${currentFieldId}"]`).data("id")) {
                    select.append(`<option value="${field.id}">${field.label}</option>`);
                }
            });
            if (selectedValue) {
                select.val(selectedValue);
            }
        });
    }

    // Event listener for editing a form field
    $("body").on("click", ".edit-field", function(e) {
        e.preventDefault();
        e.stopPropagation();
        const fieldElement = $(this).closest(".form-field");
        const fieldId = fieldElement.data("id");
        const fieldData = fieldConfig[fieldId] || {};
        
        // Add current state to fieldData for modal population
        Object.assign(fieldData, {
            id: fieldId,
            type: fieldElement.data("type"),
            name: fieldElement.find(".field-name").val(),
            label: fieldElement.find(".field-label").text().replace('*', '').trim()
        });

        currentFieldId = fieldId;
        populateFieldModal(fieldData);
    });

    // Saves changes from the field edit modal
    $("#save-field-edit").on("click", function() {
        const newLabel = $("#field-label-edit").val().trim();
        const isRequired = $("#field-required-edit").is(":checked");
        const optionsText = $("#field-options-edit").val().trim();
        const fieldElement = $(`[data-id="${currentFieldId}"]`);
        
        const rules = [];
        $("#conditional-rules .conditional-rule-row").each(function() {
            rules.push({
                field_id: $(this).find(".conditional-field-select").val(),
                operator: $(this).find(".conditional-operator-select").val(),
                value: $(this).find(".conditional-value-input").val(),
            });
        });
        
        if (newLabel !== "") {
            fieldElement.find(".field-label").html(newLabel + (isRequired ? ' <span class="required-indicator">*</span>' : ''));
            
            // Rebuild options array with value and label
            const options = optionsText.split('\n').map(o => o.trim()).filter(o => o !== '').map(o => ({
                label: o,
                value: o.toLowerCase().replace(/\s/g, '_')
            }));

            fieldConfig[currentFieldId] = {
                required: isRequired,
                options: options,
                conditional_logic: rules
            };
            updateFormStructure();
        }
        $("#field-edit-modal").fadeOut();
    });

    // Event listeners for modal interactions
    $("#add-conditional-rule").on("click", addConditionalRule);
    $("#conditional-rules").on("click", ".remove-conditional-rule", function() { $(this).closest(".conditional-rule-row").remove(); });
    $(".close-button").on("click", function() { $("#field-edit-modal").fadeOut(); $("#step-edit-modal").fadeOut(); });

    // Event listener for removing a form field
    $("body").on("click", ".remove-field", function() {
        const fieldId = $(this).closest(".form-field").data("id");
        $(this).closest(".form-field").remove();
        delete fieldConfig[fieldId];
        updateFormStructure();
    });

    // Load existing form data for editing
    if (typeof existingFormData !== 'undefined' && existingFormData && existingFormData.fields && existingFormData.fields.length > 0) {
        const formCanvas = $("#form-canvas");
        formCanvas.empty();
        existingFormData.fields.forEach(fieldData => {
            const fieldElement = createFieldElement(fieldData.type, {
                id: fieldData.id,
                name: fieldData.name,
                label: fieldData.label,
                required: fieldData.required
            });
            
            // Store the config for the loaded field
            fieldConfig[fieldData.id] = {
                required: fieldData.required,
                options: fieldData.options || [],
                conditional_logic: fieldData.conditional_logic || []
            };

            formCanvas.append(fieldElement);
        });
        updateFormStructure();
    }
});