<?php
// accountant_dashboard.php
require_once 'includes/functions.php';
require_once 'includes/db_config.php';
session_start();

// Check if user is logged in and has accountant role
if (!is_logged_in() || $_SESSION['user_role'] !== 'accountant') {
    redirect("login.php");
}

// Initialize variables
$total_members = 0;
$debug_info = [];
$currency_symbol = get_setting('currency_symbol', $link) ?? '₦'; // Fixed argument order

// Filter settings
$record_type = isset($_GET['record_type']) && in_array($_GET['record_type'], ['all', 'payments', 'expenses', 'debts', 'benefits']) ? $_GET['record_type'] : 'all';
$time_period = isset($_GET['time_period']) && in_array($_GET['time_period'], ['date', 'week', 'month', 'year', 'all']) ? $_GET['time_period'] : 'all';
$filter_date = isset($_GET['filter_date']) ? $_GET['filter_date'] : date('Y-m-d');
$start_date = null;
$end_date = null;
$type_filter = null;

// Set start and end dates based on time period
if ($time_period !== 'all') {
    switch ($time_period) {
        case 'date':
            $start_date = $filter_date;
            $end_date = $filter_date;
            break;
        case 'week':
            $start_date = date('Y-m-d', strtotime('-1 week'));
            $end_date = date('Y-m-d');
            break;
        case 'month':
            $start_date = date('Y-m-01', strtotime('-1 month'));
            $end_date = date('Y-m-t', strtotime('-1 month'));
            break;
        case 'year':
            $start_date = date('Y-01-01', strtotime('-1 year'));
            $end_date = date('Y-12-31', strtotime('-1 year'));
            break;
    }
}

// Fetch totals
$totals = get_financial_totals($link, $record_type, $start_date, $end_date);
$total_payments = $totals['total_payments'];
$total_expenses = $totals['total_expenses'];
$total_benefits = $totals['total_benefits'];
$total_debts = $totals['total_debts'];
$debug_info['totals'] = $totals;

// Fetch type summaries
$type_summaries = get_financial_type_summaries($link, $record_type, $time_period, $start_date, $end_date, $type_filter);
$payment_types_summary = $type_summaries['payment_types'];
$expense_types_summary = $type_summaries['expense_types'];
$benefit_types_summary = $type_summaries['benefit_types'];
$debug_info['type_summaries'] = $type_summaries;

// Fetch total members (if permitted)
if (user_has_permission($link, 'view_member_directory')) {
    $sql_members = "SELECT COUNT(*) AS total FROM users WHERE role IN ('member', 'pending')";
    $result_members = mysqli_query($link, $sql_members);
    if ($result_members) {
        $row = mysqli_fetch_assoc($result_members);
        $total_members = $row['total'];
        $debug_info['members_query'] = $sql_members;
        $debug_info['total_members'] = $total_members;
    } else {
        error_log("Failed to fetch members: " . mysqli_error($link));
        $debug_info['members_error'] = mysqli_error($link);
    }
}
?>

<?php require_once 'includes/header.php'; ?>

<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Accountant Dashboard</h2>
            <hr>

            <!-- Filters Section -->
            <div class="mb-4">
                <div class="row">
                    <div class="col-md-3">
                        <label for="recordTypeFilter" class="form-label">Record Type:</label>
                        <select id="recordTypeFilter" class="form-select" onchange="applyFilters()">
                            <option value="all" <?php echo $record_type === 'all' ? 'selected' : ''; ?>>All</option>
                            <option value="payments" <?php echo $record_type === 'payments' ? 'selected' : ''; ?>>Payments</option>
                            <option value="expenses" <?php echo $record_type === 'expenses' ? 'selected' : ''; ?>>Expenses</option>
                            <option value="benefits" <?php echo $record_type === 'benefits' ? 'selected' : ''; ?>>Benefits</option>
                            <option value="debts" <?php echo $record_type === 'debts' ? 'selected' : ''; ?>>Debts</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label for="timePeriodFilter" class="form-label">Time Period:</label>
                        <select id="timePeriodFilter" class="form-select" onchange="toggleDatePicker(); applyFilters()">
                            <option value="all" <?php echo $time_period === 'all' ? 'selected' : ''; ?>>All Time</option>
                            <option value="date" <?php echo $time_period === 'date' ? 'selected' : ''; ?>>Specific Date</option>
                            <option value="week" <?php echo $time_period === 'week' ? 'selected' : ''; ?>>Last Week</option>
                            <option value="month" <?php echo $time_period === 'month' ? 'selected' : ''; ?>>Last Month</option>
                            <option value="year" <?php echo $time_period === 'year' ? 'selected' : ''; ?>>Last Year</option>
                        </select>
                    </div>
                    <div class="col-md-3" id="datePickerContainer" style="display: <?php echo $time_period === 'date' ? 'block' : 'none'; ?>;">
                        <label for="filterDate" class="form-label">Select Date:</label>
                        <input type="date" id="filterDate" class="form-control" value="<?php echo htmlspecialchars($filter_date); ?>" onchange="applyFilters()">
                    </div>
                </div>
            </div>

            <!-- Cards Section (2x3 Grid) -->
            <div class="row mb-4">
                <?php if (user_has_permission($link, 'view_payments')): ?>
                    <div class="col-md-4 mb-3">
                        <div class="card text-white bg-success">
                            <div class="card-header"><i class="fas fa-money-check-alt me-2"></i>Total Payments</div>
                            <div class="card-body">
                                <h5 class="card-title"><?php echo htmlspecialchars($currency_symbol . ' ' . number_format($total_payments, 2)); ?></h5>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
                <?php if (user_has_permission($link, 'view_member_directory')): ?>
                    <div class="col-md-4 mb-3">
                        <div class="card text-white bg-info">
                            <div class="card-header"><i class="fas fa-users me-2"></i>Total Members</div>
                            <div class="card-body">
                                <h5 class="card-title"><?php echo htmlspecialchars($total_members); ?></h5>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
                <?php if (user_has_permission($link, 'view_benefits')): ?>
                    <div class="col-md-4 mb-3">
                        <div class="card text-white bg-primary">
                            <div class="card-header"><i class="fas fa-gift me-2"></i>Total Benefits</div>
                            <div class="card-body">
                                <h5 class="card-title"><?php echo htmlspecialchars($currency_symbol . ' ' . number_format($total_benefits, 2)); ?></h5>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
                <?php if (user_has_permission($link, 'view_debts')): ?>
                    <div class="col-md-4 mb-3">
                        <div class="card text-white bg-warning">
                            <div class="card-header"><i class="fas fa-exclamation-triangle me-2"></i>Total Debts</div>
                            <div class="card-body">
                                <h5 class="card-title"><?php echo htmlspecialchars($currency_symbol . ' ' . number_format($total_debts, 2)); ?></h5>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
                <?php if (!user_has_permission($link, 'view_payments') && !user_has_permission($link, 'view_member_directory') && !user_has_permission($link, 'view_debts') && !user_has_permission($link, 'view_benefits')): ?>
                    <div class="col-12">
                        <div class="alert alert-warning">
                            No features are currently assigned to your role. Contact the administrator to update your permissions.
                        </div>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Financial Breakdown Chart (Bar) -->
            <?php if (user_has_permission($link, 'view_payments') || user_has_permission($link, 'view_debts') || user_has_permission($link, 'view_benefits')): ?>
                <div class="card mb-4">
                    <div class="card-header">Financial Breakdown (Bar Chart)</div>
                    <div class="card-body">
                        <canvas id="financialBarChart" height="100"></canvas>
                        <?php if ($total_payments == 0 && $total_expenses == 0 && $total_benefits == 0 && $total_debts == 0): ?>
                            <p class="text-muted text-center mt-3">No financial data available.</p>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endif; ?>

            <!-- Additional Charts (Pie, Histogram, Line) -->
            <?php if (user_has_permission($link, 'view_payments') || user_has_permission($link, 'view_debts') || user_has_permission($link, 'view_benefits')): ?>
                <div class="row mb-4">
                    <!-- Pie Chart -->
                    <div class="col-md-4 mb-3">
                        <div class="card">
                            <div class="card-header">Financial Overview (Pie Chart)</div>
                            <div class="card-body">
                                <canvas id="financialPieChart" height="200"></canvas>
                                <?php if ($total_payments == 0 && $total_expenses == 0 && $total_benefits == 0 && $total_debts == 0): ?>
                                    <p class="text-muted text-center mt-3">No financial data available.</p>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    <!-- Histogram (Bar Chart by Type) -->
                    <div class="col-md-4 mb-3">
                        <div class="card">
                            <div class="card-header">Financial Breakdown by Type (Histogram)</div>
                            <div class="card-body">
                                <canvas id="financialHistogram" height="200"></canvas>
                                <?php if (empty($payment_types_summary) && empty($expense_types_summary) && empty($benefit_types_summary)): ?>
                                    <p class="text-muted text-center mt-3">No financial data available.</p>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    <!-- Line Chart -->
                    <div class="col-md-4 mb-3">
                        <div class="card">
                            <div class="card-header">Financial Trend (Line Chart)</div>
                            <div class="card-body">
                                <canvas id="financialLineChart" height="200"></canvas>
                                <?php if (empty($payment_types_summary) && empty($expense_types_summary) && empty($benefit_types_summary)): ?>
                                    <p class="text-muted text-center mt-3">No financial data available.</p>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            <?php endif; ?>

            <!-- Quick Actions Section -->
            <?php if (user_has_permission($link, 'view_payments') || user_has_permission($link, 'view_member_directory') || user_has_permission($link, 'view_debts') || user_has_permission($link, 'view_benefits')): ?>
                <div class="card mb-4">
                    <div class="card-header">Quick Actions</div>
                    <div class="card-body">
                        <div class="d-flex flex-wrap gap-2">
                            <?php if (user_has_permission($link, 'view_payments')): ?>
                                <a href="manage_payment_types.php" class="btn btn-outline-primary"><i class="fas fa-money-bill-wave me-2"></i>Manage Payments</a>
                            <?php endif; ?>
                            <?php if (user_has_permission($link, 'view_member_directory')): ?>
                                <a href="member_directory.php" class="btn btn-outline-primary"><i class="fas fa-users me-2"></i>View Member Directory</a>
                            <?php endif; ?>
                            <?php if (user_has_permission($link, 'view_debts')): ?>
                                <a href="manage_payment_types.php" class="btn btn-outline-primary"><i class="fas fa-money-bill-alt me-2"></i>Manage Debts</a>
                            <?php endif; ?>
                            <?php if (user_has_permission($link, 'view_benefits')): ?>
                                <a href="manage_payment_types.php" class="btn btn-outline-primary"><i class="fas fa-gift me-2"></i>Manage Benefits</a>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            <?php endif; ?>
        </main>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/chart.js@3.9.1/dist/chart.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    <?php if (user_has_permission($link, 'view_payments') || user_has_permission($link, 'view_debts') || user_has_permission($link, 'view_benefits')): ?>
        // Bar Chart (Financial Breakdown)
        const barCtx = document.getElementById('financialBarChart').getContext('2d');
        const barData = {
            labels: ['Payments', 'Expenses', 'Benefits', 'Debts'],
            datasets: [{
                label: 'Financial Overview',
                data: [
                    <?php echo $total_payments; ?>,
                    <?php echo $total_expenses; ?>,
                    <?php echo $total_benefits; ?>,
                    <?php echo $total_debts; ?>
                ],
                backgroundColor: ['#28a745', '#dc3545', '#17a2b8', '#ffc107'],
                borderColor: ['#28a745', '#dc3545', '#17a2b8', '#ffc107'],
                borderWidth: 1
            }]
        };
        if (barData.datasets[0].data.every(d => d === 0)) {
            document.getElementById('financialBarChart').style.display = 'none';
        } else {
            document.getElementById('financialBarChart').style.display = 'block';
            new Chart(barCtx, {
                type: 'bar',
                data: barData,
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: {
                        y: {
                            beginAtZero: true,
                            title: {
                                display: true,
                                text: 'Amount (<?php echo htmlspecialchars($currency_symbol); ?>)'
                            }
                        },
                        x: {
                            title: {
                                display: true,
                                text: 'Category'
                            }
                        }
                    },
                    plugins: {
                        legend: {
                            display: false
                        }
                    }
                }
            });
        }

        // Pie Chart
        const pieCtx = document.getElementById('financialPieChart').getContext('2d');
        const pieData = {
            labels: ['Payments', 'Expenses', 'Benefits', 'Debts'],
            datasets: [{
                label: 'Financial Overview',
                data: [
                    <?php echo $total_payments; ?>,
                    <?php echo $total_expenses; ?>,
                    <?php echo $total_benefits; ?>,
                    <?php echo $total_debts; ?>
                ],
                backgroundColor: ['#28a745', '#dc3545', '#17a2b8', '#ffc107'],
                borderColor: ['#ffffff', '#ffffff', '#ffffff', '#ffffff'],
                borderWidth: 2
            }]
        };
        if (pieData.datasets[0].data.every(d => d === 0)) {
            document.getElementById('financialPieChart').style.display = 'none';
        } else {
            document.getElementById('financialPieChart').style.display = 'block';
            new Chart(pieCtx, {
                type: 'pie',
                data: pieData,
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'top'
                        }
                    }
                }
            });
        }

        // Histogram (Bar Chart by Type)
        const histCtx = document.getElementById('financialHistogram').getContext('2d');
        const histData = {
            labels: [<?php
                $all_labels = array_unique(array_merge(
                    array_map(function($type) { return $type['type_name']; }, $payment_types_summary),
                    array_map(function($type) { return $type['type_name']; }, $expense_types_summary),
                    array_map(function($type) { return $type['type_name']; }, $benefit_types_summary)
                ));
                echo "'" . implode("','", array_map('htmlspecialchars', $all_labels)) . "'";
            ?>],
            datasets: [
                {
                    label: 'Payments',
                    data: [<?php
                        echo implode(',', array_map(function($label) use ($payment_types_summary) {
                            foreach ($payment_types_summary as $type) {
                                if ($type['type_name'] === $label) {
                                    return $type['total_paid'];
                                }
                            }
                            return 0;
                        }, $all_labels));
                    ?>],
                    backgroundColor: '#28a745',
                    borderColor: '#28a745',
                    borderWidth: 1
                },
                {
                    label: 'Expenses',
                    data: [<?php
                        echo implode(',', array_map(function($label) use ($expense_types_summary) {
                            foreach ($expense_types_summary as $type) {
                                if ($type['type_name'] === $label) {
                                    return $type['total'];
                                }
                            }
                            return 0;
                        }, $all_labels));
                    ?>],
                    backgroundColor: '#dc3545',
                    borderColor: '#dc3545',
                    borderWidth: 1
                },
                {
                    label: 'Benefits',
                    data: [<?php
                        echo implode(',', array_map(function($label) use ($benefit_types_summary) {
                            foreach ($benefit_types_summary as $type) {
                                if ($type['type_name'] === $label) {
                                    return $type['total'];
                                }
                            }
                            return 0;
                        }, $all_labels));
                    ?>],
                    backgroundColor: '#17a2b8',
                    borderColor: '#17a2b8',
                    borderWidth: 1
                },
                {
                    label: 'Debts',
                    data: [<?php
                        echo implode(',', array_map(function($label) use ($payment_types_summary) {
                            foreach ($payment_types_summary as $type) {
                                if ($type['type_name'] === $label) {
                                    return $type['total_debt'];
                                }
                            }
                            return 0;
                        }, $all_labels));
                    ?>],
                    backgroundColor: '#ffc107',
                    borderColor: '#ffc107',
                    borderWidth: 1
                }
            ]
        };
        if (histData.datasets.every(ds => ds.data.every(d => d === 0))) {
            document.getElementById('financialHistogram').style.display = 'none';
        } else {
            document.getElementById('financialHistogram').style.display = 'block';
            new Chart(histCtx, {
                type: 'bar',
                data: histData,
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: {
                        y: {
                            beginAtZero: true,
                            title: {
                                display: true,
                                text: 'Amount (<?php echo htmlspecialchars($currency_symbol); ?>)'
                            }
                        },
                        x: {
                            title: {
                                display: true,
                                text: 'Type'
                            }
                        }
                    },
                    plugins: {
                        legend: {
                            display: true,
                            position: 'top'
                        }
                    }
                }
            });
        }

        // Line Chart
        const lineCtx = document.getElementById('financialLineChart').getContext('2d');
        const lineData = {
            labels: [<?php echo "'" . implode("','", array_map('htmlspecialchars', $all_labels)) . "'"; ?>],
            datasets: [
                {
                    label: 'Payments',
                    data: [<?php
                        echo implode(',', array_map(function($label) use ($payment_types_summary) {
                            foreach ($payment_types_summary as $type) {
                                if ($type['type_name'] === $label) {
                                    return $type['total_paid'];
                                }
                            }
                            return 0;
                        }, $all_labels));
                    ?>],
                    fill: false,
                    borderColor: '#28a745',
                    tension: 0.1
                },
                {
                    label: 'Expenses',
                    data: [<?php
                        echo implode(',', array_map(function($label) use ($expense_types_summary) {
                            foreach ($expense_types_summary as $type) {
                                if ($type['type_name'] === $label) {
                                    return $type['total'];
                                }
                            }
                            return 0;
                        }, $all_labels));
                    ?>],
                    fill: false,
                    borderColor: '#dc3545',
                    tension: 0.1
                },
                {
                    label: 'Benefits',
                    data: [<?php
                        echo implode(',', array_map(function($label) use ($benefit_types_summary) {
                            foreach ($benefit_types_summary as $type) {
                                if ($type['type_name'] === $label) {
                                    return $type['total'];
                                }
                            }
                            return 0;
                        }, $all_labels));
                    ?>],
                    fill: false,
                    borderColor: '#17a2b8',
                    tension: 0.1
                },
                {
                    label: 'Debts',
                    data: [<?php
                        echo implode(',', array_map(function($label) use ($payment_types_summary) {
                            foreach ($payment_types_summary as $type) {
                                if ($type['type_name'] === $label) {
                                    return $type['total_debt'];
                                }
                            }
                            return 0;
                        }, $all_labels));
                    ?>],
                    fill: false,
                    borderColor: '#ffc107',
                    tension: 0.1
                }
            ]
        };
        if (lineData.datasets.every(ds => ds.data.every(d => d === 0))) {
            document.getElementById('financialLineChart').style.display = 'none';
        } else {
            document.getElementById('financialLineChart').style.display = 'block';
            new Chart(lineCtx, {
                type: 'line',
                data: lineData,
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: {
                        y: {
                            beginAtZero: true,
                            title: {
                                display: true,
                                text: 'Amount (<?php echo htmlspecialchars($currency_symbol); ?>)'
                            }
                        },
                        x: {
                            title: {
                                display: true,
                                text: 'Type'
                            }
                        }
                    },
                    plugins: {
                        legend: {
                            display: true,
                            position: 'top'
                        }
                    }
                }
            });
        }
    <?php endif; ?>

    function applyFilters() {
        const recordType = document.getElementById('recordTypeFilter').value;
        const timePeriod = document.getElementById('timePeriodFilter').value;
        const filterDate = document.getElementById('filterDate').value;
        const url = new URL(window.location);
        url.searchParams.set('record_type', recordType);
        url.searchParams.set('time_period', timePeriod);
        url.searchParams.set('filter_date', filterDate);
        window.location = url;
    }

    function toggleDatePicker() {
        const timePeriod = document.getElementById('timePeriodFilter').value;
        document.getElementById('datePickerContainer').style.display = timePeriod === 'date' ? 'block' : 'none';
    }
});
</script>

<?php require_once 'includes/footer.php'; ?>