<?php
/* --------------------------------------------------------------
   templates/financial_summary_template.php
   (REFACTORED: Unified Admin & Transaction Report)
   (FIXED: Updated to use the new sidebar/header layout)
   -------------------------------------------------------------- */
$tables = ['payments', 'expenses', 'debts', 'benefits'];
$currency_symbol_pdf = 'NGN ';
require_once 'includes/header.php';
/* Safe fall-backs for dropdowns */
$type_options_payments = $type_options_payments ?? '<option value="">All Payment Types</option>';
$type_options_expenses = $type_options_expenses ?? '<option value="">All Expense Types</option>';
$type_options_benefits = $type_options_benefits ?? '<option value="">All Benefit Types</option>';
// --- Helper function for formatting currency in this template ---
if (!function_exists('format_currency_template')) {
    function format_currency_template($amount, $symbol) {
        return $symbol . number_format((float)$amount, 2);
    }
}

// === FILE-SPECIFIC INCLUDES ===
?>
<script>
document.title = 'Financial Report - <?php echo htmlspecialchars($system_name ?? 'Club Portal'); ?>';
</script>
<!-- CDN for DataTables and Chart.js -->
<link rel="stylesheet" href="https://cdn.datatables.net/1.13.7/css/jquery.dataTables.min.css">
<script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.7.1/jquery.min.js"></script>
<script src="https://cdn.datatables.net/1.13.7/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
<!-- jsPDF and autoTable for PDF export -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf-autotable/3.5.25/jspdf.plugin.autotable.min.js"></script>
<style>
/* Base styles */
.card { border:none; border-radius: 0.5rem; box-shadow: 0 4px 12px rgba(0,0,0,.05); }
.card-header { font-weight:bold; border-bottom:none; background-color: transparent; padding-top: 1rem; padding-bottom: 0.5rem;}
.table-responsive-max { max-height: 450px; overflow-y: auto; }
.kpi-card { text-align: center; }
.kpi-card .display-6 { font-weight: 600; } /* This was the original */
.kpi-card .card-body { padding: 1.25rem 1rem; }
.kpi-card h5 { font-size: 0.9rem; font-weight: 500; margin-bottom: 0.5rem; display: flex; align-items: center; justify-content: center; gap: 0.5rem; }
.filter-section { background:#f8f9fa; padding: 1.5rem; border-radius: 0.5rem; }
.no-data { font-style:italic; color:#6c757d; }
.dataTables_wrapper .dataTables_paginate .paginate_button { padding: 0.3em 0.8em; }
.dataTables_wrapper .dataTables_length, .dataTables_wrapper .dataTables_filter { margin-bottom: 1rem; }
/* Print-specific styles */
@media print{
    body *{visibility:hidden;}
    #printableReportArea, #printableReportArea * { visibility: visible; }
    #printableReportArea { position: static; width: 100%; page-break-inside: avoid; }
    .no-print{display:none !important;}
    table{border-collapse:collapse;width:100%; page-break-inside: avoid;}
    th,td{border:1px solid #ddd;padding:8px;text-align:left;}
    th{background:#f2f2f2;font-weight:bold;}
    .card{border:none;box-shadow:none;}
    .table-responsive, .table-responsive-max { max-height: none; overflow-y: visible; }
}
</style>

<!-- 1. Include the Sidebar -->
<?php require_once 'includes/sidebar.php'; ?>

<!-- 
  2. START THE MAIN CONTENT AREA 
  - REMOVED old layout classes: "col-md-9 ms-sm-auto col-lg-10 px-md-4"
  - ADDED the correct class: "main-content"
-->
<main class="main-content">
    
    <!-- 
      3. WRAP all page content in a "container-fluid" 
      This replaces the old <div class="container-fluid"> that wrapped both sidebar and main
    -->
    <div class="container-fluid">
        <!-- ================================================================== -->
        <!-- !! UNIFIED REPORT AREA !! -->
        <!-- ================================================================== -->
        <section id="printableReportArea">
            <div class="text-center pt-3 pb-2 mb-3 border-bottom no-print">
                <h1 class="display-5 mb-3">Financial Report</h1>
                <p class="lead text-muted">Unified ledger and transaction report.</p>
            </div>
            <?php if ($is_admin && $admin_message): ?>
                <div class="alert alert-warning no-print" role="alert">
                    <i class="fas fa-exclamation-triangle me-2"></i><?= htmlspecialchars($admin_message) ?>
                </div>
            <?php endif; ?>
            <!-- ================================================================== -->
            <!-- !! UNIFIED FILTERS !! -->
            <!-- ================================================================== -->
            <div class="card shadow-sm mb-4 filter-section no-print">
                <form id="filterForm" method="GET" action="financial_summary.php">
                    <div class="row align-items-end">
                        <!-- Admin Date Filters -->
                        <div class="col-lg-2 col-md-4 mb-3">
                            <label for="admin_from_date" class="form-label fw-bold">From Date</label>
                            <input type="date" id="admin_from_date" name="admin_from_date" value="<?= htmlspecialchars($admin_from_date) ?>" class="form-control">
                        </div>
                        <div class="col-lg-2 col-md-4 mb-3">
                            <label for="admin_to_date" class="form-label fw-bold">To Date</label>
                            <input type="date" id="admin_to_date" name="admin_to_date" value="<?= htmlspecialchars($admin_to_date) ?>" class="form-control">
                        </div>
                        <!-- Transaction Record Type Filter -->
                        <div class="col-lg-2 col-md-4 mb-3">
                            <label for="record_type" class="form-label fw-bold">Record Type</label>
                            <select id="record_type" name="record_type" class="form-select">
                                <option value="all" <?php echo $record_type==='all'?'selected':''; ?>>All Types</option>
                                <option value="payments" <?php echo $record_type==='payments'?'selected':''; ?>>Payments</option>
                                <option value="expenses" <?php echo $record_type==='expenses'?'selected':''; ?>>Expenses</option>
                                <option value="benefits" <?php echo $record_type==='benefits'?'selected':''; ?>>Benefits</option>
                                <option value="debts" <?php echo $record_type==='debts'?'selected':''; ?>>Debts</option>
                            </select>
                        </div>
                       
                        <!-- Transaction Member Filter -->
                        <div class="col-lg-3 col-md-4 mb-3">
                            <label for="member_id" class="form-label fw-bold">Member</label>
                            <select id="member_id" name="member_id" class="form-select">
                                <option value="0">All Members</option>
                                <?php foreach ($members as $m): ?>
                                    <option value="<?php echo $m['id']; ?>" <?php echo $member_id==$m['id']?'selected':''; ?>>
                                        <?php echo htmlspecialchars($m['full_name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <!-- Action Buttons -->
                        <div class="col-lg-3 col-md-12 mb-3 d-flex align-items-end justify-content-start gap-2">
                            <button type="submit" class="btn btn-primary"><i class="fas fa-check me-1"></i>Apply Filters</button>
                            <a href="financial_summary.php" class="btn btn-outline-secondary">Reset</a>
                            <button class="btn btn-outline-info" type="button" data-bs-toggle="collapse" data-bs-target="#moreFilters" aria-expanded="false" aria-controls="moreFilters">
                                More <i class="fas fa-chevron-down small"></i>
                            </button>
                        </div>

                        <!-- --- NEW: Quick Date Presets --- -->
                        <div class="col-12 mt-2">
                            <label class="form-label fw-bold">Quick Date Presets:</label>
                            <div class="d-flex flex-wrap gap-2">
                                <button type="button" class="btn btn-sm btn-outline-success" id="btnThisMonth">This Month</button>
                                <button type="button" class="btn btn-sm btn-outline-success" id="btnLastMonth">Last Month</button>
                                <button type="button" class="btn btn-sm btn-outline-info" id="btnThisQuarter">This Quarter</button>
                                <button type="button" class="btn btn-sm btn-outline-info" id="btnThisYear">This Year</button>
                            </div>
                        </div>
                        <!-- --- END: Quick Date Presets --- -->

                        <!-- Collapsible "More Filters" -->
                        <div class="collapse" id="moreFilters">
                            <div class="row pt-3">
                                <div class="col-lg-3 col-md-4 mb-3">
                                    <label for="search" class="form-label">Search Term</label>
                                    <input type="text" id="search" name="search" class="form-control" value="<?php echo htmlspecialchars($search); ?>" placeholder="e.g., 'Maintenance' or ID">
                                </div>
                                <div class="col-lg-3 col-md-4 mb-3">
                                    <label for="time_period" class="form-label">Transaction Period</label>
                                    <select id="time_period" name="time_period" class="form-select" onchange="toggleDatePicker()">
                                        <option value="all" <?php echo $time_period==='all'?'selected':''; ?>>All Time</option>
                                        <option value="range" <?php echo $time_period==='range'?'selected':''; ?>>Custom Range</option>
                                        <option value="week" <?php echo $time_period==='week'?'selected':''; ?>>Last Week</option>
                                        <option value="month" <?php echo $time_period==='month'?'selected':''; ?>>Last Month</option>
                                        <option value="year" <?php echo $time_period==='year'?'selected':''; ?>>Last Year</option>
                                    </select>
                                </div>
                                <div class="col-lg-3 col-md-4 mb-3" id="datePickerContainer" style="display:<?php echo $time_period==='range'?'block':'none'; ?>;">
                                    <label class="form-label">Custom Start Date:</label>
                                    <input type="date" name="start_date" class="form-control" value="<?php echo $start_date; ?>">
                                    <label class="form-label mt-2">Custom End Date:</label>
                                    <input type="date" name="end_date" class="form-control" value="<?php echo $end_date; ?>">
                                </div>
                                <div class="col-lg-3 col-md-4 mb-3">
                                    <label for="date_field" class="form-label">Debt Date Field</label>
                                    <select id="date_field" name="date_field" class="form-select">
                                        <option value="created_at" <?php echo $date_field==='created_at'?'selected':''; ?>>Created At</option>
                                        <option value="due_date" <?php echo $date_field==='due_date'?'selected':''; ?>>Due Date</option>
                                    </select>
                                </div>
                                <div class="col-lg-3 col-md-4 mb-3">
                                    <label for="type_filter_payments" class="form-label">Payment Type</label>
                                    <select id="type_filter_payments" name="type_filter_payments" class="form-select">
                                        <option value="">All Payment Types</option>
                                        <?php echo $type_options_payments; ?>
                                    </select>
                                </div>
                                <div class="col-lg-3 col-md-4 mb-3">
                                    <label for="type_filter_expenses" class="form-label">Expense Type</label>
                                    <select id="type_filter_expenses" name="type_filter_expenses" class="form-select">
                                        <option value="">All Expense Types</option>
                                        <?php echo $type_options_expenses; ?>
                                    </select>
                                </div>
                                <div class="col-lg-3 col-md-4 mb-3">
                                    <label for="type_filter_benefits" class="form-label">Benefit Type</label>
                                    <select id="type_filter_benefits" name="type_filter_benefits" class="form-select">
                                        <option value="">All Benefit Types</option>
                                        <?php echo $type_options_benefits; ?>
                                    </select>
                                </div>
                            </div>
                        </div>
                    </div>
                   
                    <!-- Hidden fields for pagination -->
                    <?php foreach ($tables as $t): ?>
                        <input type="hidden" name="page_<?php echo $t; ?>" value="<?php echo $pagination[$t]['page']; ?>">
                        <input type="hidden" name="per_page_<?php echo $t; ?>" value="<?php echo $pagination[$t]['per_page']; ?>">
                    <?php endforeach; ?>
                </form>
            </div>
            <?php if ($is_admin): ?>
            <!-- ================================================================== -->
            <!-- !! UNIFIED KPI CARDS (ADMIN-ONLY) !! -->
            <!-- ================================================================== -->
            <div class="row mb-4">
                <div class="col-lg-3 col-md-6 mb-3">
                    <div class="card bg-success text-white kpi-card">
                        <div class="card-body">
                            <h5><i class="fas fa-arrow-up"></i>Total Inflows (Ledger)</h5>
                            <!-- FIX 1: Font size changed from display-6 to h3 -->
                            <p class="h3 fw-bold"><?= format_currency_template($admin_kpis['inflows'], $currency_symbol) ?></p>
                        </div>
                    </div>
                </div>
                <div class="col-lg-3 col-md-6 mb-3">
                    <div class="card bg-danger text-white kpi-card">
                        <div class="card-body">
                            <h5><i class="fas fa-arrow-down"></i>Total Outflows (Ledger)</h5>
                            <!-- FIX 1: Font size changed from display-6 to h3 -->
                            <p class="h3 fw-bold"><?= format_currency_template($admin_kpis['outflows'], $currency_symbol) ?></p>
                        </div>
                    </div>
                </div>
                <div class="col-lg-3 col-md-6 mb-3">
                    <div class="card bg-info text-white kpi-card">
                        <div class="card-body">
                            <h5><i class="fas fa-balance-scale"></i>Net Change (Ledger)</h5>
                            <!-- FIX 1: Font size changed from display-6 to h3 -->
                            <p class="h3 fw-bold"><?= format_currency_template($admin_kpis['net'], $currency_symbol) ?></p>
                        </div>
                    </div>
                </div>
                <div class="col-lg-3 col-md-6 mb-3">
                    <div class="card bg-warning text-dark kpi-card">
                        <div class="card-body">
                            <h5><i class="fas fa-exclamation-circle"></i>Total Debts (Txn)</h5>
                            <!-- FIX 1: Font size changed from display-6 to h3 -->
                            <p class="h3 fw-bold"><?= format_currency_template($total_debts, $currency_symbol) ?></p>
                        </div>
                    </div>
                </div>
                <div class="col-lg-4 col-md-6 mb-3">
                    <div class="card bg-light text-dark kpi-card">
                        <div class="card-body">
                            <h5><i class="fas fa-sign-in-alt"></i>Opening Balance (<?= $admin_from_date ?>)</h5>
                            <h4 class="fw-bold"><?= format_currency_template($admin_opening_balance, $currency_symbol) ?></h4>
                        </div>
                    </div>
                </div>
                <div class="col-lg-4 col-md-6 mb-3">
                    <div class="card bg-dark text-white kpi-card">
                        <div class="card-body">
                            <h5><i class="fas fa-sign-out-alt"></i>Closing Balance (<?= $admin_to_date ?>)</h5>
                            <h4 class="fw-bold"><?= format_currency_template($admin_closing_balance, $currency_symbol) ?></h4>
                        </div>
                    </div>
                </div>
                 <div class="col-lg-4 col-md-12 mb-3">
                    <div class="card bg-white kpi-card">
                        <div class="card-body">
                            <h5><i class="fas fa-university"></i>Total Liquid Assets (Current)</h5>
                            <h4 class="fw-bold text-primary"><?= format_currency_template($admin_total_liquid_assets, $currency_symbol) ?></h4>
                        </div>
                    </div>
                </div>
            </div>
            <!-- ================================================================== -->
            <!-- !! ADMIN LEDGER SUMMARIES & CHARTS !! -->
            <!-- ================================================================== -->
            <div class="row">
                <!-- Admin Bank Balances -->
                <div class="col-lg-5 col-md-12 mb-4">
                    <div class="card shadow h-100">
                        <div class="card-header bg-light">
                            <h5 class="mb-0"><i class="fas fa-wallet me-2"></i>Current Bank & Cash Balances</h5>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-sm" id="adminBalanceSummary">
                                    <thead class="table-light">
                                        <tr>
                                            <th>Account Name</th>
                                            <th class="text-end">Current Balance</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($admin_account_balances as $acc): ?>
                                            <tr>
                                                <td><?= htmlspecialchars($acc['account_name']) ?></td>
                                                <td class="text-end fw-bold"><?= format_currency_template($acc['current_balance'], $currency_symbol) ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                    <tfoot class="table-group-divider">
                                        <tr>
                                            <th class="text-end">Total Liquid Assets:</th>
                                            <th class="text-end text-primary fw-bold"><?= format_currency_template($admin_total_liquid_assets, $currency_symbol) ?></th>
                                        </tr>
                                    </tfoot>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
               
                <!-- Admin Charts -->
                <div class="col-lg-7 col-md-12 mb-4">
                    <div class="card shadow h-100 no-print">
                        <div class="card-header bg-light">
                            <h5 class="mb-0"><i class="fas fa-chart-bar me-2"></i>Daily Trend (<?= $admin_from_date ?> to <?= $admin_to_date ?>)</h5>
                        </div>
                        <div class="card-body" style="min-height: 300px;">
                            <canvas id="adminBarChart"></canvas>
                        </div>
                    </div>
                </div>
                <!-- Admin Fund Requests -->
                <div class="col-lg-6 col-md-12 mb-4">
                    <div class="card shadow h-100">
                        <div class="card-header bg-light d-flex justify-content-between align-items-center">
                            <h5 class="mb-0"><i class="fas fa-tasks me-2"></i>Fund Requests (Period)</h5>
                            <select id="admin_status_filter" class="form-select form-select-sm no-print" onchange="window.location.href='financial_summary.php?admin_from_date=<?= $admin_from_date ?>&admin_to_date=<?= $admin_to_date ?>&admin_status=' + this.value" style="width: auto;">
                                <option value="all" <?= $admin_filter_status == 'all' ? 'selected' : '' ?>>All Status</option>
                                <option value="pending" <?= $admin_filter_status == 'pending' ? 'selected' : '' ?>>Pending</option>
                                <option value="approved" <?= $admin_filter_status == 'approved' ? 'selected' : '' ?>>Approved</option>
                                <option value="rejected" <?= $admin_filter_status == 'rejected' ? 'selected' : '' ?>>Rejected</option>
                            </select>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive table-responsive-max">
                                <table class="table table-striped table-hover table-sm" id="adminFundRequestsTable">
                                    <thead>
                                        <tr>
                                            <th>ID</th>
                                            <th>User</th>
                                            <th class="text-end">Amount</th>
                                            <th>Status</th>
                                            <th>Date</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php if (empty($admin_fund_requests)): ?>
                                            <tr class="text-center text-muted"><td colspan="5">No fund requests found in period.</td></tr>
                                        <?php endif; ?>
                                        <?php foreach ($admin_fund_requests as $req):
                                            $req_date_formatted = !empty($req['display_date']) ? date('Y-m-d H:i', strtotime($req['display_date'])) : 'Unknown Date';
                                            $user_display = htmlspecialchars($req['user_display_name'] ?? 'System Request');
                                            ?>
                                            <tr>
                                                <td><?= htmlspecialchars($req['id']) ?></td>
                                                <td><?= $user_display ?></td>
                                                <td class="text-end"><?= format_currency_template($req['amount'], $currency_symbol) ?></td>
                                                <td><span class="badge bg-<?= $req['status'] == 'approved' ? 'success' : ($req['status'] == 'rejected' ? 'danger' : 'warning') ?>"><?= ucfirst(htmlspecialchars($req['status'])) ?></span></td>
                                                <td><?= $req_date_formatted ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
               
                <div class="col-lg-6 mb-4">
                    <div class="card shadow h-100 no-print">
                        <div class="card-header bg-light">
                            <h5 class="mb-0"><i class="fas fa-chart-pie me-2"></i>Inflows vs Outflows (Period)</h5>
                        </div>
                        <div class="card-body" style="min-height: 300px;">
                            <canvas id="adminPieChart"></canvas>
                        </div>
                    </div>
                </div>
               
                <!-- Admin Ledger History -->
                <div class="col-12 mb-4">
                    <div class="card shadow">
                        <div class="card-header bg-light d-flex justify-content-between align-items-center">
                            <h5 class="mb-0"><i class="fas fa-book me-2"></i>Financial Ledger History (Period)</h5>
                            <div class="no-print">
                                <button onclick="window.print()" class="btn btn-sm btn-outline-secondary">
                                    <i class="fas fa-print me-1"></i> Print
                                </button>
                                <!-- UPDATED: PDF Export Button type -->
                                <button type="button" id="exportAdminReportBtn" class="btn btn-sm btn-danger">
                                    <i class="fas fa-file-pdf me-1"></i> Export Full PDF Report
                                </button>
                            </div>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-bordered table-striped table-sm" id="adminLedgerTable">
                                    <thead>
                                        <tr>
                                            <th>Date</th>
                                            <th>Account</th>
                                            <th>Description</th>
                                            <th class="text-end">Debit (Inflow)</th>
                                            <th class="text-end">Credit (Outflow)</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php if (empty($admin_ledger_history)): ?>
                                            <tr class="text-center text-muted"><td colspan="5">No ledger history found for this period.</td></tr>
                                        <?php endif; ?>
                                        <?php foreach ($admin_ledger_history as $entry):
                                            $date_formatted = !empty($entry['date_recorded']) ? date('Y-m-d H:i', strtotime($entry['date_recorded'])) : 'N/A';
                                            // --- FIX: Updated to include new transfer types ---
                                            $debit = (in_array($entry['transaction_type'], ['income', 'transfer_in'])) ? (float)$entry['amount'] : 0;
                                            $credit = (in_array($entry['transaction_type'], ['expense', 'benefit', 'transfer_out'])) ? (float)$entry['amount'] : 0;
                                            ?>
                                            <tr>
                                                <td><?= $date_formatted ?></td>
                                                <td><?= htmlspecialchars($entry['account_name'] ?? 'N/A') ?></td>
                                                <td><?= htmlspecialchars($entry['description']) ?></td>
                                                <td class="text-end text-success fw-bold"><?= $debit > 0 ? format_currency_template($debit, $currency_symbol) : '' ?></td>
                                                <td class="text-end text-danger fw-bold"><?= $credit > 0 ? format_currency_template($credit, $currency_symbol) : '' ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
       
            <hr class="my-5 no-print">
        <?php endif; ?>
        <!-- ================================================================== -->
        <!-- !! END OF ADMIN-ONLY REPORT !! -->
        <!-- ================================================================== -->
        <!-- ================================================================== -->
        <!-- !! DETAILED TRANSACTION REPORT !! -->
        <!-- ================================================================== -->
        <div class="text-center pt-3 pb-2 mb-3 border-bottom">
             <h2 class="display-6 mb-3">Detailed Transaction Logs</h2>
             <p class="lead text-muted">A filterable list of individual transactions (payments, expenses, etc.).</p>
        </div>
        <!-- ==================== SUMMARIES ==================== -->
        <?php
        $summaries = [
            ['title'=>'Payment Types Summary','data'=>$payment_types_summary ?? [],'table'=>'payments'],
            ['title'=>'Expense Types Summary','data'=>$expense_types_summary ?? [],'table'=>'expenses'],
            ['title'=>'Benefit Types Summary','data'=>$benefit_types_summary ?? [],'table'=>'benefits']
        ];
        foreach ($summaries as $s):
            
            // --- NEW: Calculate Totals ---
            $total_required = 0;
            $total_paid = 0;
            $total_debt = 0;
            $total_excess = 0;
            $total_simple = 0;

            if (!empty($s['data'])) {
                foreach ($s['data'] as $row) {
                    if ($s['title'] === 'Payment Types Summary') {
                        $total_required += (float)($row['required_amount'] ?? 0);
                        $total_paid += (float)($row['total_paid'] ?? 0);
                        $total_debt += (float)($row['total_debt'] ?? 0);
                        $total_excess += (float)($row['excess_amount'] ?? 0);
                    } else {
                        $total_simple += (float)($row['total'] ?? 0);
                    }
                }
            }
            // --- END: Calculate Totals ---

            if (($record_type==='all' || $record_type===$s['table']) && !empty($s['data'])): ?>
                <div class="card mb-4 print-card">
                    <!-- --- NEW: Added PDF Export Button to Card Header --- -->
                    <div class="card-header bg-light d-flex justify-content-between align-items-center">
                        <span><?php echo htmlspecialchars($s['title']); ?> (Transaction Data)</span>
                        <?php if ($is_admin): ?>
                            <button type="button" class="btn btn-sm btn-outline-danger no-print" id="export<?php echo str_replace(' ', '', $s['table']); ?>SummaryPdf">
                                <i class="fas fa-file-pdf"></i> PDF
                            </button>
                        <?php endif; ?>
                    </div>
                    <div class="card-body table-responsive">
                        <table class="table table-striped table-sm">
                            <thead>
                                <tr>
                                    <th>Type Name</th>
                                    <?php if ($s['title']==='Payment Types Summary'): ?>
                                        <th class="text-end">Required</th><th class="text-end">Total Paid</th><th class="text-end">Total Debt</th><th class="text-end">Excess</th>
                                    <?php else: ?>
                                        <th class="text-end">Total</th>
                                    <?php endif; ?>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($s['data'] as $row): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($row['type_name']); ?></td>
                                        <?php if ($s['title']==='Payment Types Summary'): ?>
                                            <td class="text-end"><?php echo format_currency_template($row['required_amount'], $currency_symbol); ?></td>
                                            <td class="text-end"><?php echo format_currency_template($row['total_paid'], $currency_symbol); ?></td>
                                            <td class="text-end"><?php echo format_currency_template($row['total_debt'], $currency_symbol); ?></td>
                                            <td class="text-end"><?php echo format_currency_template($row['excess_amount'], $currency_symbol); ?></td>
                                        <?php else: ?>
                                            <td class="text-end"><?php echo format_currency_template($row['total'], $currency_symbol); ?></td>
                                        <?php endif; ?>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                            <!-- NEW: Total Row Footer -->
                            <tfoot class="table-group-divider">
                                <tr class="fw-bold" style="border-top: 2px solid #212529;">
                                    <td>Total</td>
                                    <?php if ($s['title'] === 'Payment Types Summary'): ?>
                                        <td class="text-end"><?php echo format_currency_template($total_required, $currency_symbol); ?></td>
                                        <td class="text-end"><?php echo format_currency_template($total_paid, $currency_symbol); ?></td>
                                        <td class="text-end"><?php echo format_currency_template($total_debt, $currency_symbol); ?></td>
                                        <td class="text-end"><?php echo format_currency_template($total_excess, $currency_symbol); ?></td>
                                    <?php else: ?>
                                        <td class="text-end"><?php echo format_currency_template($total_simple, $currency_symbol); ?></td>
                                    <?php endif; ?>
                                </tr>
                            </tfoot>
                            <!-- END: Total Row Footer -->
                        </table>
                    </div>
                </div>
            <?php endif;
        endforeach; ?>
        <!-- ==================== DETAILED TABLES ==================== -->
        <?php
        $tableInfo = [
            ['title'=>'Payments','data'=>$payments,'table'=>'payments','cnt'=>$total_counts['payments']],
            ['title'=>'Expenses','data'=>$expenses,'table'=>'expenses','cnt'=>$total_counts['expenses']],
            ['title'=>'Benefits','data'=>$benefits,'table'=>'benefits','cnt'=>$total_counts['benefits']],
            ['title'=>'Debts','data'=>$debts,'table'=>'debts','cnt'=>$total_counts['debts']]
        ];
        foreach ($tableInfo as $ti):
            if ($record_type!=='all' && $record_type!==$ti['table']) continue; ?>
                <div class="card mb-4 print-card">
                    <div class="card-header bg-light d-flex justify-content-between no-print">
                        <span><?php echo htmlspecialchars($ti['title']); ?> (Total: <?php echo $ti['cnt']; ?>)</span>
                        <a href="?export=<?php echo $ti['table']; ?>&<?php echo http_build_query($_GET); ?>" class="btn btn-outline-secondary btn-sm">
                            <i class="fas fa-file-csv me-1"></i>Export CSV
                        </a>
                    </div>
                    <div class="card-header d-none d-print-block"> <!-- Visible only on print -->
                         <?php echo htmlspecialchars($ti['title']); ?> (Total: <?php echo $ti['cnt']; ?>)
                    </div>
                    <div class="card-body table-responsive">
                        <table class="table table-striped table-hover table-sm">
                            <thead>
                                <tr>
                                    <?php if ($ti['title']==='Payments'): ?>
                                        <th>ID</th><th>Full Name</th><th>Payment Type</th><th class="text-end">Amount</th><th>Status</th><th>Created At</th><th>Payment Cycle</th>
                                        <?php if ($is_admin): ?><th class="no-print">Actions</th><?php endif; ?>
                                    <?php elseif ($ti['title']==='Expenses'): ?>
                                        <th>ID</th><th>Expense Type</th><th class="text-end">Amount</th><th>Description</th><th>Created At</th>
                                        <?php if ($is_admin): ?><th class="no-print">Actions</th><?php endif; ?>
                                    <?php elseif ($ti['title']==='Benefits'): ?>
                                        <th>ID</th><th>Member ID</th><th>Full Name</th><th>Benefit Type</th><th class="text-end">Amount</th><th>Created At</th>
                                        <?php if ($is_admin): ?><th class="no-print">Actions</th><?php endif; ?>
                                    <?php elseif ($ti['title']==='Debts'): ?>
                                        <th>Member ID</th><th>Full Name</th><th>Payment Type</th><th class="text-end">Required</th><th class="text-end">Total Paid</th><th class="text-end">Debt</th><th>Payment Cycle</th><th>Due Date</th>
                                        <?php if ($is_admin): ?><th class="no-print">Actions</th><?php endif; ?>
                                    <?php endif; ?>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($ti['data'])): ?>
                                    <tr class="text-center"><td colspan="10" class="no-data">No data found for this period.</td></tr>
                                <?php endif; ?>
                                <?php foreach ($ti['data'] as $row): ?>
                                    <tr>
                                        <?php if ($ti['title']==='Payments'): ?>
                                            <td><?php echo $row['id']; ?></td>
                                            <td><?php echo htmlspecialchars($row['full_name']); ?></td>
                                            <td><?php echo htmlspecialchars($row['payment_type']); ?></td>
                                            <td class="text-end"><?php echo format_currency_template($row['amount'], $currency_symbol); ?></td>
                                            <td><?php echo htmlspecialchars($row['payment_status']); ?></td>
                                            <td><?php echo date('Y-m-d', strtotime($row['created_at'])); ?></td>
                                            <td><?php echo htmlspecialchars($row['payment_cycle'] ?? 'N/A'); ?></td>
                                            <?php if ($is_admin): ?>
                                                <td class="no-print">
                                                    <a href="edit_payment.php?id=<?php echo $row['id']; ?>" class="btn btn-sm btn-outline-primary">Edit</a>
                                                    <a href="delete_payment.php?id=<?php echo $row['id']; ?>" class="btn btn-sm btn-outline-danger" onclick="return confirm('Delete?');">Delete</a>
                                                </td>
                                            <?php endif; ?>
                                        <?php elseif ($ti['title']==='Expenses'): ?>
                                            <td><?php echo $row['id']; ?></td>
                                            <td><?php echo htmlspecialchars($row['expense_type'] ?? 'N/A'); ?></td>
                                            <td class="text-end"><?php echo format_currency_template($row['amount'], $currency_symbol); ?></td>
                                            <td><?php echo htmlspecialchars($row['description']); ?></td>
                                            <td><?php echo date('Y-m-d', strtotime($row['created_at'])); ?></td>
                                            <?php if ($is_admin): ?>
                                                <td class="no-print">
                                                    <a href="edit_expense.php?id=<?php echo $row['id']; ?>" class="btn btn-sm btn-outline-primary">Edit</a>
                                                    <a href="delete_expense.php?id=<?php echo $row['id']; ?>" class="btn btn-sm btn-outline-danger" onclick="return confirm('Delete?');">Delete</a>
                                                </td>
                                            <?php endif; ?>
                                        <?php elseif ($ti['title']==='Benefits'): ?>
                                            <td><?php echo $row['id']; ?></td>
                                            <td><?php echo $row['member_id']; ?></td>
                                            <td><?php echo htmlspecialchars($row['full_name']); ?></td>
                                            <td><?php echo htmlspecialchars($row['benefit_type']); ?></td>
                                            <td class="text-end"><?php echo format_currency_template($row['amount'], $currency_symbol); ?></td>
                                            <td><?php echo date('Y-m-d', strtotime($row['created_at'])); ?></td>
                                            <?php if ($is_admin): ?>
                                                <td class="no-print">
                                                    <a href="edit_benefit.php?id=<?php echo $row['id']; ?>" class="btn btn-sm btn-outline-primary">Edit</a>
                                                    <a href="delete_benefit.php?id=<?php echo $row['id']; ?>" class="btn btn-sm btn-outline-danger" onclick="return confirm('Delete?');">Delete</a>
                                                </td>
                                            <?php endif; ?>
                                        <?php elseif ($ti['title']==='Debts'): ?>
                                            <td><?php echo $row['member_id']; ?></td>
                                            <td><?php echo htmlspecialchars($row['full_name']); ?></td>
                                            <td><?php echo htmlspecialchars($row['payment_type']); ?></td>
                                            <td class="text-end"><?php echo format_currency_template($row['required_amount'], $currency_symbol); ?></td>
                                            <td class="text-end"><?php echo format_currency_template($row['total_paid'], $currency_symbol); ?></td>
                                            <td class="text-end text-danger"><?php echo format_currency_template($row['debt'], $currency_symbol); ?></td>
                                            <td><?php echo htmlspecialchars($row['payment_cycle'] ?? 'N/A'); ?></td>
                                            <td><?php echo htmlspecialchars($row['due_date'] ?? 'N/A'); ?></td>
                                            <?php if ($is_admin): ?>
                                                <td class="no-print">
                                                    <a href="edit_member.php?id=<?php echo $row['user_id']; ?>" class="btn btn-sm btn-outline-primary">Edit Member</a>
                                                </td>
                                            <?php endif; ?>
                                        <?php endif; ?>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    <!-- Pagination -->
                    <nav aria-label="Page navigation" class="p-3 no-print">
                        <ul class="pagination pagination-sm justify-content-end mb-0">
                            <?php if ($pagination[$ti['table']]['page'] > 1): ?>
                                <li class="page-item"><a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page_'.$ti['table'] => $pagination[$ti['table']]['page']-1])); ?>">Prev</a></li>
                            <?php endif; ?>
                           
                            <li class="page-item disabled"><span class="page-link">Page <?php echo $pagination[$ti['table']]['page']; ?> of <?php echo $total_pages[$ti['table']]; ?></span></li>
                            <?php if ($pagination[$ti['table']]['page'] < $total_pages[$ti['table']]): ?>
                                <li class="page-item"><a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page_'.$ti['table'] => $pagination[$ti['table']]['page']+1])); ?>">Next</a></li>
                            <?php endif; ?>
                            <li class="page-item ms-2">
                                <select class="form-select form-select-sm d-inline w-auto" onchange="updatePerPage('<?php echo $ti['table']; ?>',this.value)">
                                    <?php foreach ([5,10,25,50,100] as $pp): ?>
                                        <option value="<?php echo $pp; ?>" <?php echo $pagination[$ti['table']]['per_page']==$pp?'selected':''; ?>><?php echo $pp; ?> per page</option>
                                    <?php endforeach; ?>
                                </select>
                            </li>
                        </ul>
                    </nav>
                </div>
            <?php endforeach; ?>
        </section>
    </div> <!-- END .container-fluid -->
</main> <!-- END .main-content -->

<script>
// ==================================================================
// !! UNIFIED SCRIPT SECTION !!
// ==================================================================
$(document).ready(function() {
    // --- Admin DataTables ---
    <?php if ($is_admin): ?>
    $('#adminBalanceSummary').DataTable({ "paging": false, "searching": false, "info": false, "ordering": false });
   
    // FIX: Check for colspan to avoid errors on "No data" row
    if ($('#adminFundRequestsTable tbody tr').length > 1 || ($('#adminFundRequestsTable tbody tr').length === 1 && !$($('#adminFundRequestsTable tbody td')[0]).attr('colspan'))) {
        $('#adminFundRequestsTable').DataTable({
            "paging": true, "searching": true, "ordering": true, "info": true,
            "lengthChange": false, "pageLength": 5, "order": [[ 4, "desc" ]]
        });
    }
   
    // FIX: Check for colspan to avoid errors on "No data" row
    if ($('#adminLedgerTable tbody tr').length > 1 || ($('#adminLedgerTable tbody tr').length === 1 && !$($('#adminLedgerTable tbody td')[0]).attr('colspan'))) {
        $('#adminLedgerTable').DataTable({
            "paging": true, "searching": true, "ordering": true, "info": true,
            "lengthChange": true, "pageLength": 10, "order": [[ 0, "desc" ]]
        });
    }
    // --- Admin Chart.js (Pie Chart) ---
    const ctxPieAdmin = document.getElementById('adminPieChart');
    if (ctxPieAdmin) {
        new Chart(ctxPieAdmin.getContext('2d'), {
            type: 'pie',
            data: <?= $admin_pie_chart_data ?>,
            options: {
                responsive: true, maintainAspectRatio: false,
                plugins: { legend: { position: 'top' } }
            }
        });
    }
    // --- Admin Chart.js (Bar Chart) ---
    const ctxBarAdmin = document.getElementById('adminBarChart');
    if (ctxBarAdmin) {
        new Chart(ctxBarAdmin.getContext('2d'), {
            type: 'bar',
            data: <?= $admin_bar_chart_data ?>,
            options: {
                responsive: true, maintainAspectRatio: false,
                plugins: { legend: { position: 'top' } },
                scales: { x: { stacked: false }, y: { stacked: false, beginAtZero: true } }
            }
        });
    }
    
    // ==================================================================
    // !! PDF EXPORT LOGIC !!
    // ==================================================================
    
    // --- Helper function to get base PDF settings ---
    function getPdfBaseData() {
        return {
            jsPDF: window.jspdf.jsPDF,
            doc: new window.jspdf.jsPDF('landscape', 'pt', 'a4'),
            orgName: "<?= htmlspecialchars($system_name ?? 'Club Portal', ENT_QUOTES) ?>",
            fromDate: "<?= htmlspecialchars($admin_from_date, ENT_QUOTES) ?>",
            toDate: "<?= htmlspecialchars($admin_to_date, ENT_QUOTES) ?>",
            currency: "<?= htmlspecialchars($currency_symbol_pdf, ENT_QUOTES) ?>",
            startY: 40
        };
    }

    // --- Main "Full Report" Button ---
    $('#exportAdminReportBtn').on('click', function() {
        try {
            const pdf = getPdfBaseData();
            
            // --- 2. Title ---
            pdf.doc.setFontSize(14);
            pdf.doc.text(`Full Financial Report for ${pdf.orgName}`, 40, 20);
            pdf.doc.setFontSize(10);
            pdf.doc.text(`Period: ${pdf.fromDate} to ${pdf.toDate} | Report Date: ` + new Date().toLocaleDateString(), 40, 30);
            
            // --- 3. Add Content to PDF ---
            // Table 1: KPIs
            pdf.doc.autoTable({
                startY: pdf.startY,
                theme: 'striped',
                head: [['Metric', 'Amount']],
                body: [
                    ['Total Inflows (Period)', pdf.currency + '<?= number_format($admin_kpis['inflows'], 2) ?>'],
                    ['Total Outflows (Period)', pdf.currency + '<?= number_format($admin_kpis['outflows'], 2) ?>'],
                    ['Net Change (Period)', pdf.currency + '<?= number_format($admin_kpis['net'], 2) ?>'],
                    ['Opening Balance (as of <?= $admin_from_date ?>)', pdf.currency + '<?= number_format($admin_opening_balance, 2) ?>'],
                    ['Closing Balance (as of <?= $admin_to_date ?>)', pdf.currency + '<?= number_format($admin_closing_balance, 2) ?>'],
                ]
            });
            // Table 2: Balances
            pdf.startY = pdf.doc.lastAutoTable.finalY + 10;
            pdf.doc.autoTable({
                startY: pdf.startY,
                theme: 'striped',
                head: [['Account Name', 'Current Balance']],
                body: <?= json_encode(array_map(function($acc) use ($currency_symbol_pdf) { return [$acc['account_name'], $currency_symbol_pdf . number_format($acc['current_balance'], 2)]; }, $admin_account_balances)) ?>,
                foot: [
                    ['Total Liquid Assets (Current)', pdf.currency + '<?= number_format($admin_total_liquid_assets, 2) ?>']
                ]
            });
            // Table 3: Fund Requests
            pdf.startY = pdf.doc.lastAutoTable.finalY + 10;
            pdf.doc.setFontSize(12);
            pdf.doc.text("Fund Requests", 40, pdf.startY);
            pdf.doc.autoTable({
                startY: pdf.startY + 5,
                theme: 'grid',
                head: [['ID', 'User', 'Amount', 'Status', 'Date']],
                body: <?= json_encode(array_map(function($req) use ($currency_symbol_pdf) {
                    return [
                        $req['id'],
                        $req['user_display_name'] ?? 'N/A',
                        $currency_symbol_pdf . number_format($req['amount'], 2),
                        ucfirst($req['status']),
                        !empty($req['display_date']) ? date('Y-m-d', strtotime($req['display_date'])) : 'N/A'
                    ];
                }, $admin_fund_requests)) ?>
            });
            // Table 4: Ledger History
            pdf.startY = pdf.doc.lastAutoTable.finalY + 10;
            if (pdf.startY > pdf.doc.internal.pageSize.height - 100) { pdf.doc.addPage(); pdf.startY = 20; }
            pdf.doc.setFontSize(12);
            pdf.doc.text("Financial Ledger History", 40, pdf.startY);
            pdf.doc.autoTable({
                startY: pdf.startY + 5,
                theme: 'grid',
                head: [['Date', 'Account', 'Description', 'Inflow (Debit)', 'Outflow (Credit)']],
                body: <?= json_encode(array_map(function($entry) use ($currency_symbol_pdf) {
                    return [
                        !empty($entry['date_recorded']) ? date('Y-m-d H:i', strtotime($entry['date_recorded'])) : 'N/A',
                        $entry['account_name'] ?? 'N/A',
                        $entry['description'],
                        (in_array($entry['transaction_type'], ['income', 'transfer_in'])) ? $currency_symbol_pdf . number_format($entry['amount'], 2) : '',
                        (in_array($entry['transaction_type'], ['expense', 'benefit', 'transfer_out'])) ? $currency_symbol_pdf . number_format($entry['amount'], 2) : ''
                    ];
                }, $admin_ledger_history)) ?>,
                columnStyles: { 3: { halign: 'right' }, 4: { halign: 'right' } }
            });
            // Payment Types Summary
            pdf.startY = pdf.doc.lastAutoTable.finalY + 10;
            if (pdf.startY > pdf.doc.internal.pageSize.height - 100) { pdf.doc.addPage(); pdf.startY = 20; }
            pdf.doc.setFontSize(12);
            pdf.doc.text("Payment Types Summary", 40, pdf.startY);
            pdf.doc.autoTable({
                startY: pdf.startY + 5,
                theme: 'striped',
                head: [['Type Name', 'Required', 'Total Paid', 'Total Debt', 'Excess']],
                body: <?= json_encode(array_map(function($row) use ($currency_symbol_pdf) {
                    return [
                        htmlspecialchars($row['type_name']),
                        $currency_symbol_pdf . number_format($row['required_amount'], 2),
                        $currency_symbol_pdf . number_format($row['total_paid'], 2),
                        $currency_symbol_pdf . number_format($row['total_debt'], 2),
                        $currency_symbol_pdf . number_format($row['excess_amount'], 2)
                    ];
                }, $payment_types_summary ?? [])) ?>,
                foot: [[
                    'Total',
                    pdf.currency + '<?= number_format(array_sum(array_column($payment_types_summary ?? [], 'required_amount')), 2) ?>',
                    pdf.currency + '<?= number_format(array_sum(array_column($payment_types_summary ?? [], 'total_paid')), 2) ?>',
                    pdf.currency + '<?= number_format(array_sum(array_column($payment_types_summary ?? [], 'total_debt')), 2) ?>',
                    pdf.currency + '<?= number_format(array_sum(array_column($payment_types_summary ?? [], 'excess_amount')), 2) ?>'
                ]],
                columnStyles: { 1: { halign: 'right' }, 2: { halign: 'right' }, 3: { halign: 'right' }, 4: { halign: 'right' } }
            });
            // Expense Types Summary
            pdf.startY = pdf.doc.lastAutoTable.finalY + 10;
            if (pdf.startY > pdf.doc.internal.pageSize.height - 100) { pdf.doc.addPage(); pdf.startY = 20; }
            pdf.doc.setFontSize(12);
            pdf.doc.text("Expense Types Summary", 40, pdf.startY);
            pdf.doc.autoTable({
                startY: pdf.startY + 5,
                theme: 'striped',
                head: [['Type Name', 'Total']],
                body: <?= json_encode(array_map(function($row) use ($currency_symbol_pdf) {
                    return [ htmlspecialchars($row['type_name']), $currency_symbol_pdf . number_format($row['total'], 2) ];
                }, $expense_types_summary ?? [])) ?>,
                foot: [[ 'Total', pdf.currency + '<?= number_format(array_sum(array_column($expense_types_summary ?? [], 'total')), 2) ?>' ]],
                columnStyles: { 1: { halign: 'right' } }
            });
            // Benefit Types Summary
            pdf.startY = pdf.doc.lastAutoTable.finalY + 10;
            if (pdf.startY > pdf.doc.internal.pageSize.height - 100) { pdf.doc.addPage(); pdf.startY = 20; }
            pdf.doc.setFontSize(12);
            pdf.doc.text("Benefit Types Summary", 40, pdf.startY);
            pdf.doc.autoTable({
                startY: pdf.startY + 5,
                theme: 'striped',
                head: [['Type Name', 'Total']],
                body: <?= json_encode(array_map(function($row) use ($currency_symbol_pdf) {
                    return [ htmlspecialchars($row['type_name']), $currency_symbol_pdf . number_format($row['total'], 2) ];
                }, $benefit_types_summary ?? [])) ?>,
                foot: [[ 'Total', pdf.currency + '<?= number_format(array_sum(array_column($benefit_types_summary ?? [], 'total')), 2) ?>' ]],
                columnStyles: { 1: { halign: 'right' } }
            });
            // Payments Detailed
            pdf.startY = pdf.doc.lastAutoTable.finalY + 10;
            if (pdf.startY > pdf.doc.internal.pageSize.height - 100) { pdf.doc.addPage(); pdf.startY = 20; }
            pdf.doc.setFontSize(12);
            pdf.doc.text("Payments", 40, pdf.startY);
            pdf.doc.autoTable({
                startY: pdf.startY + 5,
                theme: 'grid',
                head: [['ID', 'Full Name', 'Payment Type', 'Amount', 'Status', 'Created At', 'Payment Cycle']],
                body: <?= json_encode(array_map(function($row) use ($currency_symbol_pdf) {
                    return [
                        $row['id'], htmlspecialchars($row['full_name']), htmlspecialchars($row['payment_type']),
                        $currency_symbol_pdf . number_format($row['amount'], 2), htmlspecialchars($row['payment_status']),
                        date('Y-m-d', strtotime($row['created_at'])), htmlspecialchars($row['payment_cycle'] ?? 'N/A')
                    ];
                }, $payments ?? [])) ?>,
                columnStyles: { 3: { halign: 'right' } }
            });
            // Expenses Detailed
            pdf.startY = pdf.doc.lastAutoTable.finalY + 10;
            if (pdf.startY > pdf.doc.internal.pageSize.height - 100) { pdf.doc.addPage(); pdf.startY = 20; }
            pdf.doc.setFontSize(12);
            pdf.doc.text("Expenses", 40, pdf.startY);
            pdf.doc.autoTable({
                startY: pdf.startY + 5,
                theme: 'grid',
                head: [['ID', 'Expense Type', 'Amount', 'Description', 'Created At']],
                body: <?= json_encode(array_map(function($row) use ($currency_symbol_pdf) {
                    return [
                        $row['id'], htmlspecialchars($row['expense_type'] ?? 'N/A'),
                        $currency_symbol_pdf . number_format($row['amount'], 2),
                        htmlspecialchars($row['description']), date('Y-m-d', strtotime($row['created_at']))
                    ];
                }, $expenses ?? [])) ?>,
                columnStyles: { 2: { halign: 'right' } }
            });
            // Benefits Detailed
            pdf.startY = pdf.doc.lastAutoTable.finalY + 10;
            if (pdf.startY > pdf.doc.internal.pageSize.height - 100) { pdf.doc.addPage(); pdf.startY = 20; }
            pdf.doc.setFontSize(12);
            pdf.doc.text("Benefits", 40, pdf.startY);
            pdf.doc.autoTable({
                startY: pdf.startY + 5,
                theme: 'grid',
                head: [['ID', 'Member ID', 'Full Name', 'Benefit Type', 'Amount', 'Created At']],
                body: <?= json_encode(array_map(function($row) use ($currency_symbol_pdf) {
                    return [
                        $row['id'], $row['member_id'], htmlspecialchars($row['full_name']),
                        htmlspecialchars($row['benefit_type']), $currency_symbol_pdf . number_format($row['amount'], 2),
                        date('Y-m-d', strtotime($row['created_at']))
                    ];
                }, $benefits ?? [])) ?>,
                columnStyles: { 4: { halign: 'right' } }
            });
            // Debts Detailed
            pdf.startY = pdf.doc.lastAutoTable.finalY + 10;
            if (pdf.startY > pdf.doc.internal.pageSize.height - 100) { pdf.doc.addPage(); pdf.startY = 20; }
            pdf.doc.setFontSize(12);
            pdf.doc.text("Debts", 40, pdf.startY);
            pdf.doc.autoTable({
                startY: pdf.startY + 5,
                theme: 'grid',
                head: [['Member ID', 'Full Name', 'Payment Type', 'Required', 'Total Paid', 'Debt', 'Payment Cycle', 'Due Date']],
                body: <?= json_encode(array_map(function($row) use ($currency_symbol_pdf) {
                    return [
                        $row['member_id'], htmlspecialchars($row['full_name']), htmlspecialchars($row['payment_type']),
                        $currency_symbol_pdf . number_format($row['required_amount'], 2),
                        $currency_symbol_pdf . number_format($row['total_paid'], 2),
                        $currency_symbol_pdf . number_format($row['debt'], 2),
                        htmlspecialchars($row['payment_cycle'] ?? 'N/A'), htmlspecialchars($row['due_date'] ?? 'N/A')
                    ];
                }, $debts ?? [])) ?>,
                columnStyles: { 3: { halign: 'right' }, 4: { halign: 'right' }, 5: { halign: 'right' } }
            });
            // --- 4. Save the PDF ---
            pdf.doc.save(`Financial_Report_FULL_${pdf.orgName.replace(/\s+/g, '_')}_${pdf.toDate}.pdf`);
        } catch (error) {
            console.error("Error generating PDF:", error);
            alert("An error occurred while generating the PDF. Please check the console for details.");
        }
    });

    // --- NEW: Separate Payments Summary PDF ---
    $('#exportpaymentsSummaryPdf').on('click', function() {
        try {
            const pdf = getPdfBaseData();
            pdf.doc.setFontSize(14);
            pdf.doc.text(`Payment Summary Report - ${pdf.orgName}`, 40, 20);
            pdf.doc.setFontSize(10);
            pdf.doc.text(`Period: ${pdf.fromDate} to ${pdf.toDate}`, 40, 30);
            
            pdf.doc.setFontSize(12);
            pdf.doc.text("Payment Types Summary", 40, pdf.startY + 10);
            pdf.doc.autoTable({
                startY: pdf.startY + 15,
                theme: 'striped',
                head: [['Type Name', 'Required', 'Total Paid', 'Total Debt', 'Excess']],
                body: <?= json_encode(array_map(function($row) use ($currency_symbol_pdf) {
                    return [
                        htmlspecialchars($row['type_name']),
                        $currency_symbol_pdf . number_format($row['required_amount'], 2),
                        $currency_symbol_pdf . number_format($row['total_paid'], 2),
                        $currency_symbol_pdf . number_format($row['total_debt'], 2),
                        $currency_symbol_pdf . number_format($row['excess_amount'], 2)
                    ];
                }, $payment_types_summary ?? [])) ?>,
                foot: [[
                    'Total',
                    pdf.currency + '<?= number_format(array_sum(array_column($payment_types_summary ?? [], 'required_amount')), 2) ?>',
                    pdf.currency + '<?= number_format(array_sum(array_column($payment_types_summary ?? [], 'total_paid')), 2) ?>',
                    pdf.currency + '<?= number_format(array_sum(array_column($payment_types_summary ?? [], 'total_debt')), 2) ?>',
                    pdf.currency + '<?= number_format(array_sum(array_column($payment_types_summary ?? [], 'excess_amount')), 2) ?>'
                ]],
                columnStyles: { 1: { halign: 'right' }, 2: { halign: 'right' }, 3: { halign: 'right' }, 4: { halign: 'right' } }
            });

            pdf.startY = pdf.doc.lastAutoTable.finalY + 20;
            pdf.doc.setFontSize(12);
            pdf.doc.text("Detailed Payments Log", 40, pdf.startY);
             pdf.doc.autoTable({
                startY: pdf.startY + 5,
                theme: 'grid',
                head: [['ID', 'Full Name', 'Payment Type', 'Amount', 'Status', 'Created At', 'Payment Cycle']],
                body: <?= json_encode(array_map(function($row) use ($currency_symbol_pdf) {
                    return [
                        $row['id'], htmlspecialchars($row['full_name']), htmlspecialchars($row['payment_type']),
                        $currency_symbol_pdf . number_format($row['amount'], 2), htmlspecialchars($row['payment_status']),
                        date('Y-m-d', strtotime($row['created_at'])), htmlspecialchars($row['payment_cycle'] ?? 'N/A')
                    ];
                }, $payments ?? [])) ?>,
                columnStyles: { 3: { halign: 'right' } }
            });

            pdf.doc.save(`Payment_Report_${pdf.orgName.replace(/\s+/g, '_')}_${pdf.toDate}.pdf`);
        } catch (e) { console.error("Error generating Payments PDF:", e); alert("Failed to generate Payments PDF."); }
    });

    // --- NEW: Separate Expenses Summary PDF ---
    $('#exportexpensesSummaryPdf').on('click', function() {
        try {
            const pdf = getPdfBaseData();
            pdf.doc.setFontSize(14);
            pdf.doc.text(`Expense Summary Report - ${pdf.orgName}`, 40, 20);
            pdf.doc.setFontSize(10);
            pdf.doc.text(`Period: ${pdf.fromDate} to ${pdf.toDate}`, 40, 30);
            
            pdf.doc.setFontSize(12);
            pdf.doc.text("Expense Types Summary", 40, pdf.startY + 10);
            pdf.doc.autoTable({
                startY: pdf.startY + 15,
                theme: 'striped',
                head: [['Type Name', 'Total']],
                body: <?= json_encode(array_map(function($row) use ($currency_symbol_pdf) {
                    return [ htmlspecialchars($row['type_name']), $currency_symbol_pdf . number_format($row['total'], 2) ];
                }, $expense_types_summary ?? [])) ?>,
                foot: [[ 'Total', pdf.currency + '<?= number_format(array_sum(array_column($expense_types_summary ?? [], 'total')), 2) ?>' ]],
                columnStyles: { 1: { halign: 'right' } }
            });

            pdf.startY = pdf.doc.lastAutoTable.finalY + 20;
            pdf.doc.setFontSize(12);
            pdf.doc.text("Detailed Expenses Log", 40, pdf.startY);
            pdf.doc.autoTable({
                startY: pdf.startY + 5,
                theme: 'grid',
                head: [['ID', 'Expense Type', 'Amount', 'Description', 'Created At']],
                body: <?= json_encode(array_map(function($row) use ($currency_symbol_pdf) {
                    return [
                        $row['id'], htmlspecialchars($row['expense_type'] ?? 'N/A'),
                        $currency_symbol_pdf . number_format($row['amount'], 2),
                        htmlspecialchars($row['description']), date('Y-m-d', strtotime($row['created_at']))
                    ];
                }, $expenses ?? [])) ?>,
                columnStyles: { 2: { halign: 'right' } }
            });

            pdf.doc.save(`Expense_Report_${pdf.orgName.replace(/\s+/g, '_')}_${pdf.toDate}.pdf`);
        } catch (e) { console.error("Error generating Expenses PDF:", e); alert("Failed to generate Expenses PDF."); }
    });

    // --- NEW: Separate Benefits Summary PDF ---
    $('#exportbenefitsSummaryPdf').on('click', function() {
        try {
            const pdf = getPdfBaseData();
            pdf.doc.setFontSize(14);
            pdf.doc.text(`Benefit Summary Report - ${pdf.orgName}`, 40, 20);
            pdf.doc.setFontSize(10);
            pdf.doc.text(`Period: ${pdf.fromDate} to ${pdf.toDate}`, 40, 30);
            
            pdf.doc.setFontSize(12);
            pdf.doc.text("Benefit Types Summary", 40, pdf.startY + 10);
            pdf.doc.autoTable({
                startY: pdf.startY + 15,
                theme: 'striped',
                head: [['Type Name', 'Total']],
                body: <?= json_encode(array_map(function($row) use ($currency_symbol_pdf) {
                    return [ htmlspecialchars($row['type_name']), $currency_symbol_pdf . number_format($row['total'], 2) ];
                }, $benefit_types_summary ?? [])) ?>,
                foot: [[ 'Total', pdf.currency + '<?= number_format(array_sum(array_column($benefit_types_summary ?? [], 'total')), 2) ?>' ]],
                columnStyles: { 1: { halign: 'right' } }
            });

            pdf.startY = pdf.doc.lastAutoTable.finalY + 20;
            pdf.doc.setFontSize(12);
            pdf.doc.text("Detailed Benefits Log", 40, pdf.startY);
            pdf.doc.autoTable({
                startY: pdf.startY + 5,
                theme: 'grid',
                head: [['ID', 'Member ID', 'Full Name', 'Benefit Type', 'Amount', 'Created At']],
                body: <?= json_encode(array_map(function($row) use ($currency_symbol_pdf) {
                    return [
                        $row['id'], $row['member_id'], htmlspecialchars($row['full_name']),
                        htmlspecialchars($row['benefit_type']), $currency_symbol_pdf . number_format($row['amount'], 2),
                        date('Y-m-d', strtotime($row['created_at']))
                    ];
                }, $benefits ?? [])) ?>,
                columnStyles: { 4: { halign: 'right' } }
            });

            pdf.doc.save(`Benefit_Report_${pdf.orgName.replace(/\s+/g, '_')}_${pdf.toDate}.pdf`);
        } catch (e) { console.error("Error generating Benefits PDF:", e); alert("Failed to generate Benefits PDF."); }
    });


    <?php endif; ?>
    // ==================================================================
    // !! END OF ADMIN SCRIPT !!
    // ==================================================================

    // --- Transaction Report Filter JS ---
    // (This part is outside the admin check)
    
    // Check for collapse state on load
    // --- FIX: Corrected typo from search_params to searchParams ---
    if (new URL(location.href).searchParams.get('search') ||
        new URL(location.href).searchParams.get('time_period') ||
        new URL(location.href).searchParams.get('type_filter_payments')) {
        if(document.getElementById('moreFilters')) { // Add check
            document.getElementById('moreFilters').classList.add('show');
        }
    }

    // --- FIX: Moved Quick Date Filter Logic INSIDE the main $(document).ready() ---
    const fromInput = document.getElementById('admin_from_date');
    const toInput = document.getElementById('admin_to_date');

    const btnThisMonth = document.getElementById('btnThisMonth');
    const btnLastMonth = document.getElementById('btnLastMonth');
    const btnThisQuarter = document.getElementById('btnThisQuarter');
    const btnThisYear = document.getElementById('btnThisYear');

    // Return early if elements don't exist
    if (fromInput && toInput && btnThisMonth && btnLastMonth && btnThisQuarter && btnThisYear) {
        
        const today = new Date();
        const year = today.getFullYear();
        const month = today.getMonth(); // 0-11

        // Helper function to format date as YYYY-MM-DD
        function formatDate(date) {
            const y = date.getFullYear();
            const m = (date.getMonth() + 1).toString().padStart(2, '0');
            const d = date.getDate().toString().padStart(2, '0');
            return `${y}-${m}-${d}`;
        }

        btnThisMonth.addEventListener('click', () => {
            const start = new Date(year, month, 1);
            const end = new Date(year, month + 1, 0); // 0th day of next month
            fromInput.value = formatDate(start);
            toInput.value = formatDate(end);
        });

        btnLastMonth.addEventListener('click', () => {
            const start = new Date(year, month - 1, 1);
            const end = new Date(year, month, 0); // 0th day of this month
            fromInput.value = formatDate(start);
            toInput.value = formatDate(end);
        });

        btnThisQuarter.addEventListener('click', () => {
            const quarter = Math.floor(month / 3); // 0, 1, 2, 3
            const startMonth = quarter * 3; // 0, 3, 6, 9
            const start = new Date(year, startMonth, 1);
            const end = new Date(year, startMonth + 3, 0); // 0th day of 3 months later
            fromInput.value = formatDate(start);
            toInput.value = formatDate(end);
        });

        btnThisYear.addEventListener('click', () => {
            fromInput.value = `${year}-01-01`;
            toInput.value = `${year}-12-31`;
        });
    }
    // --- END: Quick Date Filter Logic ---

}); // --- END OF $(document).ready() ---

// --- Global helper functions ---
function toggleDatePicker(){
    if(document.getElementById('datePickerContainer')) { // Add check
        document.getElementById('datePickerContainer').style.display =
            document.querySelector('[name="time_period"]').value==='range' ? 'block':'none';
    }
}
function updatePerPage(table,val){
    const u=new URL(location);
    u.searchParams.set('per_page_'+table,val);
    u.searchParams.set('page_'+table,'1');
    location.href = u.toString();
}
function printDashboard(){
    // This function now only prints the main report area
    window.print();
}

</script>
<?php require_once 'includes/footer.php'; ?>