<?php
require_once 'includes/functions.php';
require_once 'includes/db_config.php';

initialize_session(); // Start session for messages and CSRF

// Generate CSRF token
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Display session messages
if (isset($_SESSION['success_message'])) {
    $success_message = $_SESSION['success_message'];
    unset($_SESSION['success_message']);
} else {
    $success_message = '';
}
if (isset($_SESSION['error_message'])) {
    $error_message = $_SESSION['error_message'];
    unset($_SESSION['error_message']);
} else {
    $error_message = '';
}
$system_name = get_setting('system_name', $link) ?? 'Club Portal';

// Preserve tab from GET or default
$active_tab = isset($_GET['tab']) ? sanitize_input($_GET['tab']) : 'roles';

// Handle POST requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['csrf_token']) && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
    
    $redirect_params = [];
    $active_tab_post = 'roles'; // Default, will be overwritten by a successful action

    // Create Role
    if (isset($_POST['create_role'])) {
        $active_tab_post = 'roles';
        $name = sanitize_input($_POST['role_name']);
        $description = sanitize_input($_POST['role_description']);
        if (create_role($link, $name, $description)) {
            $_SESSION['success_message'] = "Role created successfully: " . htmlspecialchars($name);
        } else {
            $_SESSION['error_message'] = "Failed to create role. Name may already exist, be restricted, or database error: " . mysqli_error($link);
            error_log("Role creation failed: $name - " . mysqli_error($link));
        }
    }
    // Update Role
    if (isset($_POST['update_role'])) {
        $active_tab_post = 'roles';
        $old_name = sanitize_input($_POST['old_role_name']);
        $new_name = sanitize_input($_POST['role_name']);
        $description = sanitize_input($_POST['role_description']);
        if (update_role($link, $old_name, $new_name, $description)) {
            $_SESSION['success_message'] = "Role updated successfully: " . htmlspecialchars($new_name);
        } else {
            $_SESSION['error_message'] = "Failed to update role: " . mysqli_error($link);
            error_log("Role update failed: $old_name to $new_name - " . mysqli_error($link));
            $redirect_params['edit_role'] = $old_name; // Preserve for re-edit on failure
        }
    }
    // Delete Role
    if (isset($_POST['delete_role'])) {
        $active_tab_post = 'roles';
        $name = sanitize_input($_POST['role_name']);
        if (delete_role($link, $name)) {
            $_SESSION['success_message'] = "Role deleted successfully: " . htmlspecialchars($name);
        } else {
            $_SESSION['error_message'] = "Failed to delete role. Restricted, not found, or database error: " . mysqli_error($link);
            error_log("Role deletion failed: $name - " . mysqli_error($link));
        }
    }
    // Create Position
    if (isset($_POST['create_position'])) {
        $active_tab_post = 'positions';
        $name = sanitize_input($_POST['position_name']);
        $description = sanitize_input($_POST['position_description']);
        $new_id = create_position($link, $name, $description);
        if ($new_id) {
            $_SESSION['success_message'] = "Position created successfully: " . htmlspecialchars($name) . " (ID: $new_id)";
        } else {
            $_SESSION['error_message'] = "Failed to create position: " . mysqli_error($link);
            error_log("Position creation failed: $name - " . mysqli_error($link));
        }
    }
    // Update Position
    if (isset($_POST['update_position'])) {
        $active_tab_post = 'positions';
        $id = intval($_POST['position_id']);
        $name = sanitize_input($_POST['position_name']);
        $description = sanitize_input($_POST['position_description']);
        if (update_position($link, $id, $name, $description)) {
            $_SESSION['success_message'] = "Position updated successfully: " . htmlspecialchars($name) . " (ID: $id)";
        } else {
            $_SESSION['error_message'] = "Failed to update position: " . mysqli_error($link);
            error_log("Position update failed: ID $id - " . mysqli_error($link));
            $redirect_params['edit_position'] = $id; // Preserve for re-edit on failure
        }
    }
    // Delete Position
    if (isset($_POST['delete_position'])) {
        $active_tab_post = 'positions';
        $id = intval($_POST['position_id']);
        if (delete_position($link, $id)) {
            $_SESSION['success_message'] = "Position deleted successfully (ID: $id)";
        } else {
            $_SESSION['error_message'] = "Failed to delete position: " . mysqli_error($link);
            error_log("Position deletion failed: ID $id - " . mysqli_error($link));
        }
    }
    // Assign Roles to Position
    if (isset($_POST['assign_roles'])) {
        $active_tab_post = 'assignments';
        $position_id = intval($_POST['position_id']);
        $assigned_roles = $_POST['assigned_roles'] ?? [];
        if (assign_roles_to_position($link, $position_id, $assigned_roles)) {
            $_SESSION['success_message'] = "Roles assigned successfully to position (ID: $position_id).";
        } else {
            $_SESSION['error_message'] = "Failed to assign roles: " . mysqli_error($link);
            error_log("Role assignment failed: Position ID $position_id - " . mysqli_error($link));
        }
        $redirect_params['assign_position'] = $position_id; // Always keep this position selected
    }
    // Update Permissions
    if (isset($_POST['update_permissions'])) {
        $active_tab_post = 'permissions';
        $type = sanitize_input($_POST['type']);
        $id = sanitize_input($_POST['id']);
        $selected_permissions = $_POST['permissions'] ?? [];

        if ($type === 'role') {
            if (strtolower($id) === 'super_admin' && $_SESSION['user_role'] !== 'super_admin') {
                $_SESSION['error_message'] = "You do not have permission to edit super admin permissions.";
                error_log("Attempt to edit super_admin by non-super_admin: User ID {$_SESSION['user_id']}");
            } else {
                mysqli_begin_transaction($link);
                try {
                    $sql = "DELETE FROM role_permissions WHERE role_name = ?";
                    $stmt = mysqli_prepare($link, $sql);
                    mysqli_stmt_bind_param($stmt, "s", $id);
                    if (!mysqli_stmt_execute($stmt)) {
                        throw new Exception("Delete failed: " . mysqli_stmt_error($stmt));
                    }
                    mysqli_stmt_close($stmt);
                    foreach ($selected_permissions as $permission) {
                        $sql = "INSERT INTO role_permissions (role_name, permission_key) VALUES (?, ?)";
                        $stmt = mysqli_prepare($link, $sql);
                        mysqli_stmt_bind_param($stmt, "ss", $id, $permission);
                        if (!mysqli_stmt_execute($stmt)) {
                            throw new Exception("Insert failed for $permission: " . mysqli_stmt_error($stmt));
                        }
                        mysqli_stmt_close($stmt);
                    }
                    mysqli_commit($link);
                    $_SESSION['success_message'] = "Permissions saved successfully for role: " . htmlspecialchars(ucfirst(str_replace('_', ' ', $id))) . ".";
                } catch (Exception $e) {
                    mysqli_rollback($link);
                    $_SESSION['error_message'] = "Failed to save permissions: " . $e->getMessage() . " - DB error: " . mysqli_error($link);
                    error_log("Permissions update failed for role $id: " . $e->getMessage());
                }
            }
        } elseif ($type === 'position') {
            $position_id = intval($id);
            $check_sql = "SELECT id FROM positions WHERE id = ?";
            $stmt_check = mysqli_prepare($link, $check_sql);
            mysqli_stmt_bind_param($stmt_check, "i", $position_id);
            mysqli_stmt_execute($stmt_check);
            mysqli_stmt_store_result($stmt_check);
            if (mysqli_stmt_num_rows($stmt_check) === 0) {
                $_SESSION['error_message'] = "Invalid position selected (ID: $position_id not found).";
            } else {
                mysqli_begin_transaction($link);
                try {
                    $sql = "DELETE FROM position_permissions WHERE position_id = ?";
                    $stmt = mysqli_prepare($link, $sql);
                    mysqli_stmt_bind_param($stmt, "i", $position_id);
                    if (!mysqli_stmt_execute($stmt)) {
                        throw new Exception("Delete failed: " . mysqli_stmt_error($stmt));
                    }
                    mysqli_stmt_close($stmt);
                    foreach ($selected_permissions as $permission) {
                        $sql = "INSERT INTO position_permissions (position_id, permission_key) VALUES (?, ?)";
                        $stmt = mysqli_prepare($link, $sql);
                        mysqli_stmt_bind_param($stmt, "is", $position_id, $permission);
                        if (!mysqli_stmt_execute($stmt)) {
                            throw new Exception("Insert failed for $permission: " . mysqli_stmt_error($stmt));
                        }
                        mysqli_stmt_close($stmt);
                    }
                    mysqli_commit($link);
                    $_SESSION['success_message'] = "Permissions saved successfully for position (ID: $position_id).";
                } catch (Exception $e) {
                    mysqli_rollback($link);
                    $_SESSION['error_message'] = "Failed to save permissions: " . $e->getMessage() . " - DB error: " . mysqli_error($link);
                    error_log("Permissions update failed for position $position_id: " . $e->getMessage());
                }
            }
            mysqli_stmt_close($stmt_check);
        }
        // Always keep this selection active after redirect
        $redirect_params['type'] = $type;
        $redirect_params['id'] = $id;
    }
    
    // Regenerate CSRF token
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    
    // *** THIS IS THE FIXED LINE ***
    // It now correctly merges only the parameters you've set ($redirect_params) with the active tab.
    $redirect_query = http_build_query(array_merge($redirect_params, ['tab' => $active_tab_post]));
    header("Location: role_permissions.php?$redirect_query");
    exit;
}

// Fetch all permissions
$permissions = [];
$sql = "SELECT permission_key, description FROM permissions ORDER BY description";
if ($result = mysqli_query($link, $sql)) {
    while ($row = mysqli_fetch_assoc($result)) {
        $permissions[] = $row;
    }
    mysqli_free_result($result);
} else {
    $error_message = "Failed to fetch permissions: " . mysqli_error($link);
}

// Fetch data
$roles = get_all_roles($link);
$positions = get_all_positions($link);

// For editing: If editing role or position, prefill
$edit_role = isset($_GET['edit_role']) ? sanitize_input($_GET['edit_role']) : null;
$edit_position = isset($_GET['edit_position']) ? intval($_GET['edit_position']) : null;
$assign_position = isset($_GET['assign_position']) ? intval($_GET['assign_position']) : null;

// Selected for permissions
$type = isset($_GET['type']) ? sanitize_input($_GET['type']) : 'role';
$id = isset($_GET['id']) ? sanitize_input($_GET['id']) : (empty($roles) ? '' : $roles[0]['name']);

// Ensure 'id' is valid for the 'type'
if ($type === 'position' && !is_numeric($id) && !empty($positions)) {
    $id = $positions[0]['id']; // Default to first position if 'id' is not numeric (e.g., a role name)
} elseif ($type === 'role' && is_numeric($id) && !empty($roles)) {
    $id = $roles[0]['name']; // Default to first role if 'id' is numeric
}

$current_permissions = ($type === 'role') ? get_role_permissions($link, $id) : get_position_permissions($link, intval($id));
?>
<?php require_once 'includes/header.php'; ?>
<script>
document.title = 'Role & Permission Settings - <?php echo htmlspecialchars($system_name); ?>';
</script>
<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Role & Permission Settings</h2>
            <?php if ($success_message): ?>
                <div class="alert alert-success"><?php echo htmlspecialchars($success_message); ?></div>
            <?php endif; ?>
            <?php if ($error_message): ?>
                <div class="alert alert-danger"><?php echo htmlspecialchars($error_message); ?></div>
            <?php endif; ?>
            <div class="card mb-4">
                <div class="card-header">
                    Manage Roles, Positions, Assignments & Permissions
                    <a href="add_admin.php" class="btn btn-success btn-sm float-end"><i class="fas fa-plus-circle me-1"></i> Add Admin User</a>
                </div>
                <div class="card-body">
                    <ul class="nav nav-tabs" id="rbacTabs" role="tablist">
                        <li class="nav-item" role="presentation">
                            <button class="nav-link <?php echo $active_tab === 'roles' ? 'active' : ''; ?>" id="roles-tab" data-bs-toggle="tab" data-bs-target="#roles" type="button" role="tab">Manage Roles</button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link <?php echo $active_tab === 'positions' ? 'active' : ''; ?>" id="positions-tab" data-bs-toggle="tab" data-bs-target="#positions" type="button" role="tab">Manage Positions</button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link <?php echo $active_tab === 'assignments' ? 'active' : ''; ?>" id="assignments-tab" data-bs-toggle="tab" data-bs-target="#assignments" type="button" role="tab">Manage Assignments</button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link <?php echo $active_tab === 'permissions' ? 'active' : ''; ?>" id="permissions-tab" data-bs-toggle="tab" data-bs-target="#permissions" type="button" role="tab">Set Permissions</button>
                        </li>
                    </ul>
                    <div class="tab-content mt-3">
                        <div class="tab-pane fade <?php echo $active_tab === 'roles' ? 'show active' : ''; ?>" id="roles" role="tabpanel">
                            <form method="post" action="role_permissions.php">
                                <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">
                                <?php if ($edit_role): ?>
                                    <input type="hidden" name="old_role_name" value="<?php echo htmlspecialchars($edit_role); ?>">
                                    <input type="hidden" name="update_role" value="1">
                                <?php else: ?>
                                    <input type="hidden" name="create_role" value="1">
                                <?php endif; ?>
                                <div class="mb-3">
                                    <label for="role_name">Role Name</label>
                                    <input type="text" name="role_name" id="role_name" class="form-control" value="<?php echo htmlspecialchars($edit_role ?? ''); ?>" required>
                                </div>
                                <div class="mb-3">
                                    <label for="role_description">Description</label>
                                    <textarea name="role_description" id="role_description" class="form-control"><?php if ($edit_role) { foreach ($roles as $r) { if ($r['name'] === $edit_role) echo htmlspecialchars($r['description']); } } ?></textarea>
                                </div>
                                <button type="submit" class="btn btn-primary"><?php echo $edit_role ? 'Update' : 'Create'; ?> Role</button>
                            </form>
                            <hr>
                            <table class="table table-striped">
                                <thead>
                                    <tr>
                                        <th>Name</th>
                                        <th>Description</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($roles as $role): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars(ucfirst(str_replace('_', ' ', $role['name']))); ?></td>
                                            <td><?php echo htmlspecialchars($role['description'] ?? ''); ?></td>
                                            <td>
                                                <a href="role_permissions.php?edit_role=<?php echo urlencode($role['name']); ?>&tab=roles" class="btn btn-sm btn-warning">Edit</a>
                                                <?php if (strtolower($role['name']) !== 'super_admin'): ?>
                                                    <form method="post" action="role_permissions.php" style="display:inline;">
                                                        <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">
                                                        <input type="hidden" name="role_name" value="<?php echo htmlspecialchars($role['name']); ?>">
                                                        <input type="hidden" name="delete_role" value="1">
                                                        <button type="submit" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure?');">Delete</button>
                                                    </form>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        <div class="tab-pane fade <?php echo $active_tab === 'positions' ? 'show active' : ''; ?>" id="positions" role="tabpanel">
                            <form method="post" action="role_permissions.php">
                                <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">
                                <?php if ($edit_position): ?>
                                    <input type="hidden" name="position_id" value="<?php echo $edit_position; ?>">
                                    <input type="hidden" name="update_position" value="1">
                                <?php else: ?>
                                    <input type="hidden" name="create_position" value="1">
                                <?php endif; ?>
                                <div class="mb-3">
                                    <label for="position_name">Position Name</label>
                                    <input type="text" name="position_name" id="position_name" class="form-control" value="<?php if ($edit_position) { foreach ($positions as $p) { if ($p['id'] == $edit_position) echo htmlspecialchars($p['name']); } } ?>" required>
                                </div>
                                <div class="mb-3">
                                    <label for="position_description">Description</label>
                                    <textarea name="position_description" id="position_description" class="form-control"><?php if ($edit_position) { foreach ($positions as $p) { if ($p['id'] == $edit_position) echo htmlspecialchars($p['description']); } } ?></textarea>
                                </div>
                                <button type="submit" class="btn btn-primary"><?php echo $edit_position ? 'Update' : 'Create'; ?> Position</button>
                            </form>
                            <hr>
                            <table class="table table-striped">
                                <thead>
                                    <tr>
                                        <th>Name</th>
                                        <th>Description</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($positions as $pos): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars(ucfirst(str_replace('_', ' ', $pos['name']))); ?></td>
                                            <td><?php echo htmlspecialchars($pos['description'] ?? ''); ?></td>
                                            <td>
                                                <a href="role_permissions.php?edit_position=<?php echo $pos['id']; ?>&tab=positions" class="btn btn-sm btn-warning">Edit</a>
                                                <form method="post" action="role_permissions.php" style="display:inline;">
                                                    <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">
                                                    <input type="hidden" name="position_id" value="<?php echo $pos['id']; ?>">
                                                    <input type="hidden" name="delete_position" value="1">
                                                    <button type="submit" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure?');">Delete</button>
                                                </form>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        <div class="tab-pane fade <?php echo $active_tab === 'assignments' ? 'show active' : ''; ?>" id="assignments" role="tabpanel">
                            <form method="get" action="role_permissions.php">
                                <input type="hidden" name="tab" value="assignments">
                                <div class="mb-3">
                                    <label for="assign_position">Select Position</label>
                                    <select name="assign_position" id="assign_position" class="form-control" onchange="this.form.submit()">
                                        <option value="">-- Select --</option>
                                        <?php foreach ($positions as $pos): ?>
                                            <option value="<?php echo $pos['id']; ?>" <?php echo ($assign_position == $pos['id']) ? 'selected' : ''; ?>><?php echo htmlspecialchars(ucfirst(str_replace('_', ' ', $pos['name']))); ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                            </form>
                            <?php if ($assign_position): ?>
                                <form method="post" action="role_permissions.php">
                                    <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">
                                    <input type="hidden" name="position_id" value="<?php echo $assign_position; ?>">
                                    <input type="hidden" name="assign_roles" value="1">
                                    <div class="mb-3">
                                        <label>Assign Roles</label>
                                        <?php $assigned = get_assigned_roles($link, $assign_position); ?>
                                        <?php foreach ($roles as $role): ?>
                                            <?php if (strtolower($role['name']) === 'super_admin' && $_SESSION['user_role'] !== 'super_admin') continue; ?>
                                            <div class="form-check">
                                                <input type="checkbox" name="assigned_roles[]" value="<?php echo htmlspecialchars($role['name']); ?>" class="form-check-input" <?php echo in_array($role['name'], $assigned) ? 'checked' : ''; ?>>
                                                <label class="form-check-label"><?php echo htmlspecialchars(ucfirst(str_replace('_', ' ', $role['name']))); ?></label>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>
                                    <button type="submit" class="btn btn-primary">Save Assignments</button>
                                </form>
                            <?php endif; ?>
                        </div>
                        <div class="tab-pane fade <?php echo $active_tab === 'permissions' ? 'show active' : ''; ?>" id="permissions" role="tabpanel">
                            <form method="get" action="role_permissions.php">
                                <input type="hidden" name="tab" value="permissions">
                                <div class="form-group mb-3">
                                    <label for="type">Select Type</label>
                                    <select name="type" id="type" class="form-control" onchange="this.form.submit()">
                                        <option value="role" <?php echo $type === 'role' ? 'selected' : ''; ?>>Role</option>
                                        <option value="position" <?php echo $type === 'position' ? 'selected' : ''; ?>>Position</option>
                                    </select>
                                </div>
                                <div class="form-group mb-3">
                                    <label for="id"><?php echo ($type === 'role') ? 'Select Role' : 'Select Position'; ?></label>
                                    <?php if ($type === 'role'): ?>
                                        <select name="id" id="id" class="form-control" onchange="this.form.submit()">
                                            <?php foreach ($roles as $role): ?>
                                                <option value="<?php echo htmlspecialchars($role['name']); ?>" <?php echo ($id === $role['name']) ? 'selected' : ''; ?>><?php echo htmlspecialchars(ucfirst(str_replace('_', ' ', $role['name']))); ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    <?php else: ?>
                                        <select name="id" id="id" class="form-control" onchange="this.form.submit()">
                                            <?php if (empty($positions)): ?>
                                                <option value="">-- No positions created --</option>
                                            <?php else: ?>
                                                <?php foreach ($positions as $pos): ?>
                                                    <option value="<?php echo $pos['id']; ?>" <?php echo ($id == $pos['id']) ? 'selected' : ''; ?>><?php echo htmlspecialchars(ucfirst(str_replace('_', ' ', $pos['name']))); ?></option>
                                                <?php endforeach; ?>
                                            <?php endif; ?>
                                        </select>
                                    <?php endif; ?>
                                </div>
                            </form>
                            <hr>
                            <?php if ($id): ?>
                                <h4>Permissions for '<?php echo htmlspecialchars(($type === 'role') ? ucfirst(str_replace('_', ' ', $id)) : get_position_name($link, intval($id))); ?>'</h4>
                                <form method="post" action="role_permissions.php">
                                    <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">
                                    <input type="hidden" name="type" value="<?php echo htmlspecialchars($type); ?>">
                                    <input type="hidden" name="id" value="<?php echo htmlspecialchars($id); ?>">
                                    <input type="hidden" name="update_permissions" value="1">
                                    <div class="form-group mb-3">
                                        <div class="row">
                                            <?php foreach ($permissions as $perm): ?>
                                                <div class="col-md-4">
                                                    <div class="form-check">
                                                        <input type="checkbox" name="permissions[]" value="<?php echo htmlspecialchars($perm['permission_key']); ?>" class="form-check-input" id="perm-<?php echo htmlspecialchars($perm['permission_key']); ?>" <?php echo in_array($perm['permission_key'], $current_permissions) ? 'checked' : ''; ?>>
                                                        <label class="form-check-label" for="perm-<?php echo htmlspecialchars($perm['permission_key']); ?>"><?php echo htmlspecialchars($perm['description']); ?></label>
                                                    </div>
                                                </div>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                    <button type="submit" class="btn btn-primary">Save Permissions</button>
                                </form>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>
</div>
<?php require_once 'includes/footer.php'; ?>