<?php
// record_donations.php
ob_start();
session_start();
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
if (function_exists('opcache_reset')) { opcache_reset(); }
require_once 'includes/functions.php';
require_once 'includes/db_config.php';
/* ---------- SESSION / PERMISSIONS ---------- */
$timeout = 1800;
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity']) > $timeout) {
    session_unset(); session_destroy();
    redirect("login.php?timeout=1");
}
$_SESSION['last_activity'] = time();
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
if (!is_logged_in() || !user_has_permission($link, 'view_payments') ||
    !in_array($_SESSION['user_role'], ['admin', 'accountant', 'super_admin'])) { // Added super_admin
    redirect("login.php"); exit;
}
/* ---------- SETTINGS ---------- */
$currency_symbol = get_setting('currency_symbol', $link) ?? '₦';
$token_cost = (int)get_setting('tokens_per_donation', $link) ?: 1;  // Use specific setting for donations

// --- NEW: Get User Role and ID ---
$current_user_id = $_SESSION['user_id'];
$user_role = $_SESSION['user_role']; 

/* ---------- ORGANIZATION BALANCE FUNCTION (ALL ADMINS + ACCOUNTANTS) ---------- */
/**
 * Retrieves the total active token balance across all users with 'admin' or 'accountant' roles.
 * @param mysqli $link The database connection link.
 * @return int The total organization token balance.
 */
function get_organization_token_balance($link) {
    $sql = "SELECT COALESCE(SUM(tb.balance), 0) AS total
            FROM token_balances tb
            JOIN users u ON tb.user_id = u.id
            WHERE u.role IN ('admin', 'accountant', 'super_admin')
              AND u.deleted_at IS NULL
              AND tb.balance > 0";
    $result = mysqli_query($link, $sql);
    if ($result && $row = mysqli_fetch_assoc($result)) {
        mysqli_free_result($result);
        return (int)$row['total'];
    }
    return 0;
}
/* ---------- INITIAL BALANCES ---------- */
$current_user_balance = get_user_token_balance($link, $current_user_id);
$organization_total_tokens = get_organization_token_balance($link);

/* ---------- FETCH BANK ACCOUNTS (FUNDING SOURCES) - SECURED ---------- */
$bank_accounts = [];
// --- NEW: Secure query based on user role ---
$sql_banks = "SELECT id, account_name, bank_name, account_number, is_cash_on_hand, current_balance, owner_user_id 
              FROM bank_accounts 
              WHERE is_active = 1 AND deleted_at IS NULL"; // Also check deleted_at

$params = [];
$types = "";

// --- *** SECURITY FIX V2 (Corrected) *** ---
// If user is NOT an admin, restrict to ONLY public accounts (bank or cash).
// Hide private imprest accounts and the Cheque Holding account.
if (!in_array($user_role, ['admin', 'super_admin'])) {
    $sql_banks .= " AND owner_user_id IS NULL 
                  AND account_name != 'Cheques Payable (Holding)'";
}
// --- *** END FIX *** ---

$sql_banks .= " ORDER BY is_cash_on_hand DESC, bank_name ASC";

$stmt_banks = mysqli_prepare($link, $sql_banks);
if ($stmt_banks) {
    if (!empty($params)) {
        mysqli_stmt_bind_param($stmt_banks, $types, ...$params);
    }
    mysqli_stmt_execute($stmt_banks);
    $result_banks = mysqli_stmt_get_result($stmt_banks);
    
    while ($row = mysqli_fetch_assoc($result_banks)) {
        // Use 'bank_name' if 'account_name' is missing, fallback to 'Cash at Hand'
        $account_display = $row['account_name'] ?? $row['bank_name'];
        // Create a user-friendly display name for the dropdown
        $display_balance = $currency_symbol . number_format($row['current_balance'], 2);
        if ($row['is_cash_on_hand']) {
            $row['display_name'] = htmlspecialchars($account_display) . " (Cash/Imprest) - {$display_balance}";
        } else {
            $row['display_name'] = htmlspecialchars($account_display . ' (' . $row['bank_name'] . ') - ') . $display_balance;
        }
        $bank_accounts[] = $row;
    }
    mysqli_stmt_close($stmt_banks);
}
// --- END: Secure query ---

/* ---------- FORM HANDLING ---------- */
$message = $message_type = '';
if ($_SERVER["REQUEST_METHOD"] == "POST"
    && isset($_POST['log_donation']) // This hidden field name is fine to keep
    && isset($_POST['csrf_token'])
    && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
    
    $amount = floatval($_POST['amount'] ?? 0);
    $payment_type = sanitize_input($_POST['payment_type'] ?? '');  // Get from form
    $description = sanitize_input($_POST['description'] ?? '');
    
    // NEW: Handle member ID vs. donor name
    $member_id = intval($_POST['member_id'] ?? 0);
    $member_id_to_db = ($member_id > 0) ? $member_id : null;
    
    $donor_name = sanitize_input($_POST['donor_name'] ?? '');
    $donor_email = sanitize_input($_POST['donor_email'] ?? '');
    
    // If a member is selected, clear the donor name/email to avoid confusion
    if ($member_id_to_db !== null) {
        $donor_name = '';
        $donor_email = '';
    }
    
    $status = 'Paid';
   
    // Get the bank account ID where the income will be deposited
    $bank_account_id = intval($_POST['bank_account_id'] ?? 0);
    // Required tokens for logging the transaction (single for donation)
    $required_tokens = $token_cost;
    // Check 1: User has enough tokens to perform the transaction
    if ($current_user_balance < $required_tokens) {
        $message = "Insufficient tokens. You need $required_tokens token".($required_tokens>1?'s':'')
                 .", but you have $current_user_balance. <a href='token_store.php'>Buy / request more</a>.";
        $message_type = 'danger';
    } else {
        
        // --- NEW: Validate that the user is allowed to deposit into this account ---
        $is_allowed_account = false;
        if ($bank_account_id > 0) {
            // We check against the SECURE list of $bank_accounts fetched earlier
            foreach ($bank_accounts as $allowed_acc) {
                if ($allowed_acc['id'] == $bank_account_id) {
                    $is_allowed_account = true;
                    break;
                }
            }
        }
        if (!$is_allowed_account) {
            $message = "Invalid 'Deposit To' account selection. You may not have permission to deposit into this account.";
            $message_type = 'danger';
        }
        // --- END: Account validation ---
        
        // --- UPDATE: Check 2: Validate payment type is donation OR other_income category ---
        $valid_type = false;
        if ($message_type !== 'danger') { // Only check if passes account validation
            $sql_check_type = "SELECT 1 FROM payment_types 
                               WHERE type_name = ? 
                               AND payment_category IN ('donation', 'other_income') 
                               AND deleted_at IS NULL";
            $stmt_check_type = mysqli_prepare($link, $sql_check_type);
            mysqli_stmt_bind_param($stmt_check_type, "s", $payment_type);
            mysqli_stmt_execute($stmt_check_type);
            mysqli_stmt_store_result($stmt_check_type);
            if (mysqli_stmt_num_rows($stmt_check_type) > 0) {
                $valid_type = true;
            }
            mysqli_stmt_close($stmt_check_type);
        }
       
        // Check 3: Final validation before transaction begins
        if ($message_type !== 'danger' && $amount > 0 && $valid_type && $bank_account_id > 0) {
           
            mysqli_begin_transaction($link);
            $ok = true;
            $tokens_used = 0;
            // 1. Deduct tokens from the current user
            // We use 'donation' as the token reason for both
            $tokens_used = deduct_tokens($link, $current_user_id, 'donation', $required_tokens); 
            if ($tokens_used === false) {
                $ok = false;
                $message = "Token deduction failed.";
                $message_type = 'danger';
            }
            // 2. Update Bank Account Balance (One time for the total income)
            if ($ok) {
                $sql_bank = "UPDATE bank_accounts SET current_balance = current_balance + ? WHERE id = ?";
                $stmt_bank = mysqli_prepare($link, $sql_bank);
                if ($stmt_bank) {
                    mysqli_stmt_bind_param($stmt_bank, "di", $amount, $bank_account_id);
                    if (!mysqli_stmt_execute($stmt_bank)) {
                        $ok = false;
                        $message = "Failed to update bank balance: " . mysqli_stmt_error($stmt_bank);
                        $message_type = 'danger';
                    }
                    mysqli_stmt_close($stmt_bank);
                } else {
                    $ok = false;
                    $message = "Failed to prepare bank update statement.";
                    $message_type = 'danger';
                }
            }
            // 3. Prepare SQL for Payments and Ledger
            $stmt_pay = $stmt_ledger = null; // Init vars
            if ($ok) {
                // Payment insertion statement (member_id is now included)
                // We set is_donation = 1 for both, as they are non-obligation payments
                $sql_pay = "INSERT INTO payments
                            (payment_type, amount, description, payment_status, bank_account_id, member_id, is_donation, donor_name, donor_email, payment_date, created_at)
                            VALUES (?, ?, ?, ?, ?, ?, 1, ?, ?, NOW(), NOW())";
                $stmt_pay = mysqli_prepare($link, $sql_pay);
                // Bind params: s(type), d(amt), s(desc), s(status), i(bank_id), i(member_id), s(donor_name), s(donor_email)
                mysqli_stmt_bind_param($stmt_pay, "sdssiiss", 
                    $payment_type, $amount, $description, $status, $bank_account_id, $member_id_to_db, $donor_name, $donor_email);
               
                if (!mysqli_stmt_execute($stmt_pay)) {
                    $ok = false;
                    $message = "Payment insert error: " . mysqli_stmt_error($stmt_pay);
                    $message_type = 'danger';
                } else {
                    $payment_id = mysqli_insert_id($link); // Get the ID of the payment we just made
                   
                    // --- UPDATE: Ledger insertion statement ---
                    $ledger_desc = "Income: '$payment_type' (Ref Pay ID: $payment_id)"; // Changed "Donation" to "Income"
                    if ($member_id_to_db) {
                        $ledger_desc .= " (Member ID: $member_id_to_db)";
                    } elseif ($donor_name) {
                        $ledger_desc .= " from $donor_name";
                    }

                    $sql_ledger = "INSERT INTO financial_ledger
                                   (bank_account_id, transaction_type, amount, description, reference_id, transaction_date, recorded_by_id)
                                   VALUES (?, 'income', ?, ?, ?, NOW(), ?)";
                    $stmt_ledger = mysqli_prepare($link, $sql_ledger);
                    mysqli_stmt_bind_param($stmt_ledger, "idsii",
                        $bank_account_id, $amount, $ledger_desc, $payment_id, $current_user_id);
                    if (!mysqli_stmt_execute($stmt_ledger)) {
                        $ok = false;
                        $message = "Ledger insert error: " . mysqli_stmt_error($stmt_ledger);
                        $message_type = 'danger';
                    }
                }
               
                // Close statements
                if ($stmt_pay) mysqli_stmt_close($stmt_pay);
                if ($stmt_ledger) mysqli_stmt_close($stmt_ledger);
            }
            // 4. Commit or Rollback
            if ($ok) {
                mysqli_commit($link);
                // --- UPDATE: Success message ---
                $message = "Income logged! Used $tokens_used token".($tokens_used>1?'s':'').".";
                $message_type = 'success';
                /* ---- REFRESH BALANCES ---- */
                $current_user_balance = get_user_token_balance($link, $current_user_id);
                $organization_total_tokens = get_organization_token_balance($link);
            } else {
                mysqli_rollback($link);
                // Ensure a fallback error message if a specific one wasn't set
                if (!$message) $message = "Transaction rolled back due to an unknown error.";
                $message_type = 'danger';
            }
        } else {
            // More descriptive error
            $error_reason = "";
            if ($bank_account_id <= 0) $error_reason = "a funding source was not selected.";
            elseif (!$is_allowed_account) $error_reason = "you do not have permission to deposit into that account."; // New check
            elseif (!$valid_type) $error_reason = "the payment type is invalid (not Donation or Other Income).";
            elseif ($amount <= 0) $error_reason = "the amount is zero.";
            else $error_reason = "an unexpected validation error occurred.";
           
            $message = "Invalid data – check amount or type. Reason: $error_reason";
            $message_type = 'danger';
        }
    }
    // Refresh CSRF token on POST
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
} elseif ($_SERVER["REQUEST_METHOD"] == "POST") {
    $message = "Invalid CSRF token.";
    $message_type = 'danger';
}

// --- UPDATE: PAYMENT TYPES LIST FOR DROPDOWN (DONATION & OTHER INCOME) ---
$payment_types = [];
$res = mysqli_query($link,
    "SELECT type_name FROM payment_types
     WHERE payment_category IN ('donation', 'other_income') AND deleted_at IS NULL
     ORDER BY type_name");
if ($res) {
    while ($row = mysqli_fetch_assoc($res)) $payment_types[] = $row;
    mysqli_free_result($res);
}

/* ---------- NEW: MEMBERS LIST FOR DROPDOWN ---------- */
$members = [];
$res_members = mysqli_query($link,
    "SELECT id,full_name,member_id FROM users
     WHERE role='member' AND (deleted_at IS NULL OR deleted_at>NOW())
     ORDER BY full_name");
if ($res_members) {
    while ($row = mysqli_fetch_assoc($res_members)) $members[] = $row;
    mysqli_free_result($res_members);
}

// Fetch current currency symbol for display purposes
$currency_symbol = get_setting('currency_symbol', $link) ?? '₦';
?>
<?php require_once 'includes/header.php'; ?>
<script>document.title='Record Income (Donations, etc.) - <?=htmlspecialchars(get_setting('system_name',$link)??'Club Portal')?>';</script>
<style>.main-content{padding:20px;}</style>
<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Record Income (Donations, Other Income, etc.)</h2>
            <?php if ($message): ?>
                <div class="alert alert-<?=$message_type?>" role="alert"><?=$message?></div>
            <?php endif; ?>
            <div class="card mb-4 shadow-sm">
                <div class="card-header bg-primary text-white">
                    <i class="fas fa-money-bill-wave"></i> Log New Income / Donation
                </div>
                <div class="card-body">
                    <div class="d-flex justify-content-between mb-3 border-bottom pb-2">
                        <p class="mb-0"><strong>Your Token Balance:</strong> <span class="badge bg-success"><?=$current_user_balance?> tokens</span></p>
                        <p class="mb-0"><strong>Org Token Balance:</strong> <span class="badge bg-info text-dark"><?=$organization_total_tokens?> tokens</span></p>
                    </div>
                    <form action="record_donations.php" method="post">
                        <input type="hidden" name="log_donation" value="1"> 
                        <input type="hidden" name="csrf_token" value="<?=htmlspecialchars($_SESSION['csrf_token'])?>">
                        
                        <div class="mb-3">
                            <label class="form-label fw-bold text-success">Deposit To (Funding Source)</label>
                            <select name="bank_account_id" id="bank_account_id" class="form-select" required>
                                <option value="">Select Account (Cash or Bank)</option>
                                <?php foreach ($bank_accounts as $account): ?>
                                    <option value="<?= htmlspecialchars($account['id']) ?>">
                                        <?= htmlspecialchars($account['display_name']) ?>
                                    </option>
                                <?php endforeach; ?>
                                <?php if (empty($bank_accounts)): ?>
                                    <option value="" disabled>No public deposit accounts configured. Please contact Admin.</option>
                                <?php endif; ?>
                            </select>
                            <small class="form-text text-muted">Select the public bank or cash account where this income is being deposited.</small>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label fw-bold">Income Type</label>
                            <select name="payment_type" id="payment_type" class="form-select" required>
                                <option value="">Select Type</option>
                                <?php foreach ($payment_types as $pt): ?>
                                    <option value="<?=htmlspecialchars($pt['type_name'])?>">
                                        <?=htmlspecialchars($pt['type_name'])?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <small class="form-text text-muted">Select an income or donation payment type.</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label fw-bold">Amount (<?=$currency_symbol?>)</label>
                            <input type="number" name="amount" id="amount_donation" class="form-control" step="0.01" min="0.01" required>
                            <small class="form-text text-muted">Enter the income amount.</small>
                        </div>

                        <hr>
                        <p class="text-center text-muted">Select a member OR enter non-member details below.</p>

                        <div class="mb-3">
                            <label class="form-label fw-bold">Select Member (Optional)</label>
                            <select name="member_id" id="member_id" class="form-select" onchange="toggleDonorFields()">
                                <option value="0">-- Select a Registered Member --</option>
                                <?php foreach ($members as $m): ?>
                                    <option value="<?=htmlspecialchars($m['id'])?>">
                                        <?=htmlspecialchars($m['full_name'].' (ID: '.$m['member_id'].')')?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <small class="form-text text-muted">If this income is from a registered member, select them here.</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Non-Member Donor/Source Name (optional)</label>
                            <input type="text" name="donor_name" id="donor_name" class="form-control">
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Non-Member Donor/Source Email (optional)</label>
                            <input type="email" name="donor_email" id="donor_email" class="form-control">
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Description (optional)</label>
                            <textarea name="description" id="description_donation" class="form-control" rows="2"></textarea>
                        </div>
                        <button type="submit" class="btn btn-primary w-100 mt-3">
                            <i class="fas fa-plus-circle"></i> Log Income / Donation
                        </button>
                    </form>
                </div>
            </div>
        </main>
    </div>
</div>

<script>
/**
 * Disables the Donor Name and Email fields if a registered member is selected.
 */
function toggleDonorFields() {
    const memberSelect = document.getElementById('member_id');
    const donorName = document.getElementById('donor_name');
    const donorEmail = document.getElementById('donor_email');
    
    // If a member is selected (value is not 0), disable the text fields
    if (memberSelect.value !== '0') {
        donorName.disabled = true;
        donorEmail.disabled = true;
        donorName.value = ''; // Clear the fields
        donorEmail.value = '';
    } else {
        // Otherwise, enable them
        donorName.disabled = false;
        donorEmail.disabled = false;
    }
}

// Run on page load to set the initial state
document.addEventListener('DOMContentLoaded', toggleDonorFields);
</script>

<?php require_once 'includes/footer.php'; ?>