<?php
// member_id_cards.php - FINAL: Reduced Font + Clean Spacing
require_once 'includes/functions.php';
require_once 'includes/db_config.php';
if (!is_logged_in()) { redirect("login.php"); exit(); }

$is_admin    = is_admin();
$system_name = get_setting('system_name', $link) ?? 'Club Portal';
$user_id     = $_SESSION['user_id'] ?? 0;

$temp_dir   = 'temp';
$upload_dir = 'uploads/profile_pictures';
if (!is_dir($temp_dir)) mkdir($temp_dir, 0777, true);
if (!is_dir($upload_dir)) mkdir($upload_dir, 0755, true);

// Ensure default placeholder exists
$default_path = $upload_dir . '/default.png';
if (!file_exists($default_path)) {
    $img = imagecreatetruecolor(200, 200);
    $bg = imagecolorallocate($img, 255, 255, 255);
    $text = imagecolorallocate($img, 200, 200, 200);
    imagefilledrectangle($img, 0, 0, 200, 200, $bg);
    imagestring($img, 5, 50, 90, 'NO PHOTO', $text);
    imagepng($img, $default_path);
    imagedestroy($img);
}

/* --------------------------------------------------
   Font Paths (Your Structure)
-------------------------------------------------- */
$font_dir = 'fonts/dejavu-fonts-ttf-2.37/ttf';
$font_regular = "$font_dir/DejaVuSans.ttf";
$font_bold    = "$font_dir/DejaVuSans-Bold.ttf";

$use_ttf = file_exists($font_regular) && file_exists($font_bold);

/* --------------------------------------------------
   Helper: Bold Built-in Font (Fallback)
-------------------------------------------------- */
function imagestring_bold(&$img, $font, $x, $y, $str, $col) {
    imagestring($img, $font, $x, $y, $str, $col);
    imagestring($img, $font, $x + 1, $y, $str, $col);
}

/* --------------------------------------------------
   Extract Extra Fields (Same as fetch_member_details.php)
-------------------------------------------------- */
function extractExtraFields($form_data_json, $form_structure_json) {
    $extra = ['dob' => 'N/A', 'phone' => 'N/A', 'marital_status' => 'N/A', 'address' => 'N/A'];
    $form_data = $form_data_json ? json_decode($form_data_json, true) : [];
    $form_structure = $form_structure_json ? json_decode($form_structure_json, true) : [];

    if (empty($form_data)) return $extra;

    $label_map = [];
    $structure_field_names = [];

    // Handle both flat fields and steps
    $fields = [];
    if (isset($form_structure['steps']) && is_array($form_structure['steps'])) {
        foreach ($form_structure['steps'] as $step) {
            if (!empty($step['fields'])) $fields = array_merge($fields, $step['fields']);
        }
    } elseif (isset($form_structure['fields']) && is_array($form_structure['fields'])) {
        $fields = $form_structure['fields'];
    }

    foreach ($fields as $field) {
        $name = $field['name'] ?? null;
        $label = $field['label'] ?? $name;
        if ($name) {
            $label_map[$name] = $label;
            $structure_field_names[] = $name;
        }
    }

    $data_field_names = array_keys($form_data);
    $all_field_names = array_unique(array_merge($structure_field_names, $data_field_names));

    $mapped = [];
    foreach ($all_field_names as $name) {
        $label = $label_map[$name] ?? ucfirst(str_replace(['_', '-'], ' ', $name));
        $value = $form_data[$name] ?? '';
        if (is_array($value)) $value = implode(', ', $value);
        $mapped[$name] = ['label' => $label, 'value' => (string)$value];
    }

    $targets = [
        'dob'            => ['date of birth', 'dob', 'birth date', 'birthday', 'birth'],
        'phone'          => ['phone', 'mobile', 'contact', 'cell', 'telephone'],
        'marital_status' => ['marital status', 'civil status', 'marital', 'status'],
        'address'        => ['address', 'street', 'home address', 'residence', 'full address']
    ];

    foreach ($targets as $key => $keywords) {
        foreach ($mapped as $name => $field) {
            $label_lower = strtolower($field['label']);
            $name_lower = strtolower($name);
            foreach ($keywords as $kw) {
                if (strpos($label_lower, $kw) !== false || strpos($name_lower, $kw) !== false) {
                    $extra[$key] = $field['value'] ?: 'N/A';
                    break 2;
                }
            }
        }
    }

    return $extra;
}

/* --------------------------------------------------
   Generate HD PNG ID Card (REDUCED FONT + SPACING)
-------------------------------------------------- */
function generateFullIdCardPng($system_name, $member, $status, $profile_data, $qr_data, $extra, $outputFile) {
    global $use_ttf, $font_regular, $font_bold;

    $w = 1008; $h = 630; $pad = 40;
    $img = imagecreatetruecolor($w, $h);

    $white = imagecolorallocate($img, 255, 255, 255);
    $black = imagecolorallocate($img, 0, 0, 0);
    $green = imagecolorallocate($img, 40, 167, 69);
    $red   = imagecolorallocate($img, 220, 53, 69);
    $grey  = imagecolorallocate($img, 108, 117, 125);
    imagefilledrectangle($img, 0, 0, $w, $h, $white);

    // === TTF Font Sizes (Reduced) ===
    $font_size_title = 24;   // Was 28
    $font_size_label = 16;   // Was 18
    $font_size_value = 15;   // Was 16
    $line_height = 32;       // Slightly reduced for balance

    // === Header ===
    $header_h = 80;
    imagefilledrectangle($img, 0, 0, $w, $header_h, $green);
    $title = strtoupper($system_name) . " - IDENTIFICATION";

    if ($use_ttf) {
        $bbox = imagettfbbox($font_size_title, 0, $font_bold, $title);
        $text_w = $bbox[4] - $bbox[0];
        $x = ($w - $text_w) / 2;
        $y = ($header_h - $font_size_title) / 2 + $font_size_title;
        imagettftext($img, $font_size_title, 0, $x, $y, $white, $font_bold, $title);
    } else {
        $font = 5;
        $title_w = imagefontwidth($font) * strlen($title);
        imagestring_bold($img, $font, ($w - $title_w) / 2, ($header_h - imagefontheight($font)) / 2, $title, $white);
    }

    // === Left: Photo ===
    $photo_size = 240;
    $x1 = $pad; $y1 = $header_h + $pad;
    $photo = @imagecreatefromstring($profile_data);
    if ($photo) {
        $resized = imagescale($photo, $photo_size, $photo_size);
        imagecopy($img, $resized, $x1, $y1, 0, 0, $photo_size, $photo_size);
        imagerectangle($img, $x1-1, $y1-1, $x1+$photo_size, $y1+$photo_size, $grey);
        imagedestroy($photo); imagedestroy($resized);
    } else {
        $placeholder = imagecreate($photo_size, $photo_size);
        $bg = imagecolorallocate($placeholder, 240, 240, 240);
        $text = imagecolorallocate($placeholder, 180, 180, 180);
        imagefilledrectangle($placeholder, 0, 0, $photo_size, $photo_size, $bg);
        if ($use_ttf) {
            imagettftext($placeholder, 20, 0, 30, 120, $text, $font_bold, 'NO PHOTO');
        } else {
            imagestring($placeholder, 5, 50, 100, 'NO PHOTO', $text);
        }
        imagecopy($img, $placeholder, $x1, $y1, 0, 0, $photo_size, $photo_size);
        imagedestroy($placeholder);
    }

    $y_status = $y1 + $photo_size + 30;
    $y_id = $y_status + $line_height;

    if ($use_ttf) {
        imagettftext($img, $font_size_label, 0, $x1, $y_status, $black, $font_bold, "STATUS:");
        imagettftext($img, $font_size_label, 0, $x1 + 110, $y_status, $status === 'Active' ? $green : $red, $font_bold, $status);

        imagettftext($img, $font_size_label, 0, $x1, $y_id, $black, $font_bold, "MEMBER ID:");
        imagettftext($img, $font_size_value, 0, $x1 + 150, $y_id, $grey, $font_regular, $member['member_id']);
    } else {
        imagestring_bold($img, 5, $x1, $y_status - 15, "STATUS:", $black);
        imagestring_bold($img, 5, $x1 + 110, $y_status - 15, $status, $status === 'Active' ? $green : $red);
        imagestring_bold($img, 5, $x1, $y_id - 15, "MEMBER ID:", $black);
        imagestring($img, 5, $x1 + 150, $y_id - 15, $member['member_id'], $grey);
    }

    // === Right: Details ===
    $x2 = $x1 + $photo_size + 80;
    $y2 = $header_h + $pad;

    // Name
    $name = strtoupper($member['full_name']);
    if ($use_ttf) {
        $font_size_name = $font_size_title;
        $bbox = imagettfbbox($font_size_name, 0, $font_bold, $name);
        $text_w = $bbox[4] - $bbox[0];
        if ($text_w > 420) {
            $font_size_name = 20;
            do {
                $font_size_name--;
                $bbox = imagettfbbox($font_size_name, 0, $font_bold, $name);
                $text_w = $bbox[4] - $bbox[0];
            } while ($text_w > 420 && $font_size_name > 16);
        }
        imagettftext($img, $font_size_name, 0, $x2, $y2 + $font_size_name, $black, $font_bold, $name);
        $y2 += $font_size_name + 18;
    } else {
        $name_lines = explode("\n", wordwrap($name, 35, "\n", true));
        foreach ($name_lines as $l) { 
            imagestring_bold($img, 5, $x2, $y2 - 15, $l, $black); 
            $y2 += 25; 
        }
        $y2 += 18;
    }

    // Details: Label + Value with padding & alignment
    $details = [
        'DOB'            => $extra['dob'],
        'PHONE'          => $extra['phone'],
        'MARITAL STATUS' => $extra['marital_status'],
        'EXPIRY'         => $member['expiry_date'] ? date('M d, Y', strtotime($member['expiry_date'])) : 'N/A'
    ];

    foreach ($details as $label => $val) {
        $val_display = ($val === 'N/A') ? 'N/A' : substr($val, 0, 35);

        if ($use_ttf) {
            // Label
            $label_bbox = imagettfbbox($font_size_label, 0, $font_bold, "$label:");
            $label_w = $label_bbox[4] - $label_bbox[0];
            imagettftext($img, $font_size_label, 0, $x2, $y2 + $font_size_label, $green, $font_bold, "$label:");

            // Value (aligned under label end + padding)
            $value_x = $x2 + $label_w + 15; // 15px padding
            imagettftext($img, $font_size_value, 0, $value_x, $y2 + $font_size_label, $black, $font_regular, $val_display);
        } else {
            imagestring_bold($img, 5, $x2, $y2 - 15, "$label:", $green);
            imagestring($img, 5, $x2 + 130, $y2 - 15, $val_display, $black);
        }
        $y2 += $line_height;
    }

    // Address (multi-line, padded)
    $address = $extra['address'];
    if ($use_ttf) {
        imagettftext($img, $font_size_label, 0, $x2, $y2 + $font_size_label, $green, $font_bold, "ADDRESS:");
        $y2 += $line_height;
        $addr_lines = ($address === 'N/A') ? ['N/A'] : explode("\n", wordwrap($address, 55, "\n", true));
        foreach ($addr_lines as $line) {
            imagettftext($img, $font_size_value, 0, $x2, $y2 + $font_size_value, $black, $font_regular, $line);
            $y2 += $line_height;
        }
    } else {
        imagestring_bold($img, 5, $x2, $y2 - 15, "ADDRESS:", $green);
        $y2 += 25;
        foreach (explode("\n", wordwrap($address, 45, "\n", true)) as $line) {
            imagestring($img, 5, $x2, $y2 - 15, $line, $black);
            $y2 += 25;
        }
    }

    // === QR Code ===
    $qr = @imagecreatefromstring($qr_data);
    if ($qr) {
        $qr_size = 180;
        $resized_qr = imagescale($qr, $qr_size, $qr_size);
        imagecopy($img, $resized_qr, $w - $pad - $qr_size, $h - $pad - $qr_size - 10, 0, 0, $qr_size, $qr_size);
        imagedestroy($qr); imagedestroy($resized_qr);
    }

    // === Footer ===
    if ($use_ttf) {
        imagettftext($img, 12, 0, $pad, $h - 20, $grey, $font_regular, "Scan QR code to verify authenticity.");
    } else {
        imagestring($img, 3, $pad, $h - 30, "Scan QR code to verify authenticity.", $grey);
    }

    imagepng($img, $outputFile);
    imagedestroy($img);
}

/* --------------------------------------------------
   QR Code Helper
-------------------------------------------------- */
require_once 'includes/phpqrcode/qrlib.php';
function getQRCodePngData($url) {
    ob_start();
    QRcode::png($url, null, QR_ECLEVEL_L, 8);
    return ob_get_clean();
}

/* --------------------------------------------------
   Bulk & Single Download
-------------------------------------------------- */
if ($is_admin && isset($_GET['download_all_png'])) {
    $sql = "SELECT u.id, u.full_name, u.member_id, u.profile_picture,
                   u.membership_status, u.expiry_date,
                   fd.data AS form_json, f.form_data AS form_structure
            FROM users u
            LEFT JOIN form_data fd ON u.id = fd.user_id
            LEFT JOIN forms f ON fd.form_id = f.id
            WHERE u.role IN ('member', 'pending') 
              AND (u.deleted_at IS NULL OR u.deleted_at > NOW())";
    $res = mysqli_query($link, $sql);
    $members = mysqli_fetch_all($res, MYSQLI_ASSOC);

    $zip = new ZipArchive();
    $zipFile = "$temp_dir/all_id_cards_" . date('Y-m-d_His') . ".zip";
    if ($zip->open($zipFile, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== TRUE) die("Cannot create ZIP.");

    foreach ($members as $m) {
        $extra = extractExtraFields($m['form_json'], $m['form_structure']);
        $status = (strtolower($m['membership_status']) === 'approved' && (!$m['expiry_date'] || strtotime($m['expiry_date']) > time())) ? 'Active' : 'Expired';

        $base_url = "https://{$_SERVER['HTTP_HOST']}" . dirname($_SERVER['SCRIPT_NAME']);
        $qr_url = rtrim($base_url, '/') . "/verify_id.php?id=" . urlencode($m['member_id']);
        $qr_data = getQRCodePngData($qr_url);

        $profile_path = $m['profile_picture'] && file_exists($m['profile_picture']) 
            ? $m['profile_picture'] 
            : 'uploads/profile_pictures/default.png';
        $profile_data = file_get_contents($profile_path);

        $png_file = "$temp_dir/card_{$m['id']}.png";
        generateFullIdCardPng($system_name, $m, $status, $profile_data, $qr_data, $extra, $png_file);

        if (file_exists($png_file)) {
            $zip->addFile($png_file, "id_card_{$m['member_id']}.png");
        }
    }

    $zip->close();
    header('Content-Type: application/zip');
    header('Content-Disposition: attachment; filename="all_id_cards_' . date('Y-m-d') . '.zip"');
    header('Content-Length: ' . filesize($zipFile));
    readfile($zipFile);
    @unlink($zipFile);
    array_map('unlink', glob("$temp_dir/card_*.png"));
    exit();
}

if (isset($_GET['download_single_png']) && isset($_GET['id'])) {
    $id = intval($_GET['id']);
    if (!$is_admin && $id !== $user_id) die("Permission denied.");

    $sql = "SELECT u.id, u.full_name, u.member_id, u.profile_picture,
                   u.membership_status, u.expiry_date,
                   fd.data AS form_json, f.form_data AS form_structure
            FROM users u
            LEFT JOIN form_data fd ON u.id = fd.user_id
            LEFT JOIN forms f ON fd.form_id = f.id
            WHERE u.id = ? AND u.role IN ('member', 'pending')";
    $stmt = mysqli_prepare($link, $sql);
    mysqli_stmt_bind_param($stmt, 'i', $id);
    mysqli_stmt_execute($stmt);
    $m = mysqli_fetch_assoc(mysqli_stmt_get_result($stmt));
    mysqli_stmt_close($stmt);

    if (!$m) die("Member not found.");

    $extra = extractExtraFields($m['form_json'], $m['form_structure']);
    $status = (strtolower($m['membership_status']) === 'approved' && (!$m['expiry_date'] || strtotime($m['expiry_date']) > time())) ? 'Active' : 'Expired';

    $base_url = "https://{$_SERVER['HTTP_HOST']}" . dirname($_SERVER['SCRIPT_NAME']);
    $qr_url = rtrim($base_url, '/') . "/verify_id.php?id=" . urlencode($m['member_id']);
    $qr_data = getQRCodePngData($qr_url);

    $profile_path = $m['profile_picture'] && file_exists($m['profile_picture']) 
        ? $m['profile_picture'] 
        : 'uploads/profile_pictures/default.png';
    $profile_data = file_get_contents($profile_path);

    $png_file = "$temp_dir/card_{$m['id']}_" . time() . ".png";
    generateFullIdCardPng($system_name, $m, $status, $profile_data, $qr_data, $extra, $png_file);

    if (file_exists($png_file)) {
        header('Content-Type: image/png');
        header('Content-Disposition: attachment; filename="id_card_' . $m['member_id'] . '.png"');
        header('Content-Length: ' . filesize($png_file));
        readfile($png_file);
        @unlink($png_file);
        exit();
    }
    die("Error generating ID card.");
}

/* --------------------------------------------------
   Display Members
-------------------------------------------------- */
$members = [];
if ($is_admin) {
    $sql = "SELECT u.id, u.full_name, u.member_id, u.profile_picture, 
                   u.membership_status, u.expiry_date,
                   fd.data AS form_json, f.form_data AS form_structure
            FROM users u
            LEFT JOIN form_data fd ON u.id = fd.user_id
            LEFT JOIN forms f ON fd.form_id = f.id
            WHERE u.role IN ('member', 'pending') 
              AND (u.deleted_at IS NULL OR u.deleted_at > NOW())
            ORDER BY u.full_name";
    $res = mysqli_query($link, $sql);
    while ($r = mysqli_fetch_assoc($res)) {
        $extra = extractExtraFields($r['form_json'], $r['form_structure']);
        $r['extra'] = $extra;
        $members[] = $r;
    }
} else {
    $stmt = mysqli_prepare($link, "SELECT u.id, u.full_name, u.member_id, u.profile_picture, 
                                        u.membership_status, u.expiry_date,
                                        fd.data AS form_json, f.form_data AS form_structure
                                 FROM users u
                                 LEFT JOIN form_data fd ON u.id = fd.user_id
                                 LEFT JOIN forms f ON fd.form_id = f.id
                                 WHERE u.id = ? AND u.role IN ('member', 'pending')");
    mysqli_stmt_bind_param($stmt, 'i', $user_id);
    mysqli_stmt_execute($stmt);
    $res = mysqli_stmt_get_result($stmt);
    if ($r = mysqli_fetch_assoc($res)) {
        $extra = extractExtraFields($r['form_json'], $r['form_structure']);
        $r['extra'] = $extra;
        $members[] = $r;
    }
    mysqli_stmt_close($stmt);
}

require_once 'includes/header.php'; ?>
<script>document.title = 'Digital ID Cards - <?= htmlspecialchars($system_name) ?>';</script>
<style>
.badge-status{padding:4px 8px;border-radius:15px;font-weight:bold;font-size:13px;}
.active{background:#d4edda;color:#155724;}
.expired{background:#f8d7da;color:#721c24;}
.extra-info{font-size:0.9em;color:#555;line-height:1.4;}
</style>

<div class="container-fluid">
  <div class="row">
    <?php require_once 'includes/sidebar.php'; ?>
    <main class="col-md-9 col-lg-10">
      <h2>Digital ID Cards</h2>
      <?php if ($is_admin && !empty($members)): ?>
        <a href="?download_all_png=1" class="btn btn-success mb-3">Download All as HD PNG (ZIP)</a>
      <?php endif; ?>

      <div class="row">
        <?php foreach ($members as $m):
          $status = (strtolower($m['membership_status']) === 'approved' && (!$m['expiry_date'] || strtotime($m['expiry_date']) > time())) ? 'Active' : 'Expired';
          $base_url = "https://{$_SERVER['HTTP_HOST']}" . dirname($_SERVER['SCRIPT_NAME']);
          $qr_url = rtrim($base_url, '/') . "/verify_id.php?id=" . urlencode($m['member_id']);
          $extra = $m['extra'];
        ?>
          <div class="col-md-6 col-lg-4 mb-4">
            <div class="card h-100">
              <div class="card-body text-center">
                <img src="<?= htmlspecialchars($m['profile_picture'] ?: 'uploads/profile_pictures/default.png') ?>" 
                     class="rounded-circle mb-2" width="120" height="120" style="object-fit:cover;">
                <h5 class="mt-2"><?= htmlspecialchars($m['full_name']) ?></h5>
                <span class="badge-status <?= $status==='Active'?'active':'expired' ?>"><?= $status ?></span>
                <p class="mb-1 text-muted">ID: <?= htmlspecialchars($m['member_id']) ?></p>
                <?php if ($m['expiry_date']): ?>
                  <p class="text-muted mb-1">Expiry: <?= date('M d, Y', strtotime($m['expiry_date'])) ?></p>
                <?php endif; ?>

                <div class="extra-info text-left mt-3">
                  <small>
                    <strong>DOB:</strong> <?= htmlspecialchars($extra['dob']) ?><br>
                    <strong>Phone:</strong> <?= htmlspecialchars($extra['phone']) ?><br>
                    <strong>Marital:</strong> <?= htmlspecialchars($extra['marital_status']) ?><br>
                    <strong>Address:</strong> <?= htmlspecialchars(substr($extra['address'], 0, 40)) ?><?= strlen($extra['address']) > 40 ? '...' : '' ?>
                  </small>
                </div>

                <img src="data:image/png;base64,<?= base64_encode(getQRCodePngData($qr_url)) ?>" 
                     alt="QR Code" class="mt-3" width="120" height="120">

                <div class="mt-3">
                  <a href="?download_single_png=1&id=<?= $m['id'] ?>" 
                     class="btn btn-outline-success btn-sm">Download HD PNG</a>
                </div>
              </div>
            </div>
          </div>
        <?php endforeach; ?>
      </div>
    </main>
  </div>
</div>

<?php require_once 'includes/footer.php'; ?>