<?php
// list.php (now in root)
require_once 'includes/functions.php';
require_once 'includes/db_config.php';

if (!is_logged_in() || !is_admin()) {
    redirect("login.php");
}

/* ---------- FILTERS & PAGINATION ---------- */
$search      = $_GET['search'] ?? '';
$page        = max(1, (int)($_GET['page_members'] ?? 1));
$per_page    = max(5, (int)($_GET['per_page_members'] ?? 20));
$offset      = ($page - 1) * $per_page;

$where  = "WHERE u.role IN ('member','pending') AND (u.deleted_at IS NULL OR u.deleted_at > NOW())";
$params = [];
$types  = '';

if ($search !== '') {
    $like = "%$search%";
    $where   .= " AND (u.member_id LIKE ? OR u.full_name LIKE ? OR u.email LIKE ?)";
    $params  = [$like, $like, $like];
    $types   = 'sss';
}

/* ---------- TOTAL COUNT ---------- */
$count_sql = "SELECT COUNT(*) FROM users u $where";
$stmt = $link->prepare($count_sql);
if ($types) $stmt->bind_param($types, ...$params);
$stmt->execute();
$total_rows = $stmt->get_result()->fetch_row()[0] ?? 0;
$total_pages = max(1, ceil($total_rows / $per_page));
$stmt->close();

/* ---------- FETCH MEMBERS ---------- */
$sql = "SELECT 
            u.id, u.member_id, u.full_name, u.email, u.sex, u.phone,
            u.membership_status, u.membership_expiry, u.profile_image, u.position
        FROM users u $where
        ORDER BY u.member_id DESC
        LIMIT ? OFFSET ?";

$stmt = $link->prepare($sql);
$bind_types = $types . 'ii';
$params[] = $per_page;
$params[] = $offset;
$stmt->bind_param($bind_types, ...$params);
$stmt->execute();
$result = $stmt->get_result();
$members = $result->fetch_all(MYSQLI_ASSOC);
$stmt->close();
?>

<?php require_once 'includes/header.php'; ?>

<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Members</h2><hr>

            <!-- Action Buttons -->
            <div class="mb-3">
                <a href="executives.php" class="btn btn-info">Executives</a>
                <a href="?export=pdf&search=<?=urlencode($search)?>" class="btn btn-danger">PDF</a>
                <a href="?export=csv&search=<?=urlencode($search)?>" class="btn btn-success">CSV</a>
                <button class="btn btn-outline-secondary ms-2" onclick="printTable('membersTable')">
                    Print
                </button>
            </div>

            <!-- Search Form -->
            <form method="GET" class="mb-4 filter-section">
                <div class="row g-3">
                    <div class="col-md-6">
                        <label class="form-label">Search</label>
                        <div class="input-group">
                            <input type="text" name="search" class="form-control"
                                   placeholder="ID, Name, Email…" 
                                   value="<?=htmlspecialchars($search)?>">
                            <button class="btn btn-primary">Search</button>
                        </div>
                    </div>
                    <div class="col-md-3 d-flex align-items-end">
                        <button type="button" class="btn btn-secondary" onclick="resetFilters()">Reset</button>
                    </div>
                    <input type="hidden" name="page_members" value="<?=$page?>">
                    <input type="hidden" name="per_page_members" value="<?=$per_page?>">
                </div>
            </form>

            <!-- Members Table -->
            <div class="card mb-4 print-card">
                <div class="card-header">
                    Members List (Total: <?=$total_rows?>)
                </div>
                <div class="card-body table-responsive">
                    <table class="table table-striped table-hover" id="membersTable">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Photo</th>
                                <th>Name</th>
                                <th>Sex</th>
                                <th>Email</th>
                                <th>Phone</th>
                                <th>Status</th>
                                <th>Expiry</th>
                                <th>Position</th>
                                <th class="no-print">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($members)): ?>
                                <tr><td colspan="10" class="text-center no-data">No members found.</td></tr>
                            <?php else: foreach ($members as $m): ?>
                                <tr>
                                    <td><?=htmlspecialchars($m['member_id'] ?? '-')?></td>
                                    <td>
                                        <?php if (!empty($m['profile_image'])): ?>
                                            <img src="uploads/profiles/<?=htmlspecialchars($m['profile_image'])?>"
                                                 width="40" class="rounded-circle" alt="Profile">
                                        <?php else: ?>
                                            <div class="bg-secondary rounded-circle" style="width:40px;height:40px;"></div>
                                        <?php endif; ?>
                                    </td>
                                    <td><a href="view.php?id=<?=$m['id']?>"><?=htmlspecialchars($m['full_name'])?></a></td>
                                    <td><?=ucfirst($m['sex'] ?? '-')?></td>
                                    <td><?=htmlspecialchars($m['email'] ?? '-')?></td>
                                    <td><?=htmlspecialchars($m['phone'] ?? '-')?></td>
                                    <td><span class="badge bg-<?=status_color($m['membership_status'])?>">
                                        <?=htmlspecialchars($m['membership_status'] ?? '-')?>
                                    </span></td>
                                    <td><?=format_date($m['membership_expiry'])?></td>
                                    <td><?=htmlspecialchars($m['position'] ?? '-')?></td>
                                    <td class="no-print">
                                        <a href="edit.php?id=<?=$m['id']?>" class="btn btn-sm btn-warning">Edit</a>
                                        <a href="id-card.php?id=<?=$m['id']?>" target="_blank" class="btn btn-sm btn-info">ID Card</a>
                                    </td>
                                </tr>
                            <?php endforeach; endif; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Pagination -->
                <nav aria-label="Page navigation">
                    <ul class="pagination">
                        <?php if ($page > 1): ?>
                            <li class="page-item">
                                <a class="page-link" href="?<?=http_build_query(array_merge($_GET, ['page_members' => $page-1]))?>">Prev</a>
                            </li>
                        <?php endif; ?>
                        <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                            <li class="page-item <?=($i == $page) ? 'active' : ''?>">
                                <a class="page-link" href="?<?=http_build_query(array_merge($_GET, ['page_members' => $i]))?>"><?=$i?></a>
                            </li>
                        <?php endfor; ?>
                        <?php if ($page < $total_pages): ?>
                            <li class="page-item">
                                <a class="page-link" href="?<?=http_build_query(array_merge($_GET, ['page_members' => $page+1]))?>">Next</a>
                            </li>
                        <?php endif; ?>
                        <li class="page-item">
                            <select class="form-select d-inline w-auto ms-2" onchange="updatePerPage('members', this.value)">
                                <?php foreach ([5,10,20,50,100] as $pp): ?>
                                    <option value="<?=$pp?>" <?=($per_page == $pp) ? 'selected' : ''?>><?=$pp?> per page</option>
                                <?php endforeach; ?>
                            </select>
                        </li>
                    </ul>
                </nav>

                <a href="?export=csv&<?=http_build_query($_GET)?>" class="btn btn-outline-secondary mt-2">Export CSV</a>
            </div>
        </main>
    </div>
</div>

<script>
function resetFilters() {
    const f = document.querySelector('form.filter-section');
    f.reset();
    f.querySelector('[name="search"]').value = '';
    f.querySelector('[name="page_members"]').value = '1';
    f.querySelector('[name="per_page_members"]').value = '20';
    f.submit();
}
function updatePerPage(table, val) {
    const u = new URL(location);
    u.searchParams.set('per_page_' + table, val);
    u.searchParams.set('page_' + table, '1');
    location = u.href;
}
function printTable(id) {
    const table = document.getElementById(id);
    if (!table) return;
    const clone = table.cloneNode(true);
    const win = window.open('', '', 'width=900,height=700');
    win.document.write(`
        <html><head><title>Members List</title>
        <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
        <style>@media print{
            body{font-size:10pt;}
            .table thead th{background:#f2f2f2 !important;-webkit-print-color-adjust:exact;}
            table{width:100%;border-collapse:collapse;}
            th,td{border:1px solid #dee2e6;padding:8px;}
            img,.no-print{display:none;}
        }</style>
        </head><body>
        <div class="container mt-4"><h2 class="text-center">Members List</h2>${clone.outerHTML}</div>
        </body></html>
    `);
    win.document.close();
    win.focus();
    setTimeout(() => win.print(), 500);
}
</script>

<?php
if (isset($_GET['export'])) {
    require_once 'export.php';
    exit;
}
?>

<?php require_once 'includes/footer.php'; ?>