<?php
/* --------------------------------------------------------------
  fund_requests_report.php
  - Generates a filterable report of all fund requests.
  - Accessible only by admin/super_admin.
  -------------------------------------------------------------- */
session_start();
ini_set('display_errors', 1);
error_reporting(E_ALL);

require_once 'includes/functions.php';
require_once 'includes/db_config.php';

/* ---------- Auth & Permissions ---------- */
if (!is_logged_in() || !in_array($_SESSION['user_role'], ['admin', 'super_admin'])) {
    redirect('login.php');
}
$is_admin = is_admin(); // For template checks

/* ---------- Settings ---------- */
$system_name = get_setting('system_name', $link) ?? 'Club Portal';
$currency_symbol = get_setting('currency_symbol', $link) ?? '₦';
$currency_symbol_pdf = 'NGN '; // PDF Safe currency symbol

/* ---------- Filters ---------- */
$filter_from_date = sanitize_input($_GET['from_date'] ?? date('Y-m-01'));
$filter_to_date = sanitize_input($_GET['to_date'] ?? date('Y-m-d'));
$filter_status = sanitize_input($_GET['status'] ?? 'all');
$filter_type_id = (int)($_GET['type_id'] ?? 0);

/* ---------- Fetch Data for Filters ---------- */
$request_types = [];
$sql_types = "SELECT id, type_name FROM request_types ORDER BY type_name";
if ($result_types = mysqli_query($link, $sql_types)) {
    while ($row = mysqli_fetch_assoc($result_types)) {
        $request_types[] = $row;
    }
}

/* ---------- Build Main Query ---------- */
$sql = "
    SELECT 
        fr.id, 
        fr.amount, 
        fr.purpose, 
        fr.status, 
        fr.created_at, 
        fr.admin_note, 
        fr.approval_date,
        u_req.full_name AS requester_name,
        u_app.full_name AS approver_name,
        rt.type_name AS request_type_name
    FROM 
        fund_requests fr
    LEFT JOIN 
        users u_req ON fr.requester_id = u_req.id
    LEFT JOIN 
        users u_app ON fr.admin_approver_id = u_app.id
    LEFT JOIN 
        request_types rt ON fr.request_type_id = rt.id
    WHERE 
        fr.created_at BETWEEN ? AND ?
";

$params = [
    $filter_from_date . ' 00:00:00',
    $filter_to_date . ' 23:59:59'
];
$types = "ss";

if ($filter_status !== 'all' && in_array($filter_status, ['pending', 'approved', 'rejected', 'spent'])) {
    $sql .= " AND fr.status = ?";
    $params[] = $filter_status;
    $types .= "s";
}

if ($filter_type_id > 0) {
    $sql .= " AND fr.request_type_id = ?";
    $params[] = $filter_type_id;
    $types .= "i";
}

$sql .= " ORDER BY fr.created_at DESC";

/* ---------- Execute Query ---------- */
$requests = [];
if ($stmt = mysqli_prepare($link, $sql)) {
    mysqli_stmt_bind_param($stmt, $types, ...$params);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    while ($row = mysqli_fetch_assoc($result)) {
        $requests[] = $row;
    }
    mysqli_stmt_close($stmt);
} else {
    $error_message = "Error fetching report data: " . mysqli_error($link);
}

/* ---------- Helper Function (can be moved to functions.php) ---------- */
if (!function_exists('get_status_badge')) {
    function get_status_badge($status) {
        switch ($status) {
            case 'pending': return 'badge bg-warning text-dark';
            case 'approved': return 'badge bg-success';
            case 'rejected': return 'badge bg-danger';
            case 'partially_spent': return 'badge bg-info';
            case 'spent': return 'badge bg-secondary';
            default: return 'badge bg-light text-dark';
        }
    }
}

/* ---------- Include Template ---------- */
require_once 'templates/fund_requests_template.php';
?>