<?php
// fetch_member_details.php
require_once 'includes/functions.php';
require_once 'includes/db_config.php';

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

if (!is_logged_in() || !is_admin()) {
    echo json_encode(['error' => 'Unauthorized']);
    exit;
}

$id = intval($_GET['id'] ?? 0);
if ($id <= 0) {
    echo json_encode(['error' => 'Invalid member ID.']);
    exit;
}

$system_name = get_setting('system_name', $link) ?? 'Club Portal';

$sql = "SELECT 
            u.id, 
            u.member_id, 
            u.full_name, 
            u.email, 
            u.membership_status, 
            u.profile_picture,
            fd.data AS form_data, 
            f.form_data AS form_structure
        FROM users u
        LEFT JOIN form_data fd ON u.id = fd.user_id
        LEFT JOIN forms f ON fd.form_id = f.id
        WHERE u.id = ?";

$stmt = mysqli_prepare($link, $sql);
mysqli_stmt_bind_param($stmt, 'i', $id);
mysqli_stmt_execute($stmt);
$res = mysqli_stmt_get_result($stmt);

if ($row = mysqli_fetch_assoc($res)) {
    $form_data_raw = $row['form_data'];
    $form_structure_raw = $row['form_structure'];
    
    $form_data = json_decode($form_data_raw, true) ?: [];
    $form_structure = json_decode($form_structure_raw, true) ?: [];

    $mapped_data = [];
    $label_map = [];
    $structure_field_names = [];

    // 1. Build a label map and a list of names from the structure
    if (isset($form_structure['fields']) && is_array($form_structure['fields'])) {
        foreach ($form_structure['fields'] as $field) {
            $name = $field['name'] ?? null;
            $label = $field['label'] ?? $name;
            if ($name) {
                $label_map[$name] = $label;
                $structure_field_names[] = $name; // Keep structure order
            }
        }
    }

    // 2. Get all field names from the actual saved data
    $data_field_names = array_keys($form_data);

    // 3. Combine them: structure fields first (in order), then any data-only fields
    $all_field_names = $structure_field_names;
    foreach ($data_field_names as $data_name) {
        if (!in_array($data_name, $all_field_names)) {
            $all_field_names[] = $data_name; // Add fields that exist in data but not structure
        }
    }

    // 4. Build the final mapped_data array
    foreach ($all_field_names as $name) {
        // Get label: from map, or prettify the name as a fallback
        $label = $label_map[$name] ?? ucfirst(str_replace(['_', '-'], ' ', $name));
        
        // Get value: from data, or empty string
        $value = $form_data[$name] ?? '';

        // FIX: Handle array values (e.g., checkboxes) for display in a text input
        if (is_array($value)) {
            $value = implode(', ', $value);
        }

        $mapped_data[] = [
            'label' => $label,
            'name'  => $name,
            'value' => (string) $value // Ensure value is a string for the modal
        ];
    }

    $output = [
        'id'                => $row['id'],
        'member_id'         => $row['member_id'],
        'full_name'         => $row['full_name'],
        'email'             => $row['email'],
        'membership_status' => $row['membership_status'],
        'profile_picture'   => !empty($row['profile_picture']) 
                                 ? $row['profile_picture'] 
                                 : 'uploads/profile_pictures/default.png',
        'form_data'         => $mapped_data,
        'system_name'       => $system_name
    ];

    echo json_encode($output);
} else {
    echo json_encode(['error' => 'Member not found']);
}
mysqli_stmt_close($stmt);
?>
