$(function() {
    // This is our in-memory "database" for all field properties
    // that aren't stored in the DOM, like options, required, and conditional logic.
    let fieldConfig = {};

    // --- Initialization ---

    // Initializes draggable functionality for form fields from the palette
    $(".draggable-field").draggable({
        helper: "clone",
        revert: "invalid",
        connectToSortable: "#form-canvas"
    });

    // Initializes sortable functionality for the form canvas
    $("#form-canvas").sortable({
        placeholder: "ui-state-highlight",
        forcePlaceholderSize: true,
        // stop is called when a field is dropped or sorted
        stop: function(event, ui) {
            // Check if the item is a *new* field being dragged from the palette
            if (ui.item.hasClass("draggable-field")) {
                const fieldType = ui.item.data("type");
                const fieldId = `field_${Date.now()}`;
                const fieldName = `${fieldType}_${Date.now()}`;
                const fieldLabel = ui.item.text();

                // Create the new field element
                const newField = createFieldElement(fieldType, {
                    id: fieldId,
                    name: fieldName,
                    label: fieldLabel
                });

                // Replace the temporary clone with the permanent field element
                ui.item.replaceWith(newField);
                
                // Add default config for the new field
                fieldConfig[fieldId] = {
                    required: false,
                    options: [],
                    consent_text: '', // +++ ADDED default for new field type
                    conditional_logic: {
                        action: 'show', // default action
                        match: 'all',   // default match type
                        rules: []       // empty rules
                    }
                };

                updateFormStructure();
            }
        }
    });

    // --- Core Functions ---

    /**
     * Creates a new form field HTML element for the canvas.
     */
    function createFieldElement(type, data = {}) {
        const fieldId = data.id || `field_${Date.now()}`;
        const labelText = data.label || `${type.charAt(0).toUpperCase() + type.slice(1)} Label`;
        const fieldName = data.name || `${type}_${Date.now()}`;
        const requiredIndicator = data.required ? '<span class="required-indicator">*</span>' : '';

        // This is the HTML that appears on the form canvas
        const fieldHtml = `
            <div class="form-field form-field-container" data-type="${type}" data-id="${fieldId}">
                <div class="field-header">
                    <span class="field-label">${labelText} ${requiredIndicator}</span>
                    <div class="field-actions">
                        <button type="button" class="btn btn-sm btn-secondary edit-field">Edit</button>
                        <button type="button" class="btn btn-sm btn-danger remove-field">Remove</button>
                    </div>
                </div>
                <div class="field-preview">
                    <p class="field-type-badge">${type}</p>
                </div>
                <input type="hidden" class="field-name" value="${fieldName}">
            </div>
        `;
        return $(fieldHtml);
    }
    
    /**
     * Gathers all data from the canvas and fieldConfig,
     * serializes it to JSON, and puts it in the hidden form input.
     */
    function updateFormStructure() {
        const formStructure = { fields: [] };
        $("#form-canvas > .form-field").each(function(fieldIndex) {
            const fieldElement = $(this);
            const fieldId = fieldElement.data("id");
            const fieldType = fieldElement.data("type");
            const fieldName = fieldElement.find(".field-name").val();
            const fieldLabel = fieldElement.find(".field-label").text().replace('*', '').trim();
            
            const fieldData = {
                id: fieldId,
                type: fieldType,
                name: fieldName,
                label: fieldLabel,
                order: fieldIndex
            };
            
            // Merge with fieldConfig data (required, options, conditional_logic, consent_text)
            if (fieldConfig[fieldId]) {
                Object.assign(fieldData, fieldConfig[fieldId]);
            }

            formStructure.fields.push(fieldData);
        });

        // Set the value of the hidden input
        $("#form-data-input").val(JSON.stringify(formStructure));
    }

    /**
     * Populates the field edit modal with data for the selected field.
     */
    function populateFieldModal(fieldData) {
        // Store the ID of the field being edited on the modal itself for easy access
        $("#field-edit-modal").data("currentFieldId", fieldData.id);

        $("#field-label-edit").val(fieldData.label);
        $("#field-name-edit").val(fieldData.name);
        $("#field-required-edit").prop("checked", fieldData.required);
        
        // --- +++ UPDATED to handle 'consent' field ---
        const type = fieldData.type;
        if (type === 'select' || type === 'radio' || type === 'checkbox') {
            $("#field-options-group").show();
            $("#field-consent-text-group").hide(); // explicit hide
            const options = fieldData.options || [];
            $("#field-options-edit").val(options.map(opt => opt.label).join('\n'));
        } else if (type === 'consent') {
            $("#field-options-group").hide();
            $("#field-consent-text-group").show();
            $("#field-consent-text-edit").val(fieldData.consent_text || '');
        } else {
            // For text, email, etc.
            $("#field-options-group").hide();
            $("#field-consent-text-group").hide();
            $("#field-options-edit").val('');
            $("#field-consent-text-edit").val('');
        }
        // --- +++ END UPDATE +++
        
        // --- Populate Conditional Logic ---
        const logicGroup = $("#conditional-logic-group");
        
        // ENHANCED: Inject the Action/Match dropdowns if they don't exist
        if (logicGroup.find("#conditional-action-type").length === 0) {
            const logicHeader = `
                <div class="form-row" style="margin-bottom: 15px; display: flex; gap: 10px;">
                    <div class="col" style="flex: 1;">
                        <label>Action</label>
                        <select id="conditional-action-type" class="form-control">
                            <option value="show">Show</option>
                            <option value="hide">Hide</option>
                            <option value="enable">Enable</option>
                            <option value="disable">Disable</option>
                            <option value="require">Make Required</option>
                        </select>
                    </div>
                    <div class="col" style="flex: 1;">
                        <label>Grouping</label>
                        <select id="conditional-match-type" class="form-control">
                            <option value="all">if ALL match</option>
                            <option value="any">if ANY match</option>
                        </select>
                    </div>
                </div>
                <h5>Rules</h5>`;
            // Replace the original <h5> with our new, richer header
            logicGroup.find("h5").replaceWith(logicHeader);
        }

        // ENHANCED: Read from the new logic structure
        const logic = fieldData.conditional_logic || {};
        $("#conditional-action-type").val(logic.action || 'show');
        $("#conditional-match-type").val(logic.match || 'all');
        
        // Populate rules
        const rules = logic.rules || [];
        $("#conditional-rules").empty();
        if (rules.length > 0) {
            rules.forEach(rule => { 
                addConditionalRule(rule.field_id, rule.operator, rule.value); 
            });
        }
        
        $("#field-edit-modal").fadeIn();
    }

    /**
     * Adds a new conditional rule row to the modal.
     */
    function addConditionalRule(fieldId = '', operator = 'equals', value = '') {
        // ENHANCED: Added all new operators
        const operatorsHtml = `
            <select class="form-control conditional-operator-select">
                <option value="equals" ${operator == 'equals' ? 'selected' : ''}>Equals</option>
                <option value="not_equals" ${operator == 'not_equals' ? 'selected' : ''}>Not Equals</option>
                <option value="is_empty" ${operator == 'is_empty' ? 'selected' : ''}>Is Empty</option>
                <option value="is_not_empty" ${operator == 'is_not_empty' ? 'selected' : ''}>Is Not Empty</option>
                <option value="contains" ${operator == 'contains' ? 'selected' : ''}>Contains</option>
                <option value="greater_than" ${operator == 'greater_than' ? 'selected' : ''}>Greater Than</option>
                <option value="less_than" ${operator == 'less_than' ? 'selected' : ''}>Less Than</option>
            </select>`;
        
        // ENHANCED: Use the smart function to generate the correct value input
        const valueInputHtml = generateConditionalValueInput(fieldId, value);
        
        const ruleHtml = `
            <div class="conditional-rule-row" style="display: flex; gap: 5px; margin-bottom: 5px;">
                <select class="form-control conditional-field-select" style="flex: 2;"></select>
                ${operatorsHtml}
                <div class="conditional-value-container" style="flex: 2;">${valueInputHtml}</div>
                <button type="button" class="btn btn-sm btn-danger remove-conditional-rule">X</button>
            </div>`;
            
        const ruleRow = $(ruleHtml);
        
        // Append the new row to the DOM
        $("#conditional-rules").append(ruleRow);
        
        // Populate its field dropdown *after* it's in the DOM
        updateConditionalFieldOptions(ruleRow.find('.conditional-field-select'), fieldId); 
        
        // Trigger a change on the operator to correctly show/hide the value input
        ruleRow.find('.conditional-operator-select').trigger('change');
    }
    
    /**
     * ENHANCED: Generates the correct value input (text, select, etc.)
     * based on the target field's type.
     */
    function generateConditionalValueInput(targetFieldId, currentValue = '') {
        if (!targetFieldId) {
            return `<input type="text" class="form-control conditional-value" value="${currentValue}">`;
        }
        const targetField = $(`[data-id="${targetFieldId}"]`);
        const targetType = targetField.data("type");
        const targetConfig = fieldConfig[targetFieldId];
        let html = '';

        if ((targetType === 'select' || targetType === 'radio') && targetConfig && targetConfig.options.length > 0) {
            html = `<select class="form-control conditional-value">`;
            targetConfig.options.forEach(opt => {
                html += `<option value="${opt.value}" ${opt.value == currentValue ? 'selected' : ''}>${opt.label}</option>`;
            });
            html += `</select>`;
        } else if (targetType === 'checkbox' || targetType === 'consent') { // +++ 'consent' behaves like a checkbox for logic
             html = `<select class="form-control conditional-value">
                <option value="1" ${currentValue == '1' ? 'selected' : ''}>Checked</option>
                <option value="0" ${currentValue == '0' ? 'selected' : ''}>Not Checked</option>
             </select>`;
        } else {
             html = `<input type="text" class="form-control conditional-value" value="${currentValue}">`;
        }
        return html;
    }

    /**
     * Updates the list of fields for conditional logic dropdowns.
     * Can optionally target a single dropdown (for new rows).
     */
    function updateConditionalFieldOptions(targetSelect, selectedId = null) {
        const allFields = [];
        $("#form-canvas > .form-field").each(function() {
            allFields.push({
                id: $(this).data("id"),
                label: $(this).find(".field-label").text().replace('*', '').trim()
            });
        });

        // Get the ID of the field we are *currently editing*
        const currentFieldId = $("#field-edit-modal").data("currentFieldId");
        
        // Determine which dropdowns to update
        const selectsToUpdate = targetSelect ? $(targetSelect) : $(".conditional-field-select");

        selectsToUpdate.each(function() {
            const select = $(this);
            const currentValue = selectedId || select.val(); // Use passed ID or its own value
            select.empty();
            select.append('<option value="">- Select Field -</option>');
            allFields.forEach(field => {
                // A field cannot be conditional on itself
                if (field.id !== currentFieldId) { 
                    select.append(`<option value="${field.id}">${field.label}</option>`);
                }
            });
            if (currentValue) {
                select.val(currentValue);
            }
        });
    }

    // --- Event Listeners ---

    // Event listener for clicking the "Edit" button on a form field
    $("body").on("click", ".edit-field", function(e) {
        e.preventDefault();
        e.stopPropagation();
        const fieldElement = $(this).closest(".form-field");
        const fieldId = fieldElement.data("id");
        
        // Start with the stored config (options, logic, etc.)
        const fieldData = JSON.parse(JSON.stringify(fieldConfig[fieldId] || {}));
        
        // Add current DOM state to fieldData for modal population
        Object.assign(fieldData, {
            id: fieldId,
            type: fieldElement.data("type"),
            name: fieldElement.find(".field-name").val(),
            label: fieldElement.find(".field-label").text().replace('*', '').trim()
        });

        populateFieldModal(fieldData);
    });

    // Event listener for saving changes from the field edit modal
    $("#save-field-edit").on("click", function() {
        const newLabel = $("#field-label-edit").val().trim();
        const isRequired = $("#field-required-edit").is(":checked");
        const optionsText = $("#field-options-edit").val().trim();
        const consentText = $("#field-consent-text-edit").val().trim(); // +++ NEW: Read consent text
        const currentFieldId = $("#field-edit-modal").data("currentFieldId");
        const fieldElement = $(`[data-id="${currentFieldId}"]`);
        
        // --- ENHANCED: Build new Conditional Logic object ---
        const rules = [];
        $("#conditional-rules .conditional-rule-row").each(function() {
            const ruleRow = $(this);
            rules.push({
                field_id: ruleRow.find(".conditional-field-select").val(),
                operator: ruleRow.find(".conditional-operator-select").val(),
                value: ruleRow.find(".conditional-value").val(), // Use new class
            });
        });
        
        const conditionalLogic = {
            action: $("#conditional-action-type").val(),
            match: $("#conditional-match-type").val(),
            rules: rules
        };
        // --- End Logic ---
        
        if (newLabel !== "") {
            fieldElement.find(".field-label").html(newLabel + (isRequired ? ' <span class="required-indicator">*</span>' : ''));
            
            // Rebuild options array with value and label
            const options = optionsText.split('\n').map(o => o.trim()).filter(o => o !== '').map(o => ({
                label: o,
                value: o.toLowerCase().replace(/\s/g, '_') // Create a stable value
            }));

            // Save all data back to our in-memory config
            fieldConfig[currentFieldId] = {
                required: isRequired,
                options: options,
                consent_text: consentText, // +++ NEW: Save consent text
                conditional_logic: conditionalLogic // Save the new object
            };
            
            // Re-serialize everything to the hidden input
            updateFormStructure();
        }
        $("#field-edit-modal").fadeOut();
    });

    // Event listener for adding a new conditional rule
    $("#add-conditional-rule").on("click", function() {
        addConditionalRule(); // Add an empty rule
    });

    // Event listener for removing a conditional rule
    $("#conditional-rules").on("click", ".remove-conditional-rule", function() { 
        $(this).closest(".conditional-rule-row").remove(); 
    });

    // Event listener for closing the modal
    $(".close-button").on("click", function() { 
        $("#field-edit-modal").fadeOut(); 
    });

    // Event listener for removing a form field from the canvas
    $("body").on("click", ".remove-field", function() {
        const fieldId = $(this).closest(".form-field").data("id");
        $(this).closest(".form-field").remove();
        delete fieldConfig[fieldId];
        updateFormStructure();
    });

    // --- NEW Event Listeners for Smart Logic ---

    // 1. When a user changes the *field* in a rule
    $("#conditional-rules").on("change", ".conditional-field-select", function() {
        const row = $(this).closest(".conditional-rule-row");
        const fieldId = $(this).val();
        const valueContainer = row.find(".conditional-value-container");
        
        // Regenerate the value input based on the new target field
        const newHtml = generateConditionalValueInput(fieldId, ''); 
        valueContainer.html(newHtml);
    });

    // 2. When a user changes the *operator* in a rule
    $("#conditional-rules").on("change", ".conditional-operator-select", function() {
        const row = $(this).closest(".conditional-rule-row");
        const operator = $(this).val();
        const valueContainer = row.find(".conditional-value-container");
        
        // Hide the value input if it's not needed
        if (operator === 'is_empty' || operator === 'is_not_empty') {
            valueContainer.hide();
        } else {
            valueContainer.show();
        }
    });


    // --- Load Existing Form ---
    // This runs on page load if we are editing a form
    if (typeof existingFormData !== 'undefined' && existingFormData && existingFormData.fields && existingFormData.fields.length > 0) {
        const formCanvas = $("#form-canvas");
        formCanvas.empty();
        existingFormData.fields.forEach(fieldData => {
            const fieldElement = createFieldElement(fieldData.type, {
                id: fieldData.id,
                name: fieldData.name,
                label: fieldData.label,
                required: fieldData.required
            });
            
            // Store the full config for the loaded field
            fieldConfig[fieldData.id] = {
                required: fieldData.required,
                options: fieldData.options || [],
                consent_text: fieldData.consent_text || '', // +++ NEW: Load consent text
                conditional_logic: fieldData.conditional_logic || { action: 'show', match: 'all', rules: [] } // Ensure structure
            };

            formCanvas.append(fieldElement);
        });
        updateFormStructure();
    }
});