<?php
header('Content-Type: application/json');

require_once '../includes/db_config.php';
require_once '../includes/functions.php';

// Check if the user is an admin
if (!is_admin()) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access.']);
    exit();
}

$response = ['success' => false, 'message' => 'An unknown error occurred.', 'members' => []];

try {
    $search = $_GET['search'] ?? '';
    $period = $_GET['period'] ?? 'all';
    $is_debtors = filter_var($_GET['is_debtors'] ?? false, FILTER_VALIDATE_BOOLEAN);

    // Base SQL query
    $sql = "
    SELECT
        u.id AS member_id,
        u.surname,
        u.other_names,
        u.profile_picture,
        u.membership_status,
        p.payment_type,
        p.amount AS amount_paid,
        p.payment_status,
        p.payment_date
    FROM
        users u
    LEFT JOIN
        payments p ON u.id = p.member_id
    ";

    $where_clauses = [];
    $params = [];
    $param_types = '';

    // Add search filter
    if (!empty($search)) {
        $where_clauses[] = "(u.id LIKE ? OR u.surname LIKE ? OR u.other_names LIKE ?)";
        $searchTerm = "%{$search}%";
        $params[] = $searchTerm;
        $params[] = $searchTerm;
        $params[] = $searchTerm;
        $param_types .= 'sss';
    }

    // Add period filter
    $today = date('Y-m-d');
    $start_date = '';
    
    switch ($period) {
        case 'today':
            $start_date = $today;
            $where_clauses[] = "DATE(p.payment_date) = ?";
            $params[] = $start_date;
            $param_types .= 's';
            break;
        case 'this_week':
            $start_date = date('Y-m-d', strtotime('last Sunday'));
            $where_clauses[] = "p.payment_date >= ?";
            $params[] = $start_date;
            $param_types .= 's';
            break;
        case 'this_month':
            $start_date = date('Y-m-01');
            $where_clauses[] = "p.payment_date >= ?";
            $params[] = $start_date;
            $param_types .= 's';
            break;
        case 'this_year':
            $start_date = date('Y-01-01');
            $where_clauses[] = "p.payment_date >= ?";
            $params[] = $start_date;
            $param_types .= 's';
            break;
    }

    // Add debtors filter if requested
    if ($is_debtors) {
        $where_clauses[] = "p.payment_status = 'Not Paid'";
    }

    // Build the final WHERE clause
    if (!empty($where_clauses)) {
        $sql .= " WHERE " . implode(" AND ", $where_clauses);
    }

    $sql .= " ORDER BY u.surname ASC, u.other_names ASC";
    
    $stmt = $link->prepare($sql);

    if (!empty($params)) {
        $stmt->bind_param($param_types, ...$params);
    }
    
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows > 0) {
        $members = [];
        while ($row = $result->fetch_assoc()) {
            $members[] = $row;
        }
        $response['success'] = true;
        $response['message'] = 'Data fetched successfully.';
        $response['members'] = $members;
    } else {
        $response['success'] = true;
        $response['message'] = 'No financial data found matching the criteria.';
        $response['members'] = [];
    }
} catch (Exception $e) {
    $response['message'] = 'Database error: ' . $e->getMessage();
}

$link->close();

echo json_encode($response);
?>
