<?php
/**
 * admin_financial_report.php
 * -----------------------------------------------------------------------
 * NEW Entry Point (Controller) for the Financial Report System.
 * Replaces the monolithic admin_financial_report.php with a modular design.
 * -----------------------------------------------------------------------
 */
define('IN_REPORTS', true);
session_start();

// --- Core Application Includes ---
// Ensure these paths are correct relative to this file
require_once 'includes/functions.php';      // Auth and general utility functions (must be present)
require_once 'includes/db_config.php';     // Database connection setup (must provide $link)

// --- Modular Reporting Includes (Your new structure) ---
require_once 'includes/financial_queries.php'; // Model: Data fetching functions
require_once 'includes/financial_utils.php';   // Helpers: Sanitization, dropdowns

/* ---------- Development error reporting (remove in prod) ---------- */
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

/* ---------- DB connection check ---------- */
if (!$link) {
    die("ERROR: Could not connect. " . mysqli_connect_error());
}

/* ---------- Auth ---------- */
if (!is_logged_in()) {
    redirect('login.php');
}
$is_admin = is_admin();

// --- System Settings ---
// Assumes get_setting() is defined in includes/functions.php
$system_name      = get_setting('system_name', $link) ?? 'Club Portal';
$currency_symbol  = get_setting('currency_symbol', $link) ?? '₦';

/* ---------- Sanitize & defaults for User Input (Filters) ---------- */
$valid_record_types = ['all', 'payments', 'expenses', 'debts', 'benefits'];
$valid_time_periods = ['all', 'range', 'week', 'month', 'year'];
$valid_per_page     = [5, 10, 25, 50, 100];

// Record Type Filter (e.g., 'payments', 'all')
$record_type = sanitize_filter($_GET['record_type'] ?? 'all', $valid_record_types, 'all');

// Time Period Filter (e.g., 'month', 'range')
$time_period = sanitize_filter($_GET['time_period'] ?? 'all', $valid_time_periods, 'all');

// Specific Type Filter (e.g., 'income_Dues', 'expense_Utility')
$type_filter = $_GET['type_filter'] ?? '';
$type_filter_sanitized = validate_type_filter($type_filter, $record_type);

// Date Range (if time_period is 'range')
$start_date = $_GET['start_date'] ?? null;
$end_date   = $_GET['end_date'] ?? null;

// Pagination
$per_page = sanitize_filter((int)($_GET['per_page'] ?? 10), $valid_per_page, 10);
$page     = (int)($_GET['page'] ?? 1);
$offset   = ($page - 1) * $per_page;

// Specific Filters for Ledger History
$type_filter_payments = ($record_type === 'payments' && $type_filter_sanitized) ? $type_filter_sanitized : null;
$type_filter_expenses = ($record_type === 'expenses' && $type_filter_sanitized) ? $type_filter_sanitized : null;
$type_filter_benefits = ($record_type === 'benefits' && $type_filter_sanitized) ? $type_filter_sanitized : null;
$member_id            = (int)($_GET['member_id'] ?? 0); // Assuming member_id is an integer

/* ---------- Dynamic Date Field Selection for Queries ---------- */
$date_field = 'p.date';
if ($record_type === 'expenses') {
    $date_field = 'e.date';
} elseif ($record_type === 'benefits') {
    $date_field = 'b.date';
} elseif ($record_type === 'debts') {
    $date_field = 'd.date';
}

/* ----------------------------------------------------------------
   PHASE 2 ACTION: Data Fetching (Model Calls)
   ---------------------------------------------------------------- */

// 1. KPI Counts (Used for Pie Chart and Summary Cards)
$kpis = get_total_counts($link, $record_type, $time_period, $start_date, $end_date, $type_filter_sanitized);

// 2. Summary of Types (Used for Histogram)
$payment_types_summary = get_payment_types_summary($link, $record_type, $time_period,
                                                   $start_date, $end_date, $type_filter_payments);
$expense_types_summary = get_expense_types_summary($link, $record_type, $time_period,
                                                   $start_date, $end_date, $type_filter_expenses);
$benefit_types_summary = get_benefit_types_summary($link, $record_type, $time_period,
                                                   $start_date, $end_date, $type_filter_benefits,
                                                   $member_id);

// 3. Line Chart Data (Monthly Trend)
$line_chart_data = get_line_chart_data($link, $end_date, $date_field);


/* --- MISSING FEATURE INTEGRATION (To be added to financial_queries.php next) --- */

// NOTE: These variables are required to maintain full feature parity with the old report.
// The functions (get_bank_account_balances, get_fund_requests) need to be created in
// 'includes/financial_queries.php' in the next step.

// $bank_balances = get_bank_account_balances($link);
// $fund_requests = get_fund_requests($link, $time_period, $start_date, $end_date);


/* ---------- Build safe histogram arrays (for Chart.js) ---------- */
$histogram_labels = $histogram_paid = $histogram_excess = [];

// Payments
foreach (($payment_types_summary ?? []) as $row) {
    $histogram_labels[] = "'" . addslashes($row['type_name']) . " (P)'";
    $histogram_paid[]   = $row['total_paid'];
    $histogram_excess[] = $row['excess_amount'];
}
// Expenses
foreach (($expense_types_summary ?? []) as $row) {
    $histogram_labels[] = "'" . addslashes($row['type_name']) . " (E)'";
    $histogram_paid[]   = $row['total'];
    $histogram_excess[] = 0;
}
// Benefits
foreach (($benefit_types_summary ?? []) as $row) {
    $histogram_labels[] = "'" . addslashes($row['type_name']) . " (B)'";
    $histogram_paid[]   = $row['total'];
    $histogram_excess[] = 0;
}

/* ----------------------------------------------------------------
   Ledger History (The main table data)
   ---------------------------------------------------------------- */
$ledger_history_data = get_ledger_history($link, $record_type, $time_period, $start_date, $end_date,
                                          $type_filter_sanitized, $per_page, $offset, $member_id);

// Total records for pagination
$total_records = $ledger_history_data['total_count'];
$ledger_records = $ledger_history_data['records'];
$total_pages = ceil($total_records / $per_page);

// Generate Options for Dropdowns using financial_utils.php
$type_options_payments = get_type_options($link, 'payments', $type_filter_sanitized);
$type_options_expenses = get_type_options($link, 'expenses', $type_filter_sanitized);
$type_options_benefits = get_type_options($link, 'benefits', $type_filter_sanitized);


/* ----------------------------------------------------------------
   PHASE 3 ACTION: Render View
   ---------------------------------------------------------------- */

// Includes the template and displays all the data, charts, and tables.
// Ensure this path is correct relative to this file
require_once 'templates/financial_summary_template.php';

// Close database connection
mysqli_close($link);
?>
