<?php

// admin_config_finance.php - Allows Admin to configure financial settings, manage bank accounts,
// and review fund requests.

ob_start();
session_start();
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Enable mysqli reporting for consistent error handling
// We leave this commented out as per the original file's observation about potential errors.
// mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT); 

if (function_exists('opcache_reset')) { opcache_reset(); }

require_once 'includes/functions.php';
require_once 'includes/db_config.php';

/* ---------- SESSION / PERMISSIONS ---------- */
$timeout = 1800; // 30 minutes
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity']) > $timeout) {
    session_unset(); session_destroy();
    redirect("login.php?timeout=1");
}
$_SESSION['last_activity'] = time();

if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$csrf_token = $_SESSION['csrf_token'];

// Only Admin or Super Admin can access this page
if (!is_logged_in() || !in_array($_SESSION['user_role'], ['admin', 'super_admin'])) {
    redirect("login.php"); exit;
}

/* ------------------ FORM PROCESSING & DATA FETCHING (PHP LOGIC) ------------------- */

$success_message = '';
$error_message = '';
$bank_accounts = [];
$fund_requests = [];
$account_owners = []; // <-- For owner dropdown
// Use get_setting with a link variable if available, otherwise just default
$currency_symbol = get_setting('currency_symbol', $link) ?? '₦';

// --- Fetch users for owner dropdown ---
try {
    // Fetches users who can own an account
    $result_users = $link->query("SELECT id, full_name, role FROM users WHERE role IN ('admin', 'super_admin', 'accountant') AND deleted_at IS NULL ORDER BY full_name");
    $account_owners = $result_users->fetch_all(MYSQLI_ASSOC);
} catch (Exception $e) {
    $error_message .= " Could not fetch users list: " . htmlspecialchars($e->getMessage());
}


// --- 1. Bank Account Management ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['account_action']) && hash_equals($csrf_token, $_POST['csrf_token'])) {
    $action = $_POST['account_action'];
    $account_id = isset($_POST['account_id']) ? (int)$_POST['account_id'] : 0;
    
    // Values for Add/Edit
    $account_name = sanitize_input($_POST['account_name'] ?? '');
    $account_number = sanitize_input($_POST['account_number'] ?? '');
    $bank_name = sanitize_input($_POST['bank_name'] ?? '');
    $is_cash_account = isset($_POST['is_cash']) ? 1 : 0;
    $is_active = isset($_POST['is_active']) ? 1 : 0; 
    
    // --- Get Owner ID ---
    $owner_user_id = (int)($_POST['owner_user_id'] ?? 0);
    // Save as NULL in database if "0" (None) is selected
    $owner_id_to_db = $owner_user_id > 0 ? $owner_user_id : null; 
    
    // --- NEW: Get Initial Balance (for 'add' only) ---
    $initial_balance = (float)($_POST['initial_balance'] ?? 0.00);
    
    // Default to active for new accounts
    if ($action === 'add_account') {
        $is_active = 1; 
    }

    // Basic validation
    if ($action !== 'delete_account' && empty($account_name)) {
        $error_message = "Account Nickname is required.";
    } elseif ($action !== 'delete_account' && $is_cash_account == 0 && empty($bank_name)) {
        $error_message = "Bank Name is required for non-cash accounts.";
    }

    if (empty($error_message)) {
        $link->begin_transaction(); // Start transaction
        try {
            if ($action === 'add_account') {
                // --- UPDATED: Added initial_balance and current_balance fields ---
                $stmt = $link->prepare("INSERT INTO bank_accounts (account_name, account_number, bank_name, is_cash_on_hand, is_active, owner_user_id, initial_balance, current_balance) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
                // Bind param string updated to "sssiiidd" (s=string, i=int, d=double/decimal)
                $stmt->bind_param("sssiiidd", $account_name, $account_number, $bank_name, $is_cash_account, $is_active, $owner_id_to_db, $initial_balance, $initial_balance);
                $stmt->execute();
                
                $new_account_id = $link->insert_id;
                $stmt->close(); // Close the first statement

                // --- NEW: Create Opening Balance Ledger Entry ---
                if ($new_account_id > 0 && $initial_balance != 0.00) {
                    $ledger_type = ($initial_balance > 0) ? 'income' : 'expense';
                    $ledger_amount = abs($initial_balance);
                    $description = "Opening Balance";
                    
                    $ledger_stmt = $link->prepare(
                        "INSERT INTO financial_ledger (account_id, transaction_type, amount, description, balance_after, transaction_date, created_by_id) 
                         VALUES (?, ?, ?, ?, ?, NOW(), ?)"
                    );
                    // Bind params: i=account_id, s=type, d=amount, s=desc, d=balance_after, i=user_id
                    $ledger_stmt->bind_param("isdsdi", $new_account_id, $ledger_type, $ledger_amount, $description, $initial_balance, $_SESSION['user_id']);
                    $ledger_stmt->execute();
                    $ledger_stmt->close();
                    
                    $success_message = "Bank Account '{$account_name}' added with opening balance of {$currency_symbol}" . number_format($initial_balance, 2) . ".";
                } else {
                    $success_message = "Bank Account '{$account_name}' added successfully.";
                }

            } elseif ($action === 'edit_account' && $account_id > 0) {
                // --- UPDATED: Added owner_user_id ---
                $stmt = $link->prepare("UPDATE bank_accounts SET account_name = ?, account_number = ?, bank_name = ?, is_cash_on_hand = ?, is_active = ?, owner_user_id = ? WHERE id = ?");
                // Bind param string corrected to "sssiiii" (owner_id and account_id are integers)
                $stmt->bind_param("sssiiii", $account_name, $account_number, $bank_name, $is_cash_account, $is_active, $owner_id_to_db, $account_id);
                $stmt->execute();
                $stmt->close();
                $success_message = "Bank Account '{$account_name}' updated successfully.";
                
            } elseif ($action === 'delete_account' && $account_id > 0) {
                // Soft Delete/Archive Logic
                
                // 1. Check if balance is zero before deletion
                $balance_check = $link->prepare("SELECT current_balance FROM bank_accounts WHERE id = ?");
                $balance_check->bind_param("i", $account_id);
                $balance_check->execute();
                $result = $balance_check->get_result()->fetch_assoc();
                $balance_check->close();

                if ($result && abs((float)$result['current_balance']) > 0.00) {
                    $error_message = "Cannot archive account: **Balance is not zero** ({$currency_symbol}" . number_format($result['current_balance'], 2) . "). Please zero out the balance first using Fund Transfer.";
                } else {
                    // 2. Perform Soft Delete
                    $stmt = $link->prepare("UPDATE bank_accounts SET deleted_at = NOW(), is_active = 0 WHERE id = ?");
                    $stmt->bind_param("i", $account_id);
                    $stmt->execute();
                    
                    if ($stmt->affected_rows > 0) {
                        $success_message = "Bank Account #{$account_id} successfully archived (soft-deleted).";
                    } else {
                        $error_message = "Error archiving account #{$account_id}. It may already be archived.";
                    }
                    $stmt->close();
                }
            } 
            
            $link->commit(); // Commit all successful changes
            
        } catch (Exception $e) {
            $link->rollback(); // Rollback on error
            $error_message = "Database Error: " . htmlspecialchars($e->getMessage());
        }
    }
    
    // Regenerate token after submission or error
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    $csrf_token = $_SESSION['csrf_token'];
}

// --- 2. Fund Request Review Management (Unchanged Logic) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && (isset($_POST['approve_request']) || isset($_POST['reject_request'])) && hash_equals($csrf_token, $_POST['csrf_token'])) {
    $request_id = (int)($_POST['request_id'] ?? 0);
    $action_status = isset($_POST['approve_request']) ? 'approved' : 'rejected'; 
    $review_comment = sanitize_input($_POST['review_comment'] ?? 'N/A');

    if ($request_id > 0) {
        try {
            $sql_update = "UPDATE fund_requests SET status = ?, admin_note = ?, admin_approver_id = ?, approval_date = NOW() WHERE id = ? AND status = 'pending'";
            $stmt = $link->prepare($sql_update);
            $stmt->bind_param("ssii", $action_status, $review_comment, $_SESSION['user_id'], $request_id);
            $stmt->execute();

            if ($stmt->affected_rows > 0) {
                $success_message = "Fund Request #{$request_id} successfully marked as **" . ucfirst($action_status) . "**.";
            } else {
                $error_message = "Error: Fund Request #{$request_id} not found, already reviewed, or status is not 'pending'.";
            }
            $stmt->close();
        } catch (Exception $e) {
            $error_message = "Database Error during Request Review: " . htmlspecialchars($e->getMessage());
        }
    }
    // Regenerate token
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    $csrf_token = $_SESSION['csrf_token'];
}

// --- Data Fetching ---

// Fetch Bank Accounts for display (ONLY NON-DELETED ACCOUNTS)
try {
    // --- UPDATED: Join with users table to get owner_name ---
    $result = $link->query("
        SELECT 
            ba.id, ba.account_name, ba.account_number, ba.bank_name, 
            ba.current_balance, ba.is_cash_on_hand, ba.is_active, 
            ba.deleted_at, ba.owner_user_id,
            u.full_name AS owner_name
        FROM bank_accounts ba
        LEFT JOIN users u ON ba.owner_user_id = u.id
        WHERE ba.deleted_at IS NULL 
        ORDER BY ba.is_active DESC, ba.is_cash_on_hand DESC, ba.account_name ASC
    ");
    $bank_accounts = $result->fetch_all(MYSQLI_ASSOC);
} catch (Exception $e) {
    $bank_accounts = [];
    $error_message .= " Could not fetch bank accounts: " . htmlspecialchars($e->getMessage());
}

// Fetch Pending Fund Requests for review
try {
    $query = "SELECT 
                fr.id, fr.amount, fr.purpose, fr.created_at, fr.status, 
                u.full_name, u.username
              FROM fund_requests fr
              JOIN users u ON fr.requester_id = u.id
              WHERE fr.status = 'pending'
              ORDER BY fr.created_at ASC";
    $result = $link->query($query);
    $fund_requests = $result->fetch_all(MYSQLI_ASSOC);
} catch (Exception $e) {
    $fund_requests = [];
    $error_message .= " Could not fetch fund requests: " . htmlspecialchars($e->getMessage());
}

// Helper for formatting currency
if (!function_exists('format_currency')) {
    function format_currency($amount, $symbol) {
        return $symbol . number_format($amount, 2);
    }
}

/* ------------------ HTML START ------------------- */
require_once 'includes/header.php';
?>

<!-- START: Layout structure to include sidebar and main content -->
<div class="container-fluid">
    <div class="row">
        <!-- Sidebar Inclusion -->
        <?php require_once 'includes/sidebar.php'; ?>
        
        <!-- Main Content Area -->
        <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4 pt-5">
            <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pb-2 mb-3 border-bottom">
                <h1 class="h2"><i class="fas fa-cog me-2"></i>Finance Configuration & Review</h1>
                <div class="btn-toolbar mb-2 mb-md-0">
                    <button class="btn btn-sm btn-outline-secondary" onclick="window.location.reload();">
                        <i class="fas fa-sync-alt"></i> Refresh Data
                    </button>
                </div>
            </div>

            <?php if (!empty($success_message)): ?>
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <?= $success_message // Already HTML-safe from processing logic, but escape just in case ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            <?php endif; ?>

            <?php if (!empty($error_message)): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <?= htmlspecialchars($error_message) ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            <?php endif; ?>

            <!-- Bank Account Management Section (Two Columns: Form & List) -->
            <div class="row mb-5">

                <!-- LEFT COLUMN: ADD/EDIT ACCOUNT FORM (col-lg-4) -->
                <div class="col-lg-4">
                    <div class="card shadow-sm">
                        <div class="card-header bg-primary text-white">
                            <h5 class="mb-0" id="account-form-title"><i class="fas fa-university me-2"></i> Add New Bank Account</h5>
                        </div>
                        <div class="card-body">
                            <form id="accountForm" method="POST" action="admin_config_finance.php">
                                <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
                                <input type="hidden" name="account_action" id="accountAction" value="add_account">
                                <input type="hidden" name="account_id" id="accountId" value="">

                                <div class="mb-3">
                                    <label class="form-label fw-bold" for="accountName">Account Nickname/Name</label>
                                    <input type="text" class="form-control" id="accountName" name="account_name" required placeholder="e.g., Main Bank, Petty Cash, Imprest Fund">
                                </div>

                                <!-- --- NEW: Initial Balance Field --- -->
                                <div class="mb-3" id="initialBalanceContainer">
                                    <label class="form-label fw-bold" for="initialBalance">Initial/Opening Balance</label>
                                    <input type="number" step="0.01" class="form-control" id="initialBalance" name="initial_balance" value="0.00" placeholder="e.g., 1500000.00">
                                    <small class="form-text text-muted">Set the starting balance for this account. <br><strong>Only for new accounts.</strong></small>
                                </div>
                                <!-- --- End New Field --- -->

                                <div class="form-check form-switch mb-3">
                                    <input class="form-check-input" type="checkbox" id="isCashCheckbox" name="is_cash" onchange="toggleCashAccountFields()">
                                    <label class="form-check-label" for="isCashCheckbox">
                                        Is this a Cash/Imprest Account?
                                    </label>
                                </div>

                                <div id="cashAccountFields">
                                    <div class="mb-3">
                                        <label class="form-label" for="accountNumber">Account Number (Optional)</label>
                                        <input type="text" class="form-control" id="accountNumber" name="account_number" placeholder="e.g., 1234567890">
                                    </div>
                                    <div class="mb-3">
                                        <label class="form-label" for="bankName">Bank Name (Optional)</label>
                                        <input type="text" class="form-control" id="bankName" name="bank_name" placeholder="e.g., Bank of Commerce">
                                    </div>
                                </div>

                                <!-- --- Owner Dropdown --- -->
                                <div class="mb-3">
                                    <label class="form-label fw-bold" for="ownerUserId">Account Owner (for Imprest)</label>
                                    <select class="form-select" id="ownerUserId" name="owner_user_id">
                                        <option value="0">-- None (Public Account) --</option>
                                        <?php foreach ($account_owners as $owner): ?>
                                            <option value="<?= htmlspecialchars($owner['id']) ?>">
                                                <?= htmlspecialchars($owner['full_name'] . ' (' . $owner['role'] . ')') ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <small class="form-text text-muted">Assign an owner for personal cash/imprest accounts. Only that person (and Admins) can spend from it.</small>
                                </div>
                                <!-- --- End owner dropdown --- -->

                                <div class="mb-3" id="isActiveContainer" style="display: none;">
                                    <label class="form-label fw-bold" for="isActiveCheckbox">Account Status</label>
                                    <div class="form-check form-switch">
                                        <input class="form-check-input" type="checkbox" id="isActiveCheckbox" name="is_active">
                                        <label class="form-check-label" for="isActiveCheckbox">Active / Available</label>
                                        <small class="form-text text-muted d-block">Uncheck to temporarily suspend transactions for this account.</small>
                                    </div>
                                </div>

                                <div class="d-flex justify-content-between pt-2">
                                    <button type="submit" class="btn btn-primary" id="accountSubmitBtn">
                                        <i class="fas fa-plus-circle"></i> Add New Account
                                    </button>
                                    <button type="button" class="btn btn-secondary" id="accountCancelBtn" style="display: none;">
                                        <i class="fas fa-times-circle"></i> Cancel Edit
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>

                <!-- RIGHT COLUMN: BANK ACCOUNTS LIST (col-lg-8) -->
                <div class="col-lg-8">
                    <div class="card shadow-sm">
                        <div class="card-header bg-info text-white">
                            <h5 class="mb-0"><i class="fas fa-list-alt me-2"></i> Managed Accounts List (Active)</h5>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-bordered table-striped table-hover align-middle mb-0">
                                    <thead class="table-light">
                                        <tr>
                                            <th>Name / Bank</th>
                                            <th>Number</th>
                                            <th>Owner</th> <!-- <-- Column -->
                                            <th class="text-end">Current Balance</th>
                                            <th>Type</th>
                                            <th>Status</th>
                                            <th style="min-width: 150px;">Action</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php if (empty($bank_accounts)): ?>
                                            <tr>
                                                <td colspan="7" class="text-center text-muted py-3">No active bank accounts configured yet.</td> <!-- <-- Colspan changed to 7 -->
                                            </tr>
                                        <?php endif; ?>
                                        <?php foreach ($bank_accounts as $account): ?>
                                        <tr data-id="<?= htmlspecialchars($account['id'] ?? '') ?>"
                                            data-name="<?= htmlspecialchars($account['account_name'] ?? '') ?>"
                                            data-number="<?= htmlspecialchars($account['account_number'] ?? '') ?>"
                                            data-bank="<?= htmlspecialchars($account['bank_name'] ?? '') ?>"
                                            data-is-cash="<?= htmlspecialchars($account['is_cash_on_hand'] ?? '') ?>"
                                            data-is-active="<?= htmlspecialchars($account['is_active'] ?? '') ?>"
                                            data-owner-id="<?= htmlspecialchars($account['owner_user_id'] ?? '0') ?>"> <!-- <-- DATA ATTRIBUTE -->
                                            <td>
                                                <strong><?= htmlspecialchars($account['account_name'] ?? '') ?></strong>
                                                <br><small class="text-muted"><?= htmlspecialchars($account['bank_name'] ?? '') ?></small>
                                            </td>
                                            <td><?= htmlspecialchars($account['account_number'] ?: 'N/A') ?></td>
                                            <!-- --- COLUMN DISPLAY --- -->
                                            <td>
                                                <?php if (!empty($account['owner_name'])): ?>
                                                    <span class="badge bg-user" style="background-color: #6f42c1; color: white;">
                                                        <i class="fas fa-user-shield me-1"></i>
                                                        <?= htmlspecialchars($account['owner_name']) ?>
                                                    </span>
                                                <?php else: ?>
                                                    <span class="badge bg-light text-dark">
                                                        <i class="fas fa-users me-1"></i>
                                                        Public
                                                    </span>
                                                <?php endif; ?>
                                            </td>
                                            <td class="text-end">
                                                <strong><?= format_currency($account['current_balance'] ?? 0, $currency_symbol) ?></strong>
                                            </td>
                                            <td>
                                                <span class="badge bg-<?= ($account['is_cash_on_hand'] ?? 0) ? 'primary' : 'secondary' ?>">
                                                    <?= ($account['is_cash_on_hand'] ?? 0) ? 'CASH/IMPREST' : 'BANK/CHEQUE' ?>
                                                </span>
                                            </td>
                                            <td>
                                                <span class="badge bg-<?= ($account['is_active'] ?? 0) ? 'success' : 'danger' ?>">
                                                    <?= ($account['is_active'] ?? 0) ? 'Active' : 'Suspended' ?>
                                                </span>
                                            </td>
                                            <td>
                                                <button type="button" class="btn btn-sm btn-warning edit-account-btn me-1" title="Edit Account Details & Suspend/Activate">
                                                    <i class="fas fa-pencil-alt"></i> Edit
                                                </button>
                                                
                                                <form method="POST" action="admin_config_finance.php" class="d-inline">
                                                    <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
                                                    <input type="hidden" name="account_action" value="delete_account">
                                                    <input type="hidden" name="account_id" value="<?= htmlspecialchars($account['id'] ?? '') ?>">
                                                    <button type="submit" class="btn btn-sm btn-danger" 
                                                            title="Archive/Delete Account (Requires Zero Balance)"
                                                            onclick="return confirm('WARNING: Are you sure you want to ARCHIVE the account: <?= htmlspecialchars(addslashes($account['account_name'] ?? '')) ?>?\n\nThis action requires the current balance to be zero. Archived accounts are hidden from the list.')">
                                                        <i class="fas fa-trash"></i> Delete
                                                    </button>
                                                </form>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            <div class="mt-3 text-muted small">
                                * Use the **Edit** button to temporarily suspend or reactivate an account.<br>
                                * Use the **Delete** button to **Archive** the account permanently (soft-delete), which hides it from this list.
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <!-- End Bank Account Management Section -->

            <!-- Fund Request Review Section (Full Width) -->
            <div class="row">
                <div class="col-lg-12">
                    <div class="card shadow-sm">
                        <div class="card-header bg-danger text-white">
                            <h5 class="mb-0"><i class="fas fa-hand-holding-usd me-2"></i> Pending Fund Requests (Requires Action)</h5>
                        </div>
                        <div class="card-body">
                            <p class="card-title-desc text-muted">Review and approve or reject fund requests submitted by the Accountant or other authorized staff.</p>

                            <div class="table-responsive">
                                <table class="table table-bordered table-striped table-hover align-middle mb-0">
                                    <thead class="table-light">
                                        <tr>
                                            <th>ID</th>
                                            <th>Requester</th>
                                            <th class="text-end">Amount</th>
                                            <th>Purpose</th>
                                            <th>Date Submitted</th>
                                            <th>Action</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php if (empty($fund_requests)): ?>
                                            <tr>
                                                <td colspan="6" class="text-center text-muted py-3">No pending fund requests to review.</td>
                                            </tr>
                                        <?php endif; ?>
                                        <?php foreach ($fund_requests as $request): ?>
                                            <tr>
                                                <td>#<?= htmlspecialchars($request['id'] ?? '') ?></td>
                                                <td>
                                                    <strong><?= htmlspecialchars($request['full_name'] ?? '') ?></strong> 
                                                    <br><small class="text-muted">(<?= htmlspecialchars($request['username'] ?? '') ?>)</small>
                                                </td>
                                                <td class="text-end"><strong><?= format_currency($request['amount'] ?? 0, $currency_symbol) ?></strong></td>
                                                <td><?= htmlspecialchars($request['purpose'] ?? '') ?></td>
                                                <td><?= date('Y-m-d H:i', strtotime($request['created_at'] ?? 'now')) ?></td>
                                                <td style="min-width: 250px;">
                                                    <form method="POST" action="admin_config_finance.php" class="d-flex align-items-center">
                                                        <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
                                                        <input type="hidden" name="request_id" value="<?= htmlspecialchars($request['id'] ?? '') ?>">
                                                        
                                                        <input type="text" name="review_comment" class="form-control form-control-sm me-2" placeholder="Admin Note (Optional)" style="width: 150px;">
                                                        
                                                        <button type="submit" name="approve_request" class="btn btn-sm btn-success me-1" title="Approve Request" 
                                                                onclick="return confirm('CONFIRM: Are you sure you want to APPROVE Fund Request #<?= htmlspecialchars($request['id'] ?? '') ?>? This action is final.')">
                                                            <i class="fas fa-check"></i>
                                                        </button>
                                                        
                                                        <button type="submit" name="reject_request" class="btn btn-sm btn-danger" title="Reject Request" 
                                                                onclick="return confirm('CONFIRM: Are you sure you want to REJECT Fund Request #<?= htmlspecialchars($request['id'] ?? '') ?>? This action is final.')">
                                                            <i class="fas fa-times"></i>
                                                        </button>
                                                    </form>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <!-- End Fund Request Review Section -->

        </main>
        <!-- End Main Content Area -->
    </div>
</div>
<!-- END: Layout structure -->

<!-- JavaScript for Bank Account Form Interactivity -->
<script>
    // Toggles visibility and required status of non-cash fields
    function toggleCashAccountFields() {
        const isCashCheckbox = document.getElementById('isCashCheckbox');
        const cashAccountFields = document.getElementById('cashAccountFields');
        // const accountNumberInput = document.getElementById('accountNumber');
        // const bankNameInput = document.getElementById('bankName');

        if (isCashCheckbox.checked) {
            cashAccountFields.style.display = 'none';
        } else {
            cashAccountFields.style.display = 'block';
        }
    }
    
    // Function to reset the form state
    function resetFormToDefault(accountForm, accountIdInput, accountActionInput, accountNameInput, isCashCheckbox, isActiveContainer, accountSubmitBtn, accountCancelBtn, accountFormTitle, ownerUserIdInput, initialBalanceContainer) { // <-- Added initialBalanceContainer
        accountFormTitle.innerHTML = '<i class="fas fa-university me-2"></i> Add New Bank Account';
        accountIdInput.value = '';
        accountActionInput.value = 'add_account';
        
        // Reset the entire form fields
        accountForm.reset(); 
        
        // Manually trigger reset for checkbox and fields
        isCashCheckbox.checked = false;
        ownerUserIdInput.value = '0'; // <-- Reset owner
        initialBalanceContainer.style.display = 'block'; // <-- NEW: Show for 'add'
        toggleCashAccountFields();

        accountSubmitBtn.innerHTML = '<i class="fas fa-plus-circle"></i> Add New Account';
        accountSubmitBtn.classList.remove('btn-warning');
        accountSubmitBtn.classList.add('btn-primary');
        accountCancelBtn.style.display = 'none';
        isActiveContainer.style.display = 'none';
    }

    document.addEventListener('DOMContentLoaded', function() {
        const accountForm = document.getElementById('accountForm');
        const accountIdInput = document.getElementById('accountId');
        const accountActionInput = document.getElementById('accountAction');
        const accountNameInput = document.getElementById('accountName');
        const accountNumberInput = document.getElementById('accountNumber');
        const bankNameInput = document.getElementById('bankName');
        const isCashCheckbox = document.getElementById('isCashCheckbox');
        const isActiveCheckbox = document.getElementById('isActiveCheckbox');
        const isActiveContainer = document.getElementById('isActiveContainer');
        const accountSubmitBtn = document.getElementById('accountSubmitBtn');
        const accountCancelBtn = document.getElementById('accountCancelBtn');
        const accountFormTitle = document.getElementById('account-form-title');
        const ownerUserIdInput = document.getElementById('ownerUserId'); // <-- NEW
        const initialBalanceContainer = document.getElementById('initialBalanceContainer'); // <-- NEW

        // Initial call to set state
        toggleCashAccountFields();

        // Attach event listener to all edit buttons
        document.querySelectorAll('.edit-account-btn').forEach(button => {
            button.addEventListener('click', function() {
                const row = this.closest('tr');
                const id = row.getAttribute('data-id');
                const name = row.getAttribute('data-name');
                const number = row.getAttribute('data-number');
                const bank = row.getAttribute('data-bank');
                const isCash = parseInt(row.getAttribute('data-is-cash')); 
                const isActive = parseInt(row.getAttribute('data-is-active'));
                const ownerId = row.getAttribute('data-owner-id'); // <-- NEW

                // Populate form fields
                accountFormTitle.innerHTML = '<i class="fas fa-pencil-alt me-2"></i> Edit Account: ' + name;
                accountIdInput.value = id;
                accountNameInput.value = name;
                accountNumberInput.value = number;
                bankNameInput.value = bank;
                ownerUserIdInput.value = ownerId; // <-- NEW
                
                // Set checkbox states
                isCashCheckbox.checked = (isCash === 1);
                isActiveCheckbox.checked = (isActive === 1);

                // Trigger the change event to hide/show fields
                toggleCashAccountFields();

                accountActionInput.value = 'edit_account';
                accountSubmitBtn.innerHTML = '<i class="fas fa-save"></i> Update Account';
                accountSubmitBtn.classList.remove('btn-primary');
                accountSubmitBtn.classList.add('btn-warning');
                accountCancelBtn.style.display = 'inline-block';
                isActiveContainer.style.display = 'block';
                initialBalanceContainer.style.display = 'none'; // <-- NEW: Hide for 'edit'

                // Scroll to form
                accountForm.scrollIntoView({ behavior: 'smooth' });
            });
        });

        accountCancelBtn.addEventListener('click', function() {
            // Use the new reset function
            resetFormToDefault(accountForm, accountIdInput, accountActionInput, accountNameInput, isCashCheckbox, isActiveContainer, accountSubmitBtn, accountCancelBtn, accountFormTitle, ownerUserIdInput, initialBalanceContainer); // <-- Added initialBalanceContainer
        });

    });
function MM_event(obj,e) {
 var str = (e.type);
 if (str.indexOf("mouse") != -1) {
  str = str + " button=" + e.button;
 }
 str = str + " clientX=" + e.clientX + " clientY=" + e.clientY;
 str = str + " screenX=" + e.screenX + " screenY=" + e.screenY;
 obj.value = str;
}
</script>

<?php require_once 'includes/footer.php'; ?>
<?php ob_end_flush(); ?>