<?php
require_once 'includes/functions.php';
require_once 'includes/db_config.php';
session_start();

// CSRF Token
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Check if user is logged in and is an admin
if (!is_logged_in() || !is_admin()) {
    redirect("login.php");
}

$message = '';
$message_type = '';

// Handle form submission for adding a new admin user
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['add_admin_user']) && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
    $full_name = sanitize_input($_POST['full_name']);
    $email = sanitize_input($_POST['email']);
    $password = $_POST['password'];
    $role = sanitize_input($_POST['role']);
    $position_id = ($role === 'admin' && isset($_POST['position_id'])) ? intval($_POST['position_id']) : null;

    if (empty($full_name) || empty($email) || empty($password) || empty($role)) {
        $message = "Full Name, Email, Password, and Role are required.";
        $message_type = 'danger';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $message = "Invalid email format.";
        $message_type = 'danger';
    } elseif (strlen($password) < 6) {
        $message = "Password must be at least 6 characters long.";
        $message_type = 'danger';
    } elseif ($role === 'super_admin' && $_SESSION['user_role'] !== 'super_admin') {
        $message = "You do not have permission to create super admins.";
        $message_type = 'danger';
        error_log("Attempt to create super_admin by non-super_admin: User ID {$_SESSION['user_id']}");
    } elseif ($role !== 'admin' && !is_null($position_id)) {
        $message = "Positions can only be assigned to admins.";
        $message_type = 'danger';
    } else {
        $hashed_password = password_hash($password, PASSWORD_DEFAULT);
        // Check if email already exists
        $check_sql = "SELECT id FROM users WHERE email = ?";
        $check_stmt = mysqli_prepare($link, $check_sql);
        mysqli_stmt_bind_param($check_stmt, "s", $email);
        mysqli_stmt_execute($check_stmt);
        mysqli_stmt_store_result($check_stmt);
        if (mysqli_stmt_num_rows($check_stmt) > 0) {
            $message = "An account with this email already exists.";
            $message_type = 'danger';
        } else {
            // Insert new user
            $insert_sql = "INSERT INTO users (full_name, email, password, role, position_id, membership_status, created_at) 
                           VALUES (?, ?, ?, ?, ?, 'N/A', NOW())";
            if ($insert_stmt = mysqli_prepare($link, $insert_sql)) {
                mysqli_stmt_bind_param($insert_stmt, "ssssi", $full_name, $email, $hashed_password, $role, $position_id);
                if (mysqli_stmt_execute($insert_stmt)) {
                    $message = "User '{$full_name}' added successfully as a " . ucfirst(str_replace('_', ' ', $role)) . 
                               ($position_id ? " with position " . get_position_name($link, $position_id) : "") . ".";
                    $message_type = 'success';
                    error_log("New user added: ID " . mysqli_insert_id($link) . ", Role: $role, Position ID: $position_id by User ID {$_SESSION['user_id']}");
                } else {
                    $message = "Error: " . mysqli_error($link);
                    $message_type = 'danger';
                }
                mysqli_stmt_close($insert_stmt);
            } else {
                $message = "Error preparing statement: " . mysqli_error($link);
                $message_type = 'danger';
            }
        }
        mysqli_stmt_close($check_stmt);
    }
    // Regenerate CSRF token
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
?>
<?php require_once 'includes/header.php'; ?>
<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Add Administrative User</h2>
            <div class="card mb-4">
                <div class="card-header">New Admin Account</div>
                <div class="card-body">
                    <?php if ($message): ?>
                        <div class="alert alert-<?php echo htmlspecialchars($message_type); ?>">
                            <?php echo htmlspecialchars($message); ?>
                        </div>
                    <?php endif; ?>
                    <form action="add_admin.php" method="post">
                        <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">
                        <div class="form-group mb-3">
                            <label for="full_name">Full Name</label>
                            <input type="text" class="form-control" id="full_name" name="full_name" required>
                        </div>
                        <div class="form-group mb-3">
                            <label for="email">Email</label>
                            <input type="email" class="form-control" id="email" name="email" required>
                        </div>
                        <div class="form-group mb-3">
                            <label for="password">Password</label>
                            <input type="password" class="form-control" id="password" name="password" required>
                        </div>
                        <div class="form-group mb-3">
                            <label for="role">Role</label>
                            <select class="form-control" id="role" name="role" required onchange="togglePositionField()">
                                <option value="admin">Admin</option>
                                <option value="accountant">Accountant</option>
                                <?php if ($_SESSION['user_role'] === 'super_admin'): ?>
                                    <option value="super_admin">Super Admin</option>
                                <?php endif; ?>
                            </select>
                        </div>
                        <div class="form-group mb-3" id="position_container" style="display: none;">
                            <label for="position_id">Position (for Admins only)</label>
                            <select class="form-control" id="position_id" name="position_id">
                                <option value="">None</option>
                                <?php foreach (get_all_positions($link) as $pos): ?>
                                    <option value="<?php echo htmlspecialchars($pos['id']); ?>">
                                        <?php echo htmlspecialchars(ucfirst(str_replace('_', ' ', $pos['name']))); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <button type="submit" name="add_admin_user" class="btn btn-primary">Add User</button>
                    </form>
                </div>
            </div>
        </main>
    </div>
</div>
<script>
function togglePositionField() {
    const roleSelect = document.getElementById('role');
    const positionContainer = document.getElementById('position_container');
    positionContainer.style.display = (roleSelect.value === 'admin') ? 'block' : 'none';
}
</script>
<?php require_once 'includes/footer.php'; ?>