<?php
require_once 'includes/functions.php';
require_once 'includes/db_config.php';

// Check if user is logged in and is an admin
if (!is_logged_in() || !is_admin()) {
    redirect("login.php");
}

$member = null;
$form_data_array = null;
$form_structure_array = null;

// Handle CSV export request
if (isset($_GET['export']) && $_GET['export'] == 'csv') {
    if (isset($_GET['id'])) {
        $member_id_numeric = intval($_GET['id']);
        
        $sql_export = "SELECT 
                    u.member_id, 
                    u.email, 
                    u.membership_status,
                    u.created_at,
                    fd.data AS form_data,
                    f.form_data AS form_structure
                FROM users u
                LEFT JOIN form_data fd ON u.id = fd.user_id
                LEFT JOIN forms f ON fd.form_id = f.id
                WHERE u.id = ? AND u.role = 'member'";
        
        if ($stmt_export = mysqli_prepare($link, $sql_export)) {
            mysqli_stmt_bind_param($stmt_export, "i", $member_id_numeric);
            mysqli_stmt_execute($stmt_export);
            $result_export = mysqli_stmt_get_result($stmt_export);
            $member_export = mysqli_fetch_assoc($result_export);

            if ($member_export) {
                // Set headers for CSV download
                header('Content-Type: text/csv; charset=utf-8');
                header('Content-Disposition: attachment; filename="' . $member_export['member_id'] . '_data.csv"');

                $output = fopen('php://output', 'w');
                
                // Write headers to CSV
                fputcsv($output, ['Field', 'Value']);

                // Write core member details
                fputcsv($output, ['Member ID', $member_export['member_id']]);
                fputcsv($output, ['Email', $member_export['email']]);
                fputcsv($output, ['Status', $member_export['membership_status']]);
                fputcsv($output, ['Registration Date', $member_export['created_at']]);

                // Write custom form data
                $form_data_array_export = json_decode($member_export['form_data'], true);
                $form_structure_array_export = json_decode($member_export['form_structure'], true);
                
                if ($form_data_array_export && $form_structure_array_export && isset($form_structure_array_export['fields'])) {
                    // Create a lookup map for labels
                    $field_labels_export = [];
                    foreach ($form_structure_array_export['fields'] as $field) {
                        $field_labels_export[$field['name']] = $field['label'];
                    }

                    foreach ($form_data_array_export as $name => $value) {
                        $label = isset($field_labels_export[$name]) ? $field_labels_export[$name] : $name;
                        $formatted_value = is_array($value) ? implode(', ', array_map('htmlspecialchars', $value)) : htmlspecialchars($value);
                        fputcsv($output, [$label, $formatted_value]);
                    }
                }
                
                fclose($output);
                exit();
            }
        }
    }
}

// Fetch member data for display on the page
if (isset($_GET['id'])) {
    $member_id = intval($_GET['id']);

    // SQL query to get user data and custom form data
    $sql = "SELECT 
                u.id, 
                u.member_id,
                u.full_name, 
                u.email, 
                u.membership_status,
                u.created_at,
                fd.data AS form_data,
                f.form_data AS form_structure
            FROM users u
            LEFT JOIN form_data fd ON u.id = fd.user_id
            LEFT JOIN forms f ON fd.form_id = f.id
            WHERE u.id = ? AND u.role = 'member'";
    
    if ($stmt = mysqli_prepare($link, $sql)) {
        mysqli_stmt_bind_param($stmt, "i", $member_id);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
        
        if (mysqli_num_rows($result) > 0) {
            $row = mysqli_fetch_assoc($result);
            $member = $row;
            // Decode the submitted form data
            $form_data_array = json_decode($row['form_data'], true);
            // Decode the original form structure
            $form_structure_array = json_decode($row['form_structure'], true);
        }
        mysqli_stmt_close($stmt);
    }
}
?>

<?php require_once 'includes/header.php'; ?>
<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Member Details</h2>
            <div class="member-details-container">
                <?php if ($member): ?>
                    <div class="member-actions mb-4">
                        <a href="?id=<?php echo htmlspecialchars($_GET['id']); ?>&export=csv" class="btn btn-primary">Export to CSV</a>
                        <button onclick="window.print()" class="btn btn-secondary">Print</button>
                    </div>
                    <div class="card">
                        <div class="card-header">
                            <h3><?php echo htmlspecialchars($member['full_name']); ?></h3>
                        </div>
                        <div class="card-body">
                            <p><strong>Member ID:</strong> <?php echo htmlspecialchars($member['member_id']); ?></p>
                            <p><strong>Email:</strong> <?php echo htmlspecialchars($member['email']); ?></p>
                            <p><strong>Status:</strong> <?php echo htmlspecialchars(ucfirst($member['membership_status'])); ?></p>
                            <p><strong>Registration Date:</strong> <?php echo htmlspecialchars($member['created_at']); ?></p>
                        </div>
                    </div>
                    
                    <?php if ($form_data_array): ?>
                        <div class="card mt-4">
                            <div class="card-header">
                                <h3>Registration Form Data</h3>
                            </div>
                            <div class="card-body">
                                <table class="table table-striped">
                                    <tbody>
                                        <?php 
                                            // Create a lookup map from the form structure
                                            $field_labels = [];
                                            if ($form_structure_array && isset($form_structure_array['fields'])) {
                                                foreach ($form_structure_array['fields'] as $field) {
                                                    $field_labels[$field['name']] = $field['label'];
                                                }
                                            }
                                            
                                            foreach ($form_data_array as $name => $value):
                                        ?>
                                            <tr>
                                                <th><?php 
                                                    // Use the label from the form structure if available, otherwise fallback
                                                    if (isset($field_labels[$name])) {
                                                        echo htmlspecialchars($field_labels[$name]);
                                                    } else {
                                                        // Fallback: Clean up the dynamic name if label is not found
                                                        $clean_label = preg_replace('/[a-zA-Z]+\d+/', '', $name);
                                                        $clean_label = ucwords(str_replace(['_', '-'], ' ', $clean_label));
                                                        echo htmlspecialchars($clean_label);
                                                    }
                                                ?></th>
                                                <td>
                                                    <?php 
                                                        if (is_array($value)) {
                                                            echo implode(', ', array_map('htmlspecialchars', $value));
                                                        } else {
                                                            echo htmlspecialchars($value);
                                                        }
                                                    ?>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    <?php endif; ?>
                    
                    <a href="member_management.php" class="btn btn-secondary mt-3">Back to Manage Members</a>
                <?php else: ?>
                    <div class="alert alert-warning">
                        Member not found.
                    </div>
                    <a href="member_management.php" class="btn btn-secondary mt-3">Back to Manage Members</a>
                <?php endif; ?>
            </div>
        </main>
    </div>
</div>
<?php require_once 'includes/footer.php'; ?>