<?php
// record_payments.php
ob_start();
session_start();
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
if (function_exists('opcache_reset')) { opcache_reset(); }

require_once 'includes/functions.php';
require_once 'includes/db_config.php';

/* ---------- SESSION / PERMISSIONS ---------- */
$timeout = 1800;
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity']) > $timeout) {
    session_unset(); session_destroy();
    redirect("login.php?timeout=1");
}
$_SESSION['last_activity'] = time();

if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
if (!is_logged_in() || !user_has_permission($link, 'view_payments') ||
    !in_array($_SESSION['user_role'], ['admin', 'accountant'])) {
    redirect("login.php"); exit;
}

/* ---------- SETTINGS ---------- */
$currency_symbol = get_setting('currency_symbol', $link) ?? '₦';
$token_cost      = (int)get_setting('tokens_per_payment', $link) ?: 1;
$deduct_per_member = (int)get_setting('deduct_per_member', $link) ?? 0;

/* ---------- ORGANIZATION BALANCE FUNCTION (ALL ADMINS + ACCOUNTANTS) ---------- */
function get_organization_token_balance($link) {
    $sql = "SELECT COALESCE(SUM(tb.balance), 0) AS total
            FROM token_balances tb
            JOIN users u ON tb.user_id = u.id
            WHERE u.role IN ('admin', 'accountant')
              AND u.deleted_at IS NULL
              AND tb.balance > 0";
    $result = mysqli_query($link, $sql);
    if ($result && $row = mysqli_fetch_assoc($result)) {
        mysqli_free_result($result);
        return (int)$row['total'];
    }
    return 0;
}

/* ---------- INITIAL BALANCES ---------- */
$current_user_balance      = get_user_token_balance($link, $_SESSION['user_id']);
$organization_total_tokens = get_organization_token_balance($link);

/* ---------- FORM HANDLING ---------- */
$message = $message_type = '';
if ($_SERVER["REQUEST_METHOD"] == "POST"
    && isset($_POST['log_payment'])
    && isset($_POST['csrf_token'])
    && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {

    $user_ids       = $_POST['user_ids'] ?? [];
    $amount         = floatval($_POST['amount'] ?? 0);
    $obligation_id  = intval($_POST['obligation_id'] ?? 0);
    $payment_type   = sanitize_input($_POST['payment_type'] ?? '');
    $description    = sanitize_input($_POST['description'] ?? '');
    $status         = 'Paid';
    $current_user_id = $_SESSION['user_id'];

    $num_logs        = $deduct_per_member ? count($user_ids) : 1;
    $required_tokens = $token_cost * $num_logs;

    if ($current_user_balance < $required_tokens) {
        $message = "Insufficient tokens. You need $required_tokens token".($required_tokens>1?'s':'')
                 .", but you have $current_user_balance. <a href='token_store.php'>Buy / request more</a>.";
        $message_type = 'danger';
    } else {
        $valid_obligation = false;
        $debt_amount = 0;
        $is_freewill = false;
        if ($obligation_id > 0) {
            $sql_check = "SELECT po.id,
                             (COALESCE(po.required_amount,0)-COALESCE(SUM(p.amount),0)) AS debt,
                             pt.payment_category
                      FROM payment_obligations po
                      LEFT JOIN payments p ON p.obligation_id=po.id AND p.payment_status='Paid'
                      JOIN payment_types pt ON po.payment_type=pt.type_name AND pt.deleted_at IS NULL
                      WHERE po.id=? AND po.user_id IN (".implode(',',array_fill(0,count($user_ids),'?')).")
                        AND po.deleted_at IS NULL
                      GROUP BY po.id,pt.payment_category
                      HAVING debt>0";
            $stmt_check = mysqli_prepare($link, $sql_check);
            $bind = array_merge([$obligation_id], array_map('intval',$user_ids));
            $types = 'i'.str_repeat('i',count($user_ids));
            mysqli_stmt_bind_param($stmt_check, $types, ...$bind);
            mysqli_stmt_execute($stmt_check);
            $res = mysqli_stmt_get_result($stmt_check);
            if ($row = mysqli_fetch_assoc($res)) {
                $valid_obligation = true;
                $debt_amount = $row['debt'];
                $is_freewill = $row['payment_category'] === 'freewill';
            }
            mysqli_stmt_close($stmt_check);
        }

        if (!empty($user_ids) && $amount>0 && $valid_obligation && ($is_freewill || $amount<=$debt_amount)) {
            mysqli_begin_transaction($link);
            $ok = true;

            $tokens_used = deduct_tokens($link, $current_user_id, 'payment', $required_tokens);
            if ($tokens_used === false) { $ok = false; $message = "Token deduction failed."; $message_type='danger'; }

            if ($ok) {
                $sql = "INSERT INTO payments
                        (user_id,payment_type,amount,description,payment_status,obligation_id,payment_date,created_at)
                        VALUES (?,?,?,?,?, ?,NOW(),NOW())";
                $stmt = mysqli_prepare($link, $sql);
                foreach ($user_ids as $uid) {
                    $uid = intval($uid);
                    mysqli_stmt_bind_param($stmt, "isdssi",
                        $uid,$payment_type,$amount,$description,$status,$obligation_id);
                    if (!mysqli_stmt_execute($stmt)) { $ok = false; $message="Insert error: ".mysqli_stmt_error($stmt); $message_type='danger'; break; }
                }
                mysqli_stmt_close($stmt);
            }

            if ($ok) {
                mysqli_commit($link);
                $message = "Payment logged! Used $tokens_used token".($tokens_used>1?'s':'').".";
                $message_type = 'success';

                /* ---- REFRESH BALANCES ---- */
                $current_user_balance      = get_user_token_balance($link, $_SESSION['user_id']);
                $organization_total_tokens = get_organization_token_balance($link);
            } else {
                mysqli_rollback($link);
                if (!$message) $message = "Transaction rolled back.";
                $message_type = 'danger';
            }
        } else {
            $message = "Invalid data – check members, amount, obligation, or debt limit.";
            $message_type = 'danger';
        }
    }
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
} elseif ($_SERVER["REQUEST_METHOD"] == "POST") {
    $message = "Invalid CSRF token.";
    $message_type = 'danger';
}

/* ---------- MEMBERS LIST ---------- */
$members = [];
$res = mysqli_query($link,
    "SELECT id,full_name,member_id FROM users
     WHERE role='member' AND (deleted_at IS NULL OR deleted_at>NOW())
     ORDER BY full_name");
while ($row = mysqli_fetch_assoc($res)) $members[] = $row;
mysqli_free_result($res);
?>
<?php require_once 'includes/header.php'; ?>
<script>document.title='Record Payments - <?=htmlspecialchars(get_setting('system_name',$link)??'Club Portal')?>';</script>
<style>.main-content{padding:20px;}</style>

<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Record Payments</h2>
            <?php if ($message): ?>
                <div class="alert alert-<?=$message_type?>" role="alert"><?=$message?></div>
            <?php endif; ?>

            <div class="card mb-4">
                <div class="card-header">Log New Payment</div>
                <div class="card-body">
                    <p><strong>Your Token Balance:</strong> <?=$current_user_balance?> tokens</p>
                    <p><strong>Organization Token Balance:</strong> <?=$organization_total_tokens?> tokens</p>

                    <form action="record_payments.php" method="post">
                        <input type="hidden" name="log_payment" value="1">
                        <input type="hidden" name="csrf_token" value="<?=htmlspecialchars($_SESSION['csrf_token'])?>">

                        <div class="mb-3">
                            <label class="form-label">Select Member(s)</label>
                            <select name="user_ids[]" id="user_ids" class="form-select" multiple size="10" required>
                                <?php foreach ($members as $m): ?>
                                    <option value="<?=htmlspecialchars($m['id'])?>">
                                        <?=htmlspecialchars($m['full_name'].' (ID: '.$m['member_id'].')')?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <small class="form-text text-muted">
                                Hold Ctrl/Cmd to select many. <?=$token_cost?> token<?=$token_cost>1?'s':''?> per log
                                <?= $deduct_per_member ? ' (per member)' : '' ?>.
                            </small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Select Obligation</label>
                            <select name="obligation_id" id="obligation_id" class="form-select" required onchange="updatePaymentType()">
                                <option value="">Select an Obligation</option>
                                <?php
                                $sql_ob = "SELECT po.id, po.payment_type,
                                          (COALESCE(po.required_amount,0)-COALESCE(SUM(p.amount),0)) AS debt,
                                          pt.payment_category
                                   FROM payment_obligations po
                                   LEFT JOIN payments p ON p.obligation_id=po.id AND p.payment_status='Paid'
                                   JOIN payment_types pt ON po.payment_type=pt.type_name AND pt.deleted_at IS NULL
                                   WHERE po.deleted_at IS NULL
                                   GROUP BY po.id,po.payment_type,pt.payment_category
                                   HAVING debt>0";
                                $res_ob = mysqli_query($link, $sql_ob);
                                while ($ob = mysqli_fetch_assoc($res_ob)) {
                                    $suffix = $ob['payment_category']==='freewill' ? ' (Freewill)' : '';
                                    echo "<option value='{$ob['id']}' data-payment-type='{$ob['payment_type']}'>"
                                        .$ob['payment_type'].$suffix." (".$currency_symbol.number_format($ob['debt'],2).")</option>";
                                }
                                mysqli_free_result($res_ob);
                                ?>
                            </select>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Payment Type</label>
                            <input type="text" name="payment_type" id="payment_type" class="form-control" readonly>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Amount (<?=$currency_symbol?>)</label>
                            <input type="number" name="amount" id="amount_payment" class="form-control" step="0.01" min="0.01" required>
                            <small class="form-text text-muted">Freewill donations accept any positive amount.</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Description (optional)</label>
                            <textarea name="description" id="description_payment" class="form-control" rows="3"></textarea>
                        </div>

                        <button type="submit" class="btn btn-primary">
                            Log Payment (<?=$token_cost?> token<?=$token_cost>1?'s':''?>)
                        </button>
                    </form>
                </div>
            </div>
        </main>
    </div>
</div>

<script>
function updatePaymentType() {
    const sel = document.getElementById('obligation_id');
    const pt  = document.getElementById('payment_type');
    const opt = sel.options[sel.selectedIndex];
    pt.value = opt && opt.dataset.paymentType ? opt.dataset.paymentType : '';
}
function updateObligations() {
    const userSel = document.getElementById('user_ids');
    const oblSel  = document.getElementById('obligation_id');
    const users   = Array.from(userSel.selectedOptions).map(o=>o.value);
    if (!users.length) { oblSel.innerHTML='<option value="">Select an Obligation</option>'; return; }

    fetch('api/get_obligations.php',{
        method:'POST',
        headers:{'Content-Type':'application/json'},
        body:JSON.stringify({user_ids:users})
    })
    .then(r=>r.json())
    .then(d=>{
        oblSel.innerHTML='<option value="">Select an Obligation</option>';
        d.obligations.forEach(o=>{
            const opt=document.createElement('option');
            opt.value=o.id;
            const suf=o.payment_category==='freewill'?' (Freewill)':'';
            opt.text=`${o.payment_type}${suf} (<?=$currency_symbol?>${parseFloat(o.debt).toFixed(2)})`;
            opt.dataset.paymentType=o.payment_type;
            oblSel.appendChild(opt);
        });
        updatePaymentType();
    })
    .catch(()=>{ oblSel.innerHTML='<option value="">Error loading</option>'; });
}
document.addEventListener('DOMContentLoaded',()=>{ 
    document.getElementById('user_ids').addEventListener('change',updateObligations);
    updateObligations();
});
</script>

<?php require_once 'includes/footer.php'; ?>