<?php
// record_benefits.php
ob_start();
session_start();
ini_set('display_errors',1);
ini_set('display_startup_errors',1);
error_reporting(E_ALL);
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
if (function_exists('opcache_reset')) { opcache_reset(); }

require_once 'includes/functions.php';
require_once 'includes/db_config.php';

/* ---------- SESSION / PERMISSIONS ---------- */
$timeout = 1800;
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity']) > $timeout) {
    session_unset(); session_destroy();
    redirect("login.php?timeout=1");
}
$_SESSION['last_activity'] = time();

if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
if (!is_logged_in() || !user_has_permission($link, 'view_benefits') ||
    !in_array($_SESSION['user_role'], ['admin','accountant'])) {
    redirect("login.php"); exit;
}

/* ---------- SETTINGS ---------- */
$currency_symbol = get_setting('currency_symbol',$link) ?? '₦';
$token_cost      = (int)get_setting('tokens_per_benefit',$link) ?: 1;
$deduct_per_member = (int)get_setting('deduct_per_member',$link) ?? 0;

/* ---------- ORGANIZATION BALANCE FUNCTION ---------- */
function get_organization_token_balance($link) {
    $sql = "SELECT COALESCE(SUM(tb.balance), 0) AS total
            FROM token_balances tb
            JOIN users u ON tb.user_id = u.id
            WHERE u.role IN ('admin', 'accountant')
              AND u.deleted_at IS NULL
              AND tb.balance > 0";
    $result = mysqli_query($link, $sql);
    if ($result && $row = mysqli_fetch_assoc($result)) {
        mysqli_free_result($result);
        return (int)$row['total'];
    }
    return 0;
}

/* ---------- INITIAL BALANCES ---------- */
$current_user_balance      = get_user_token_balance($link, $_SESSION['user_id']);
$organization_total_tokens = get_organization_token_balance($link);

/* ---------- FORM HANDLING ---------- */
$message = $message_type = '';
if ($_SERVER['REQUEST_METHOD']==='POST'
    && isset($_POST['log_benefit'])
    && isset($_POST['csrf_token'])
    && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {

    $member_ids      = $_POST['member_ids'] ?? [];
    $amount          = floatval($_POST['amount'] ?? 0);
    $benefit_type_id = intval($_POST['benefit_type_id'] ?? 0);
    $benefit_date    = $_POST['benefit_date'] ?? date('Y-m-d');
    $benefit_status  = sanitize_input($_POST['benefit_status'] ?? 'Approved');
    $current_user_id = $_SESSION['user_id'];

    $num_logs        = $deduct_per_member ? count($member_ids) : 1;
    $required_tokens = $token_cost * $num_logs;

    if ($current_user_balance < $required_tokens) {
        $message = "Insufficient tokens. Need $required_tokens token".($required_tokens>1?'s':'')
                 .", you have $current_user_balance. <a href='token_store.php'>Buy / request more</a>.";
        $message_type = 'danger';
        goto end_form;
    }
    if (empty($member_ids) || $amount<=0 || $benefit_type_id<=0 || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $benefit_date)) {
        $message = "Please fill all required fields correctly.";
        $message_type = 'danger';
        goto end_form;
    }

    $stmt = mysqli_prepare($link,"SELECT type_name FROM benefit_types WHERE id=? AND deleted_at IS NULL");
    mysqli_stmt_bind_param($stmt,"i",$benefit_type_id);
    mysqli_stmt_execute($stmt);
    $res = mysqli_stmt_get_result($stmt);
    if (!($row = mysqli_fetch_assoc($res))) {
        $message = "Invalid benefit type."; $message_type='danger';
        mysqli_stmt_close($stmt);
        goto end_form;
    }
    $benefit_type = $row['type_name'];
    mysqli_stmt_close($stmt);

    mysqli_begin_transaction($link);
    $ok = true;

    $deducted = deduct_tokens($link,$current_user_id,'benefit',$required_tokens);
    if ($deducted===false) { $ok=false; $message="Token deduction failed."; $message_type='danger'; }

    if ($ok) {
        $sql = "INSERT INTO member_benefits
                (user_id,member_id,benefit_type_id,benefit_type,amount,benefit_date,benefit_status,created_at)
                VALUES (?,?,?,?,?,?,?,NOW())";
        $stmt = mysqli_prepare($link,$sql);
        foreach ($member_ids as $mid) {
            $mid = (int)$mid;
            $chk = mysqli_prepare($link,"SELECT 1 FROM users WHERE id=? AND role='member' AND (deleted_at IS NULL OR deleted_at>NOW())");
            mysqli_stmt_bind_param($chk,"i",$mid);
            mysqli_stmt_execute($chk);
            mysqli_stmt_store_result($chk);
            if (mysqli_stmt_num_rows($chk)===0) {
                $ok=false; $message="Invalid member ID $mid."; $message_type='danger';
                mysqli_stmt_close($chk); break;
            }
            mysqli_stmt_close($chk);

            mysqli_stmt_bind_param($stmt,"iissdss",
                $current_user_id,$mid,$benefit_type_id,$benefit_type,$amount,$benefit_date,$benefit_status);
            if (!mysqli_stmt_execute($stmt)) {
                $ok=false; $message="Insert error: ".mysqli_stmt_error($stmt); $message_type='danger';
                break;
            }
        }
        mysqli_stmt_close($stmt);
    }

    if ($ok) {
        mysqli_commit($link);
        $message = "Benefit logged! Used $deducted token".($deducted>1?'s':'').".";
        $message_type = 'success';

        /* ---- REFRESH BALANCES ---- */
        $current_user_balance      = get_user_token_balance($link, $_SESSION['user_id']);
        $organization_total_tokens = get_organization_token_balance($link);
    } else {
        mysqli_rollback($link);
        if (!$message) $message = "Transaction rolled back.";
        $message_type = 'danger';
    }

    end_form:
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

/* ---------- SELECT DATA ---------- */
$members = $benefit_types = [];
$res = mysqli_query($link,
    "SELECT id,full_name,member_id FROM users
     WHERE role='member' AND (deleted_at IS NULL OR deleted_at>NOW())
     ORDER BY full_name");
while ($row = mysqli_fetch_assoc($res)) $members[] = $row;
mysqli_free_result($res);

$res = mysqli_query($link,
    "SELECT id,type_name FROM benefit_types WHERE deleted_at IS NULL ORDER BY type_name");
while ($row = mysqli_fetch_assoc($res)) $benefit_types[] = $row;
mysqli_free_result($res);
?>
<?php require_once 'includes/header.php'; ?>
<script>document.title='Record Benefits - <?=htmlspecialchars(get_setting('system_name',$link)??'Club Portal')?>';</script>
<style>.main-content{padding:20px;}</style>

<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Record Benefits</h2>
            <?php if ($message): ?>
                <div class="alert alert-<?=$message_type?>" role="alert"><?=$message?></div>
            <?php endif; ?>

            <div class="card mb-4">
                <div class="card-header">Log New Benefit</div>
                <div class="card-body">
                    <p><strong>Your Token Balance:</strong> <?=$current_user_balance?> tokens</p>
                    <p><strong>Organization Token Balance:</strong> <?=$organization_total_tokens?> tokens</p>

                    <form action="record_benefits.php" method="post">
                        <input type="hidden" name="log_benefit" value="1">
                        <input type="hidden" name="csrf_token" value="<?=htmlspecialchars($_SESSION['csrf_token'])?>">

                        <div class="mb-3">
                            <label class="form-label">Member(s)</label>
                            <select name="member_ids[]" class="form-select" multiple size="10" required>
                                <?php foreach ($members as $m): ?>
                                    <option value="<?=htmlspecialchars($m['id'])?>">
                                        <?=htmlspecialchars($m['full_name'].' (ID: '.$m['member_id'].')')?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <small class="form-text text-muted">
                                Hold Ctrl/Cmd to select many. <?=$token_cost?> token<?=$token_cost>1?'s':''?> per log
                                <?= $deduct_per_member ? ' (per member)' : '' ?>.
                            </small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Benefit Type</label>
                            <select name="benefit_type_id" class="form-select" required>
                                <option value="">-- select --</option>
                                <?php foreach ($benefit_types as $bt): ?>
                                    <option value="<?=htmlspecialchars($bt['id'])?>"><?=htmlspecialchars($bt['type_name'])?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Amount (<?=$currency_symbol?>)</label>
                            <input type="number" name="amount" class="form-control" step="0.01" min="0.01" required>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Benefit Date</label>
                            <input type="date" name="benefit_date" class="form-control" value="<?=date('Y-m-d')?>" required>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Status</label>
                            <select name="benefit_status" class="form-select" required>
                                <option value="Approved">Approved</option>
                                <option value="Pending">Pending</option>
                                <option value="Rejected">Rejected</option>
                            </select>
                        </div>

                        <button type="submit" class="btn btn-primary">
                            Log Benefit (<?=$token_cost?> token<?=$token_cost>1?'s':''?>)
                        </button>
                    </form>
                </div>
            </div>
        </main>
    </div>
</div>

<?php require_once 'includes/footer.php'; ?>