<?php
// members/export.php
// Handles CSV & PDF export for members and executives
require_once '../includes/functions.php';
require_once '../includes/db_config.php';
require_once '../vendor/tcpdf/tcpdf.php';

if (!is_logged_in() || !is_admin()) exit;

// === GET PARAMETERS ===
$export   = $_GET['export'] ?? 'csv';      // csv or pdf
$scope    = $_GET['scope'] ?? 'members';   // members or executives
$search   = $_GET['search'] ?? '';

// === BUILD WHERE CLAUSE ===
$where = "WHERE (u.role IN ('member','pending')) AND (u.deleted_at IS NULL OR u.deleted_at > NOW())";
$params = []; $types = '';

if ($scope === 'executives') {
    $where = "WHERE u.position IS NOT NULL AND u.position != '' AND (u.deleted_at IS NULL OR u.deleted_at > NOW())";
}

if ($search) {
    $where .= " AND (u.member_id LIKE ? OR u.full_name LIKE ? OR u.email LIKE ? OR u.phone LIKE ?)";
    $like = "%$search%";
    $params = array_fill(0, 4, $like);
    $types = 'ssss';
}

// === FETCH DATA ===
$sql = "SELECT 
            u.member_id, u.full_name, u.sex, u.address, u.email, u.phone,
            u.membership_status, u.membership_expiry, u.profile_image, u.position,
            u.tenure_start, u.tenure_end
        FROM users u $where
        ORDER BY u.member_id";

$stmt = $link->prepare($sql);
if ($types) $stmt->bind_param($types, ...$params);
$stmt->execute();
$result = $stmt->get_result();
$rows = $result->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// === EXPORT TO CSV ===
if ($export === 'csv') {
    $filename = $scope === 'executives' ? 'executives.csv' : 'members.csv';
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=' . $filename);

    $output = fopen('php://output', 'w');
    $header = [
        'Member ID', 'Full Name', 'Sex', 'Address', 'Email', 'Phone',
        'Status', 'Expiry Date', 'Position', 'Tenure Start', 'Tenure End'
    ];
    fputcsv($output, $header);

    foreach ($rows as $row) {
        fputcsv($output, [
            $row['member_id'],
            $row['full_name'],
            $row['sex'] ?? '-',
            $row['address'] ?? '-',
            $row['email'],
            $row['phone'] ?? '-',
            $row['membership_status'],
            format_date($row['membership_expiry']),
            $row['position'] ?? 'Member',
            format_date($row['tenure_start']),
            format_date($row['tenure_end'])
        ]);
    }
    fclose($output);
    exit;
}

// === EXPORT TO PDF ===
if ($export === 'pdf') {
    $pdf = new TCPDF('L', 'mm', 'A4', true, 'UTF-8', false);
    $pdf->SetCreator('Club Portal');
    $pdf->SetTitle($scope === 'executives' ? 'Executive List' : 'Members List');
    $pdf->SetMargins(10, 10, 10);
    $pdf->SetAutoPageBreak(true, 10);
    $pdf->AddPage();

    $title = $scope === 'executives' ? 'EXECUTIVE BOARD' : 'MEMBERS LIST';
    $pdf->SetFont('helvetica', 'B', 16);
    $pdf->Cell(0, 10, $title, 0, 1, 'C');
    $pdf->Ln(5);

    $pdf->SetFont('helvetica', '', 8);
    $html = '<table border="1" cellpadding="4">
        <thead>
            <tr style="background-color:#f0f0f0;">
                <th><b>ID</b></th>
                <th><b>Name</b></th>
                <th><b>Sex</b></th>
                <th><b>Email</b></th>
                <th><b>Phone</b></th>
                <th><b>Status</b></th>
                <th><b>Expiry</b></th>
                <th><b>Position</b></th>
                <th><b>Tenure</b></th>
            </tr>
        </thead>
        <tbody>';

    foreach ($rows as $row) {
        $tenure = '';
        if ($row['tenure_start']) {
            $tenure = format_date($row['tenure_start']);
            if ($row['tenure_end']) $tenure .= ' → ' . format_date($row['tenure_end']);
        }

        $html .= '<tr>
            <td>' . htmlspecialchars($row['member_id']) . '</td>
            <td>' . htmlspecialchars($row['full_name']) . '</td>
            <td>' . ($row['sex'] ?? '-') . '</td>
            <td>' . htmlspecialchars($row['email']) . '</td>
            <td>' . ($row['phone'] ?? '-') . '</td>
            <td>' . htmlspecialchars($row['membership_status']) . '</td>
            <td>' . format_date($row['membership_expiry']) . '</td>
            <td>' . htmlspecialchars($row['position'] ?? 'Member') . '</td>
            <td>' . $tenure . '</td>
        </tr>';
    }

    $html .= '</tbody></table>';
    $pdf->writeHTML($html, true, false, true, false, '');
    $pdf->Ln(5);
    $pdf->SetFont('helvetica', 'I', 8);
    $pdf->Cell(0, 10, 'Generated on: ' . date('Y-m-d H:i'), 0, 1, 'R');

    $filename = $scope === 'executives' ? 'executives.pdf' : 'members.pdf';
    $pdf->Output($filename, 'I');
    exit;
}