<?php
require_once 'includes/functions.php';
require_once 'includes/db_config.php';
session_start();

// Check if user is logged in and has permission to manage payments
if (!is_logged_in() || !user_has_permission($link, 'view_payments')) {
    redirect("login.php");
}

// Check column existence
$columns_query = "SHOW COLUMNS FROM payment_obligations";
$columns_result = mysqli_query($link, $columns_query);
$columns = [];
while ($row = mysqli_fetch_assoc($columns_result)) {
    $columns[] = $row['Field'];
}
$has_due_date = in_array('due_date', $columns);
$has_required_amount = in_array('required_amount', $columns);
$has_deleted_at = in_array('deleted_at', $columns);

// Trigger obligation generation only on explicit action
$message = '';
$message_type = '';
$currency_symbol = get_setting('currency_symbol', $link) ?? '₦';

if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['generate_obligations'])) {
    include 'generate_obligations.php';
    $message = "Obligations generated successfully!";
    $message_type = 'success';
}

// Handle form submissions for adding/removing obligations
if ($_SERVER["REQUEST_METHOD"] == "POST" && !isset($_POST['generate_obligations'])) {
    if (isset($_POST['add_obligation'])) {
        $user_id = intval($_POST['user_id']);
        $payment_type_name = sanitize_input($_POST['payment_type_name']);
        $payment_cycle = intval($_POST['payment_cycle']);
        $required_amount = $has_required_amount ? floatval($_POST['required_amount']) : 0.00;
        $due_date = $has_due_date ? sanitize_input($_POST['due_date'] ?? null) : null;

        // Validate payment type exists and fetch its category
        $sql_check_type = "SELECT payment_category, required_amount, recurrence_interval, event_date 
                           FROM payment_types 
                           WHERE type_name = ? AND deleted_at IS NULL";
        if ($stmt_check = mysqli_prepare($link, $sql_check_type)) {
            mysqli_stmt_bind_param($stmt_check, "s", $payment_type_name);
            mysqli_stmt_execute($stmt_check);
            $type_result = mysqli_stmt_get_result($stmt_check);
            $type_data = mysqli_fetch_assoc($type_result);
            mysqli_stmt_close($stmt_check);

            if (!$type_data) {
                $message = "Invalid payment type selected.";
                $message_type = 'danger';
            } elseif ($type_data['payment_category'] === 'event' && $has_due_date && empty($due_date)) {
                $message = "Due date is required for event-based payments.";
                $message_type = 'danger';
            } else {
                $sql = "INSERT INTO payment_obligations (user_id, payment_type, payment_cycle" . 
                       ($has_required_amount ? ", required_amount" : "") . 
                       ($has_due_date ? ", due_date" : "") . ", created_at) 
                        VALUES (?, ?, ?" . ($has_required_amount ? ", ?" : "") . ($has_due_date ? ", ?" : "") . ", NOW())";
                if ($stmt = mysqli_prepare($link, $sql)) {
                    $bind_types = "isi" . ($has_required_amount ? "d" : "") . ($has_due_date ? "s" : "");
                    $bind_params = [&$user_id, &$payment_type_name, &$payment_cycle];
                    if ($has_required_amount) {
                        $bind_params[] = &$required_amount;
                    }
                    if ($has_due_date) {
                        $bind_params[] = &$due_date;
                    }
                    mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
                    if (mysqli_stmt_execute($stmt)) {
                        $message = "Payment obligation added successfully!";
                        $message_type = 'success';
                    } else {
                        $message = "Error adding obligation: " . mysqli_error($link);
                        $message_type = 'danger';
                    }
                    mysqli_stmt_close($stmt);
                }
            }
        }
    } elseif (isset($_POST['delete_obligation'])) {
        $obligation_id = intval($_POST['obligation_id']);
        $sql = $has_deleted_at 
            ? "UPDATE payment_obligations SET deleted_at = NOW() WHERE id = ?" 
            : "DELETE FROM payment_obligations WHERE id = ?";
        if ($stmt = mysqli_prepare($link, $sql)) {
            mysqli_stmt_bind_param($stmt, "i", $obligation_id);
            if (mysqli_stmt_execute($stmt)) {
                $message = "Payment obligation deleted successfully!";
                $message_type = 'success';
            } else {
                $message = "Error deleting obligation: " . mysqli_error($link);
                $message_type = 'danger';
            }
            mysqli_stmt_close($stmt);
        }
    }
}

// Fetch all members
$members_sql = "SELECT id, full_name, member_id, role, membership_status 
                FROM users 
                WHERE role = 'member' AND (deleted_at IS NULL OR deleted_at > NOW()) 
                ORDER BY full_name ASC";
$members_result = mysqli_query($link, $members_sql);
$all_members = mysqli_fetch_all($members_result, MYSQLI_ASSOC);

// Fetch all payment types with additional fields
$types_sql = "SELECT id, type_name, required_amount, payment_category, recurrence_interval, event_date 
              FROM payment_types 
              WHERE deleted_at IS NULL 
              ORDER BY type_name ASC";
$types_result = mysqli_query($link, $types_sql);
$all_types = mysqli_fetch_all($types_result, MYSQLI_ASSOC);

$user_id = isset($_GET['user_id']) ? intval($_GET['user_id']) : null;
$selected_member = null;
$obligations = [];

if ($user_id) {
    // Fetch details for the selected member
    $member_details_sql = "SELECT id, full_name, member_id 
                          FROM users 
                          WHERE id = ? AND (deleted_at IS NULL OR deleted_at > NOW())";
    if ($member_details_stmt = mysqli_prepare($link, $member_details_sql)) {
        mysqli_stmt_bind_param($member_details_stmt, "i", $user_id);
        mysqli_stmt_execute($member_details_stmt);
        $member_details_result = mysqli_stmt_get_result($member_details_stmt);
        $selected_member = mysqli_fetch_assoc($member_details_result);
        mysqli_stmt_close($member_details_stmt);
    }

    // Fetch existing obligations for the selected member
    $select_fields = "po.id, po.payment_type, po.payment_cycle";
    if ($has_required_amount) {
        $select_fields .= ", COALESCE(po.required_amount, 0) AS required_amount";
    }
    if ($has_due_date) {
        $select_fields .= ", po.due_date";
    }
    $where_clause = $has_deleted_at ? " AND po.deleted_at IS NULL" : "";
    $obligations_sql = "SELECT $select_fields
                        FROM payment_obligations po
                        WHERE po.user_id = ?$where_clause";
    if ($obligations_stmt = mysqli_prepare($link, $obligations_sql)) {
        mysqli_stmt_bind_param($obligations_stmt, "i", $user_id);
        mysqli_stmt_execute($obligations_stmt);
        $obligations_result = mysqli_stmt_get_result($obligations_stmt);
        $obligations = mysqli_fetch_all($obligations_result, MYSQLI_ASSOC);
        mysqli_stmt_close($obligations_stmt);
    }
}
?>

<?php require_once 'includes/header.php'; ?>
<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Manage Member Obligations</h2>
            <?php if ($message) { ?>
                <div class="alert alert-<?php echo htmlspecialchars($message_type); ?>">
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php } ?>

            <div class="card mb-4">
                <div class="card-header">Select a Member</div>
                <div class="card-body">
                    <form action="manage_obligations.php" method="get">
                        <div class="form-group mb-3">
                            <select name="user_id" class="form-control" onchange="this.form.submit()">
                                <option value="">-- Select Member --</option>
                                <?php foreach ($all_members as $member) { ?>
                                    <option value="<?php echo htmlspecialchars($member['id']); ?>"
                                        <?php echo ($user_id == $member['id']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($member['full_name'] . ' (' . $member['member_id'] . ')'); ?>
                                    </option>
                                <?php } ?>
                            </select>
                        </div>
                    </form>
                </div>
            </div>

            <?php if ($selected_member) { ?>
            <div class="card mb-4">
                <div class="card-header">
                    Obligations for: <?php echo htmlspecialchars($selected_member['full_name']); ?>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead>
                                <tr>
                                    <th>Payment Type</th>
                                    <th>Cycle (Year)</th>
                                    <?php if ($has_required_amount) { ?>
                                        <th>Amount (<?php echo htmlspecialchars($currency_symbol); ?>)</th>
                                    <?php } ?>
                                    <?php if ($has_due_date) { ?>
                                        <th>Due Date</th>
                                    <?php } ?>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($obligations)) { ?>
                                    <tr>
                                        <td colspan="<?php echo 3 + ($has_required_amount ? 1 : 0) + ($has_due_date ? 1 : 0); ?>" class="text-center">No obligations found for this member.</td>
                                    </tr>
                                <?php } else { ?>
                                    <?php foreach ($obligations as $obligation) { ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($obligation['payment_type']); ?></td>
                                        <td><?php echo htmlspecialchars($obligation['payment_cycle']); ?></td>
                                        <?php if ($has_required_amount) { ?>
                                            <td><?php echo htmlspecialchars($currency_symbol . ' ' . number_format($obligation['required_amount'] ?? 0.00, 2)); ?></td>
                                        <?php } ?>
                                        <?php if ($has_due_date) { ?>
                                            <td><?php echo htmlspecialchars($obligation['due_date'] ? date('Y-m-d H:i', strtotime($obligation['due_date'])) : 'N/A'); ?></td>
                                        <?php } ?>
                                        <td>
                                            <form method="post" action="manage_obligations.php?user_id=<?php echo $user_id; ?>" style="display:inline;">
                                                <input type="hidden" name="obligation_id" value="<?php echo $obligation['id']; ?>">
                                                <input type="hidden" name="delete_obligation" value="1">
                                                <button type="submit" class="btn btn-danger btn-sm" onclick="return confirm('Are you sure you want to delete this obligation?');">
                                                    Delete
                                                </button>
                                            </form>
                                        </td>
                                    </tr>
                                    <?php } ?>
                                <?php } ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <div class="card mb-4">
                <div class="card-header">Add New Obligation</div>
                <div class="card-body">
                    <form method="post" action="manage_obligations.php?user_id=<?php echo $user_id; ?>">
                        <input type="hidden" name="user_id" value="<?php echo $user_id; ?>">
                        <div class="form-group mb-3">
                            <label for="payment_type_name">Payment Type</label>
                            <select name="payment_type_name" id="payment_type_name" class="form-control" required onchange="updateObligationFields()">
                                <option value="">-- Select Payment Type --</option>
                                <?php foreach ($all_types as $type) { ?>
                                    <option value="<?php echo htmlspecialchars($type['type_name']); ?>" 
                                            data-required-amount="<?php echo htmlspecialchars($type['required_amount']); ?>"
                                            data-payment-category="<?php echo htmlspecialchars($type['payment_category']); ?>"
                                            data-event-date="<?php echo htmlspecialchars($type['event_date'] ? date('Y-m-d\TH:i', strtotime($type['event_date'])) : ''); ?>">
                                        <?php echo htmlspecialchars($type['type_name']); ?>
                                    </option>
                                <?php } ?>
                            </select>
                        </div>
                        <div class="form-group mb-3">
                            <label for="payment_cycle">Payment Cycle (Year)</label>
                            <input type="number" name="payment_cycle" id="payment_cycle" class="form-control" value="<?php echo date('Y'); ?>" required>
                        </div>
                        <?php if ($has_required_amount) { ?>
                        <div class="form-group mb-3">
                            <label for="required_amount">Required Amount (<?php echo htmlspecialchars($currency_symbol); ?>)</label>
                            <input type="number" name="required_amount" id="required_amount" class="form-control" step="0.01" min="0" required readonly>
                        </div>
                        <?php } ?>
                        <?php if ($has_due_date) { ?>
                        <div class="form-group mb-3" id="due_date_container">
                            <label for="due_date">Due Date</label>
                            <input type="datetime-local" name="due_date" id="due_date" class="form-control">
                        </div>
                        <?php } ?>
                        <button type="submit" name="add_obligation" class="btn btn-success">Add Obligation</button>
                    </form>
                </div>
            </div>
            <div class="card mb-4">
                <div class="card-header">Generate Obligations</div>
                <div class="card-body">
                    <form method="post" action="manage_obligations.php?user_id=<?php echo $user_id; ?>">
                        <button type="submit" name="generate_obligations" class="btn btn-primary">Generate Obligations for All Members</button>
                    </form>
                </div>
            </div>
            <?php } ?>
        </main>
    </div>
</div>

<script>
function updateObligationFields() {
    const paymentTypeSelect = document.getElementById('payment_type_name');
    const requiredAmountInput = document.getElementById('required_amount');
    const dueDateInput = document.getElementById('due_date');
    const dueDateContainer = document.getElementById('due_date_container');

    const selectedOption = paymentTypeSelect.options[paymentTypeSelect.selectedIndex];
    const requiredAmount = selectedOption ? selectedOption.getAttribute('data-required-amount') : '';
    const paymentCategory = selectedOption ? selectedOption.getAttribute('data-payment-category') : '';
    const eventDate = selectedOption ? selectedOption.getAttribute('data-event-date') : '';

    <?php if ($has_required_amount) { ?>
        requiredAmountInput.value = requiredAmount || '';
    <?php } ?>
    <?php if ($has_due_date) { ?>
        if (paymentCategory === 'event') {
            dueDateContainer.style.display = 'block';
            dueDateInput.value = eventDate || '';
            dueDateInput.required = true;
        } else if (paymentCategory === 'recurring') {
            dueDateContainer.style.display = 'block';
            dueDateInput.value = '';
            dueDateInput.required = true;
        } else {
            dueDateContainer.style.display = 'none';
            dueDateInput.value = '';
            dueDateInput.required = false;
        }
    <?php } ?>
}
</script>

<?php require_once 'includes/footer.php'; ?>