<?php
// login.php
ob_start(); // Start output buffering
require_once 'includes/db_config.php';
require_once 'includes/functions.php';
session_start();

// Session timeout
$timeout = 1800; // 30 minutes
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity']) > $timeout) {
    session_unset();
    session_destroy();
    $message = "Session timed out. Please log in again.";
}
$_SESSION['last_activity'] = time();

// Generate CSRF token
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

$message = "";
$captcha_question = "";

// Check if a form was submitted
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // Verify CSRF token
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        $message = "Invalid CSRF token.";
        error_log("CSRF token validation failed in login.php");
    } else {
        $email = sanitize_input($_POST['email']);
        $password = $_POST['password'];
        $captcha_input = sanitize_input($_POST['captcha']);

        // Check if captcha is correct
        if (!isset($_SESSION['captcha_answer']) || $captcha_input != $_SESSION['captcha_answer']) {
            $message = "Incorrect captcha. Please try again.";
        } else {
            // Captcha is correct, proceed with login
            unset($_SESSION['captcha_answer']); // Unset for security

            // Retrieve user by email
            $sql = "SELECT id, full_name, email, password, role, membership_status FROM users WHERE email = ?";
            if ($stmt = mysqli_prepare($link, $sql)) {
                mysqli_stmt_bind_param($stmt, "s", $email);
                mysqli_stmt_execute($stmt);
                $result = mysqli_stmt_get_result($stmt);

                if (mysqli_num_rows($result) == 1) {
                    $user = mysqli_fetch_assoc($result);

                    // Verify password
                    if (password_verify($password, $user['password'])) {
                        // Password is correct, set session variables
                        $_SESSION['user_id'] = $user['id'];
                        $_SESSION['user_name'] = $user['full_name'];
                        $_SESSION['user_email'] = $user['email'];
                        $_SESSION['user_role'] = $user['role'];
                        $_SESSION['last_activity'] = time(); // Update activity time

                        // Log the successful login
                        error_log("User logged in successfully: ID={$user['id']}, Role={$user['role']}");

                        // Redirect based on the user's role
                        if ($user['role'] === 'super_admin') {
                            redirect("super_admin_dashboard.php");
                        } elseif ($user['role'] === 'admin') {
                            redirect("admin_dashboard.php");
                        } elseif ($user['role'] === 'accountant') {
                            redirect("accountant_dashboard.php");
                        } else {
                            redirect("member_dashboard.php");
                        }
                    } else {
                        $message = "Invalid email or password.";
                    }
                } else {
                    $message = "Invalid email or password.";
                }
                mysqli_stmt_close($stmt);
                mysqli_free_result($result);
            } else {
                $message = "Error preparing statement: " . mysqli_error($link);
                error_log("Error preparing statement in login.php: " . mysqli_error($link));
            }
        }
    }
    // Regenerate CSRF token after form submission
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Generate a new math captcha
$num1 = rand(1, 10);
$num2 = rand(1, 10);
$_SESSION['captcha_answer'] = $num1 + $num2;
$captcha_question = "What is " . $num1 . " + " . $num2 . "?";

// Check if user is already logged in
if (is_logged_in()) {
    if ($_SESSION['user_role'] === 'super_admin') {
        redirect("super_admin_dashboard.php");
    } elseif ($_SESSION['user_role'] === 'admin') {
        redirect("admin_dashboard.php");
    } elseif ($_SESSION['user_role'] === 'accountant') {
        redirect("accountant_dashboard.php");
    } else {
        redirect("member_dashboard.php");
    }
}
?>

<?php require_once 'includes/header.php'; ?>
<div class="container mt-5">
    <div class="row justify-content-center">
        <div class="col-md-6">
            <div class="card">
                <div class="card-header">
                    <h3 class="text-center">Login</h3>
                </div>
                <div class="card-body">
                    <?php if (!empty($message)): ?>
                        <div class="alert alert-danger"><?php echo htmlspecialchars($message); ?></div>
                    <?php endif; ?>
                    <form action="login.php" method="post">
                        <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">
                        <div class="form-group mb-3">
                            <label for="email">Email address</label>
                            <input type="email" class="form-control" id="email" name="email" required>
                        </div>
                        <div class="form-group mb-3">
                            <label for="password">Password</label>
                            <input type="password" class="form-control" id="password" name="password" required>
                        </div>
                        <div class="form-group mb-3">
                            <label for="captcha"><?php echo htmlspecialchars($captcha_question); ?></label>
                            <input type="text" class="form-control" id="captcha" name="captcha" required>
                        </div>
                        <div class="d-grid gap-2">
                            <button type="submit" class="btn btn-primary">Login</button>
                        </div>
                    </form>
                </div>
                <div class="card-footer text-center">
                    <p class="mb-1"><a href="forgot_password.php">Forgot Password?</a></p>
                    <p>Don't have an account? <a href="register.php">Register here</a></p>
                </div>
            </div>
        </div>
    </div>
</div>
<?php require_once 'includes/footer.php'; ?>
<?php ob_end_flush(); // Flush output buffer ?>