<?php
define('IN_REPORTS', true);
/* --------------------------------------------------------------
   financial_summary.php
   -------------------------------------------------------------- */
session_start();

require_once 'includes/functions.php';
require_once 'includes/db_config.php';
require_once 'includes/financial_queries.php';
require_once 'includes/financial_utils.php';

/* ---------- Development error reporting (remove in prod) ---------- */
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

/* ---------- DB connection check ---------- */
if (!$link) {
    die("ERROR: Could not connect. " . mysqli_connect_error());
}

/* ---------- Auth ---------- */
if (!is_logged_in()) {
    redirect('login.php');
}
$is_admin = is_admin();

$system_name      = get_setting('system_name', $link) ?? 'Club Portal';
$currency_symbol  = get_setting('currency_symbol', $link) ?? '₦';

/* ---------- Sanitize & defaults ---------- */
$valid_record_types = ['all', 'payments', 'expenses', 'debts', 'benefits'];
$valid_time_periods = ['all', 'range', 'week', 'month', 'year'];
$valid_per_page     = [5, 10, 25, 50, 100];

$record_type = sanitize_filter($_GET['record_type'] ?? 'all', $valid_record_types, 'all');
$time_period = sanitize_filter($_GET['time_period'] ?? 'all', $valid_time_periods, 'all');

$start_date = $_GET['start_date'] ?? date('Y-m-01');
$end_date   = $_GET['end_date']   ?? date('Y-m-d');

$search = $_GET['search'] ?? '';
$member_id = (int)($_GET['member_id'] ?? 0);
$date_field = ($_GET['date_field'] ?? '') === 'due_date' ? 'due_date' : 'created_at';

/* Individual type filters */
$type_filter_payments = $_GET['type_filter_payments'] ?? '';
$type_filter_expenses = $_GET['type_filter_expenses'] ?? '';
$type_filter_benefits = $_GET['type_filter_benefits'] ?? '';

/* ---------- Pagination ---------- */
$tables = ['payments', 'expenses', 'debts', 'benefits'];
$pagination = [];

foreach ($tables as $table) {
    $per_page = (int)($_GET["per_page_$table"] ?? 10);
    $per_page = in_array($per_page, $valid_per_page) ? $per_page : 10;

    $page = max(1, (int)($_GET["page_$table"] ?? 1));
    $offset = ($page - 1) * $per_page;

    $pagination[$table] = compact('per_page', 'page', 'offset');
}

/* ---------- Members for filter ---------- */
$members = [];
$sql = "SELECT id, full_name FROM users WHERE role = 'member' AND (deleted_at IS NULL OR deleted_at > NOW()) ORDER BY full_name";
$res = mysqli_query($link, $sql);
while ($row = mysqli_fetch_assoc($res)) {
    $members[] = $row;
}
mysqli_free_result($res);

/* ---------- CSV Export ---------- */
if (isset($_GET['export']) && in_array($_GET['export'], $tables)) {
    $export_table = $_GET['export'];
    $full_pagination = ['per_page' => PHP_INT_MAX, 'page' => 1, 'offset' => 0];
    $data = [];

    switch ($export_table) {
        case 'payments':
            $data = get_payments($link, $record_type, $time_period, $start_date, $end_date,
                                 $type_filter_payments, $search, $full_pagination, $member_id);
            break;
        case 'expenses':
            $data = get_expenses($link, $record_type, $time_period, $start_date, $end_date,
                                 $type_filter_expenses, $search, $full_pagination);
            break;
        case 'benefits':
            $data = get_benefits($link, $record_type, $time_period, $start_date, $end_date,
                                 $type_filter_benefits, $search, $full_pagination, $member_id);
            break;
        case 'debts':
            $data = get_debts($link, $record_type, $time_period, $start_date, $end_date,
                              '', $search, $date_field, $full_pagination, $member_id);
            break;
    }

    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="' . $export_table . '.csv"');
    $out = fopen('php://output', 'w');

    switch ($export_table) {
        case 'payments':
            fputcsv($out, ['ID','Full Name','Payment Type','Amount','Status','Created At','Payment Cycle']);
            foreach ($data as $r) {
                fputcsv($out, [
                    $r['id'],
                    $r['full_name'],
                    $r['payment_type'],
                    $r['amount'],
                    $r['payment_status'],
                    $r['created_at'],
                    $r['payment_cycle'] ?? 'N/A'
                ]);
            }
            break;
        case 'expenses':
            fputcsv($out, ['ID','Expense Type','Amount','Description','Created At']);
            foreach ($data as $r) {
                fputcsv($out, [
                    $r['id'],
                    $r['expense_type'] ?? 'N/A',
                    $r['amount'],
                    $r['description'],
                    $r['created_at']
                ]);
            }
            break;
        case 'benefits':
            fputcsv($out, ['ID','Member ID','Full Name','Benefit Type','Amount','Created At']);
            foreach ($data as $r) {
                fputcsv($out, [
                    $r['id'],
                    $r['member_id'],
                    $r['full_name'],
                    $r['benefit_type'],
                    $r['amount'],
                    $r['created_at']
                ]);
            }
            break;
        case 'debts':
            fputcsv($out, ['Member ID','Full Name','Payment Type','Required Amount','Total Paid','Debt','Payment Cycle','Due Date']);
            foreach ($data as $r) {
                fputcsv($out, [
                    $r['member_id'],
                    $r['full_name'],
                    $r['payment_type'],
                    $r['required_amount'],
                    $r['total_paid'],
                    $r['debt'],
                    $r['payment_cycle'] ?? 'N/A',
                    $r['due_date'] ?? 'N/A'
                ]);
            }
            break;
    }
    fclose($out);
    exit;
}

/* ---------- Totals & counts ---------- */
$total_counts = get_total_counts($link, $record_type, $time_period, $start_date, $end_date,
                                 '', $search, $date_field, $pagination, $member_id);

$total_pages = [];
foreach ($tables as $t) {
    $total_pages[$t] = max(1, (int)ceil($total_counts[$t] / $pagination[$t]['per_page']));
}

$total_payments = get_total_payments($link, $record_type, $time_period, $start_date, $end_date,
                                    $type_filter_payments, $member_id);
$total_expenses = get_total_expenses($link, $record_type, $time_period, $start_date, $end_date,
                                    $type_filter_expenses);
$total_benefits = get_total_benefits($link, $record_type, $time_period, $start_date, $end_date,
                                    $type_filter_benefits, $member_id);
$total_debts    = get_total_debts($link, $record_type, $time_period, $start_date, $end_date,
                                  '', $date_field, $member_id);

$current_balance = $total_payments - $total_expenses - $total_benefits;

/* ---------- Previous period comparison ---------- */
$prev_month_start = date('Y-m-01', strtotime('-1 month'));
$prev_month_end   = date('Y-m-t', strtotime('-1 month'));

$prev_total_payments = $prev_total_expenses = $prev_total_benefits = $prev_total_debts = 0;

/* Payments */
$sql = "SELECT COALESCE(SUM(amount),0) FROM payments WHERE payment_status='Paid' AND created_at BETWEEN ? AND ?";
if ($stmt = mysqli_prepare($link, $sql)) {
    mysqli_stmt_bind_param($stmt, 'ss', $prev_month_start, $prev_month_end);
    mysqli_stmt_execute($stmt);
    $prev_total_payments = mysqli_fetch_array(mysqli_stmt_get_result($stmt))[0] ?? 0;
    mysqli_stmt_close($stmt);
}

/* Expenses */
$sql = "SELECT COALESCE(SUM(amount),0) FROM expenses WHERE created_at BETWEEN ? AND ?";
if ($stmt = mysqli_prepare($link, $sql)) {
    mysqli_stmt_bind_param($stmt, 'ss', $prev_month_start, $prev_month_end);
    mysqli_stmt_execute($stmt);
    $prev_total_expenses = mysqli_fetch_array(mysqli_stmt_get_result($stmt))[0] ?? 0;
    mysqli_stmt_close($stmt);
}

/* Benefits */
$sql = "SELECT COALESCE(SUM(amount),0) FROM member_benefits WHERE created_at BETWEEN ? AND ?";
if ($stmt = mysqli_prepare($link, $sql)) {
    mysqli_stmt_bind_param($stmt, 'ss', $prev_month_start, $prev_month_end);
    mysqli_stmt_execute($stmt);
    $prev_total_benefits = mysqli_fetch_array(mysqli_stmt_get_result($stmt))[0] ?? 0;
    mysqli_stmt_close($stmt);
}

/* Debts (same complex query) */
$sql_prev_debts = "SELECT COALESCE(SUM(debt),0) AS total_debt
                   FROM (
                       SELECT GREATEST(COALESCE(po.required_amount,0) - COALESCE(SUM(p.amount),0),0) AS debt
                       FROM payment_obligations po
                       JOIN users u ON po.user_id = u.id
                       JOIN payment_types pt ON po.payment_type = pt.type_name
                       LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status='Paid'
                       WHERE po.deleted_at IS NULL
                         AND u.role='member'
                         AND (u.deleted_at IS NULL OR u.deleted_at > NOW())
                         AND COALESCE(po.required_amount,0) > 0
                         AND " . ($date_field === 'due_date' ? 'po.due_date' : 'po.created_at') . " BETWEEN ? AND ?
                       GROUP BY po.id, po.required_amount
                       HAVING debt > 0
                   ) AS sub";
if ($stmt = mysqli_prepare($link, $sql_prev_debts)) {
    mysqli_stmt_bind_param($stmt, 'ss', $prev_month_start, $prev_month_end);
    mysqli_stmt_execute($stmt);
    $prev_total_debts = mysqli_fetch_array(mysqli_stmt_get_result($stmt))[0] ?? 0;
    mysqli_stmt_close($stmt);
}

$payment_change  = $total_payments - $prev_total_payments;
$expenses_change = $total_expenses - $prev_total_expenses;
$benefits_change = $total_benefits - $prev_total_benefits;
$debts_change    = $total_debts    - $prev_total_debts;

/* ---------- Misc ---------- */
$total_members   = get_total_members($link);
$total_payment_types = get_total_payment_types($link);
$prev_balances   = get_previous_balances($link, $end_date, $date_field);

/* ---------- Type option dropdowns ---------- */
$type_options_payments = get_type_options($link, 'income',   $type_filter_payments);
$type_options_expenses = get_type_options($link, 'expenses',$type_filter_expenses);
$type_options_benefits = get_type_options($link, 'benefits',$type_filter_benefits);

/* ---------- Paginated data ---------- */
$payments = get_payments($link, $record_type, $time_period, $start_date, $end_date,
                         $type_filter_payments, $search, $pagination['payments'], $member_id);
$expenses = get_expenses($link, $record_type, $time_period, $start_date, $end_date,
                         $type_filter_expenses, $search, $pagination['expenses']);
$benefits = get_benefits($link, $record_type, $time_period, $start_date, $end_date,
                         $type_filter_benefits, $search, $pagination['benefits'], $member_id);
$debts    = get_debts($link, $record_type, $time_period, $start_date, $end_date,
                      '', $search, $date_field, $pagination['debts'], $member_id);

/* ---------- Summary data for charts & tables ---------- */
$payment_types_summary = get_payment_types_summary($link, $record_type, $time_period,
                                                  $start_date, $end_date, $type_filter_payments,
                                                  $date_field, $member_id);
$expense_types_summary = get_expense_types_summary($link, $record_type, $time_period,
                                                   $start_date, $end_date, $type_filter_expenses);
$benefit_types_summary = get_benefit_types_summary($link, $record_type, $time_period,
                                                   $start_date, $end_date, $type_filter_benefits,
                                                   $member_id);

$line_chart_data = get_line_chart_data($link, $end_date, $date_field);

/* ---------- Build safe histogram arrays ---------- */
$histogram_labels = $histogram_paid = $histogram_excess = [];

foreach (($payment_types_summary ?? []) as $row) {
    $histogram_labels[] = "'Payment: " . addslashes($row['type_name']) . "'";
    $histogram_paid[]   = $row['total_paid'];
    $histogram_excess[] = $row['excess_amount'];
}
foreach (($expense_types_summary ?? []) as $row) {
    $histogram_labels[] = "'Expense: " . addslashes($row['type_name']) . "'";
    $histogram_paid[]   = $row['total'];
    $histogram_excess[] = 0;
}
foreach (($benefit_types_summary ?? []) as $row) {
    $histogram_labels[] = "'Benefit: " . addslashes($row['type_name']) . "'";
    $histogram_paid[]   = $row['total'];
    $histogram_excess[] = 0;
}

/* ---------- Total excess donations (for card) ---------- */
$total_excess_donations = array_sum(array_column($payment_types_summary ?? [], 'excess_amount'));

/* ---------- Include template ---------- */
require_once 'templates/financial_summary_template.php';
?>