<?php
require_once 'includes/functions.php';
require_once 'includes/db_config.php';
session_start();

// Check if user is logged in and is an admin
if (!is_logged_in() || !is_admin()) {
    redirect("login.php");
}

$message = '';
$error = '';

// Handle expense logging
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['log_expense'])) {
    $amount = filter_input(INPUT_POST, 'amount', FILTER_VALIDATE_FLOAT);
    $expense_type_id = filter_input(INPUT_POST, 'expense_type_id', FILTER_VALIDATE_INT);
    $description = sanitize_input($_POST['description'] ?? '');

    if ($amount === false || $expense_type_id === false || empty($description)) {
        $error = "Invalid input. Please fill all fields correctly.";
    } else {
        $sql = "INSERT INTO expenses (expense_type_id, amount, description) VALUES (?, ?, ?)";
        if ($stmt = mysqli_prepare($link, $sql)) {
            mysqli_stmt_bind_param($stmt, "ids", $expense_type_id, $amount, $description);
            if (mysqli_stmt_execute($stmt)) {
                $message = "Expense logged successfully!";
            } else {
                $error = "Error logging expense: " . mysqli_error($link);
            }
            mysqli_stmt_close($stmt);
        } else {
            $error = "Database error. Please try again.";
        }
    }
}

// Handle CSV export
if (isset($_GET['export']) && $_GET['export'] == 'csv') {
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=expenses.csv');

    $output = fopen('php://output', 'w');
    fputcsv($output, array('ID', 'Date', 'Type', 'Amount', 'Description'));

    $sql = "SELECT e.id, e.created_at, et.type_name, e.amount, e.description
            FROM expenses e
            JOIN expense_types et ON e.expense_type_id = et.id
            ORDER BY e.created_at DESC";
    $result = mysqli_query($link, $sql);

    while ($row = mysqli_fetch_assoc($result)) {
        fputcsv($output, $row);
    }
    fclose($output);
    exit();
}

// Fetch all expense types to populate the form
$expense_types = get_all_expense_types($link);

// Fetch expense history for display
$sql_history = "SELECT e.id, e.created_at, et.type_name, e.amount, e.description
                FROM expenses e
                JOIN expense_types et ON e.expense_type_id = et.id
                ORDER BY e.created_at DESC";
$result_history = mysqli_query($link, $sql_history);

// Fetch totals by month and year
$sql_monthly = "SELECT YEAR(created_at) as year, MONTHNAME(created_at) as month, SUM(amount) as total
                FROM expenses
                GROUP BY YEAR(created_at), MONTH(created_at)
                ORDER BY created_at DESC";
$result_monthly = mysqli_query($link, $sql_monthly);
$monthly_totals = [];
while ($row = mysqli_fetch_assoc($result_monthly)) {
    $monthly_totals[] = $row;
}

$sql_yearly = "SELECT YEAR(created_at) as year, SUM(amount) as total
               FROM expenses
               GROUP BY YEAR(created_at)
               ORDER BY year DESC";
$result_yearly = mysqli_query($link, $sql_yearly);
$yearly_totals = [];
while ($row = mysqli_fetch_assoc($result_yearly)) {
    $yearly_totals[] = $row;
}
?>

<?php require_once 'includes/header.php'; ?>
<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Expense Management</h2>

            <?php if ($message): ?>
                <div class="message success-message"><?php echo htmlspecialchars($message); ?></div>
            <?php endif; ?>
            <?php if ($error): ?>
                <div class="message error-message"><?php echo htmlspecialchars($error); ?></div>
            <?php endif; ?>

            <div class="card mb-4">
                <div class="card-header">Log a New Expense</div>
                <div class="card-body">
                    <form action="expense_management.php" method="post">
                        <div class="form-group">
                            <label for="amount">Amount</label>
                            <input type="number" step="0.01" id="amount" name="amount" class="form-control" required>
                        </div>
                        <div class="form-group">
                            <label for="expense_type_id">Expense Type</label>
                            <select id="expense_type_id" name="expense_type_id" class="form-control" required>
                                <option value="">Select a type</option>
                                <?php foreach ($expense_types as $type): ?>
                                    <option value="<?php echo htmlspecialchars($type['id']); ?>"><?php echo htmlspecialchars($type['type_name']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group">
                            <label for="description">Description</label>
                            <textarea id="description" name="description" class="form-control" rows="3" required></textarea>
                        </div>
                        <button type="submit" name="log_expense" class="btn btn-primary">Log Expense</button>
                    </form>
                </div>
            </div>

            <div class="card mb-4">
                <div class="card-header">Expense Totals</div>
                <div class="card-body">
                    <h4>Monthly Totals</h4>
                    <table class="table table-striped">
                        <thead>
                            <tr>
                                <th>Month</th>
                                <th>Year</th>
                                <th>Total Amount</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($monthly_totals)): ?>
                                <tr><td colspan="3">No monthly expense data.</td></tr>
                            <?php else: ?>
                                <?php foreach ($monthly_totals as $total): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($total['month']); ?></td>
                                        <td><?php echo htmlspecialchars($total['year']); ?></td>
                                        <td><?php echo htmlspecialchars($total['total']); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                    
                    <h4>Yearly Totals</h4>
                    <table class="table table-striped">
                        <thead>
                            <tr>
                                <th>Year</th>
                                <th>Total Amount</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($yearly_totals)): ?>
                                <tr><td colspan="2">No yearly expense data.</td></tr>
                            <?php else: ?>
                                <?php foreach ($yearly_totals as $total): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($total['year']); ?></td>
                                        <td><?php echo htmlspecialchars($total['total']); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <div class="card">
                <div class="card-header">Expense History <a href="?export=csv" class="btn btn-sm btn-secondary float-right">Export to CSV</a></div>
                <div class="card-body">
                    <table class="table table-striped">
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th>Type</th>
                                <th>Amount</th>
                                <th>Description</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (mysqli_num_rows($result_history) > 0): ?>
                                <?php while ($row = mysqli_fetch_assoc($result_history)): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($row['created_at']); ?></td>
                                        <td><?php echo htmlspecialchars($row['type_name']); ?></td>
                                        <td><?php echo htmlspecialchars($row['amount']); ?></td>
                                        <td><?php echo htmlspecialchars($row['description']); ?></td>
                                    </tr>
                                <?php endwhile; ?>
                            <?php else: ?>
                                <tr>
                                    <td colspan="4">No expenses found.</td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </main>
    </div>
</div>
<?php require_once 'includes/footer.php'; ?>