<?php
require_once 'includes/functions.php';
require_once 'includes/db_config.php';

// Check if user is logged in and is an admin
if (!is_logged_in() || !is_admin()) {
    redirect("login.php");
}

$message = '';
$edit_id = null;
$edit_title = '';
$edit_content = '';

// Handle form submission (add/update)
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $title = sanitize_input($_POST['title']);
    $content = sanitize_input($_POST['content']);
    $user_id = $_SESSION['user_id'];

    if (isset($_POST['announcement_id']) && !empty($_POST['announcement_id'])) {
        // Update existing announcement
        $announcement_id = intval($_POST['announcement_id']);
        $sql = "UPDATE announcements SET title = ?, content = ? WHERE id = ?";
        if ($stmt = mysqli_prepare($link, $sql)) {
            mysqli_stmt_bind_param($stmt, "ssi", $title, $content, $announcement_id);
            if (mysqli_stmt_execute($stmt)) {
                $message = "Announcement updated successfully!";
            } else {
                $message = "Error updating announcement.";
            }
            mysqli_stmt_close($stmt);
        }
    } else {
        // Add new announcement
        $sql = "INSERT INTO announcements (title, content, author_id, created_at) VALUES (?, ?, ?, NOW())";
        if ($stmt = mysqli_prepare($link, $sql)) {
            mysqli_stmt_bind_param($stmt, "ssi", $title, $content, $user_id);
            if (mysqli_stmt_execute($stmt)) {
                $message = "Announcement created successfully!";
            } else {
                $message = "Error creating announcement.";
            }
            mysqli_stmt_close($stmt);
        }
    }
}

// Handle delete action
if (isset($_GET['delete_id'])) {
    $delete_id = intval($_GET['delete_id']);
    $sql = "DELETE FROM announcements WHERE id = ?";
    if ($stmt = mysqli_prepare($link, $sql)) {
        mysqli_stmt_bind_param($stmt, "i", $delete_id);
        if (mysqli_stmt_execute($stmt)) {
            $message = "Announcement deleted successfully!";
        } else {
            $message = "Error deleting announcement.";
        }
        mysqli_stmt_close($stmt);
    }
}

// Handle edit action
if (isset($_GET['edit_id'])) {
    $edit_id = intval($_GET['edit_id']);
    $sql = "SELECT title, content FROM announcements WHERE id = ?";
    if ($stmt = mysqli_prepare($link, $sql)) {
        mysqli_stmt_bind_param($stmt, "i", $edit_id);
        mysqli_stmt_execute($stmt);
        mysqli_stmt_bind_result($stmt, $edit_title, $edit_content);
        mysqli_stmt_fetch($stmt);
        mysqli_stmt_close($stmt);
    }
}
?>

<?php require_once 'includes/header.php'; ?>
<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Manage Announcements</h2>

            <?php if ($message): ?>
                <div class="message <?php echo strpos($message, 'Error') !== false ? 'error-message' : 'success-message'; ?>">
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>

            <div class="form-section">
                <h3><?php echo $edit_id ? 'Edit Announcement' : 'Create New Announcement'; ?></h3>
                <form action="announcement_management.php" method="post">
                    <input type="hidden" name="announcement_id" value="<?php echo htmlspecialchars($edit_id); ?>">
                    <div class="form-group">
                        <label for="title">Title</label>
                        <input type="text" name="title" id="title" class="form-control" value="<?php echo htmlspecialchars($edit_title); ?>" required>
                    </div>
                    <div class="form-group">
                        <label for="content">Content</label>
                        <textarea name="content" id="content" class="form-control" rows="5" required><?php echo htmlspecialchars($edit_content); ?></textarea>
                    </div>
                    <button type="submit" class="btn btn-primary">Save Announcement</button>
                </form>
            </div>

            <hr>

            <div class="list-section">
                <h3>Existing Announcements</h3>
                <table class="announcement-table">
                    <thead>
                        <tr>
                            <th>Title</th>
                            <th>Posted By</th>
                            <th>Date</th>
                            <th>Views</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php
                        $sql = "SELECT a.id, a.title, a.created_at, a.views_count, u.full_name
                                FROM announcements a
                                JOIN users u ON a.author_id = u.id
                                ORDER BY a.created_at DESC";
                        $result = mysqli_query($link, $sql);
                        if ($result && mysqli_num_rows($result) > 0) {
                            while ($row = mysqli_fetch_assoc($result)) {
                                echo "<tr>";
                                echo "<td>" . htmlspecialchars($row['title'] ?? '') . "</td>";
                                echo "<td>" . htmlspecialchars($row['full_name'] ?? '') . "</td>";
                                echo "<td>" . $row['created_at'] . "</td>";
                                echo "<td>" . ($row['views_count'] ?? '0') . "</td>";
                                echo "<td>";
                                echo "<a href='announcement_management.php?edit_id=" . $row['id'] . "' class='btn btn-secondary btn-sm'>Edit</a> ";
                                echo "<a href='announcement_management.php?delete_id=" . $row['id'] . "' class='btn btn-danger btn-sm' onclick='return confirm(\"Are you sure?\");'>Delete</a>";
                                echo "</td>";
                                echo "</tr>";
                            }
                        } else {
                            echo "<tr><td colspan='5'>No announcements found.</td></tr>";
                        }
                        ?>
                    </tbody>
                </table>
            </div>
        </main>
    </div>
</div>
<?php require_once 'includes/footer.php'; ?>