<?php
// token_settings.php
ob_start();
session_start();

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once 'includes/functions.php';
require_once 'includes/db_config.php';

// Session timeout
$timeout = 1800; // 30 minutes
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity']) > $timeout) {
    session_unset();
    session_destroy();
    redirect("login.php?timeout=1");
}
$_SESSION['last_activity'] = time();

// Restrict to super_admin role
if (!is_logged_in() || $_SESSION['user_role'] !== 'super_admin') {
    redirect("login.php");
}

// Generate CSRF token
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

$message = '';
$message_type = '';
$currency_symbol = get_setting('currency_symbol', $link) ?? '₦';

// Handle token package submission
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['save_package']) && isset($_POST['csrf_token']) && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
    $package_id = isset($_POST['package_id']) ? intval($_POST['package_id']) : 0;
    $package_name = sanitize_input($_POST['package_name'] ?? '');
    $quantity = intval($_POST['quantity'] ?? 0);
    $price = floatval($_POST['price'] ?? 0);
    $status = sanitize_input($_POST['status'] ?? 'active');

    if (empty($package_name) || $quantity <= 0 || $price < 0 || !in_array($status, ['active', 'inactive'])) {
        $message = "Please fill in all fields with valid values for token package.";
        $message_type = 'danger';
    } else {
        if ($package_id > 0) {
            // Update existing package
            $sql = "UPDATE token_packages SET package_name = ?, quantity = ?, price = ?, status = ? WHERE id = ? AND created_by = ?";
            if ($stmt = mysqli_prepare($link, $sql)) {
                mysqli_stmt_bind_param($stmt, "sidsii", $package_name, $quantity, $price, $status, $package_id, $_SESSION['user_id']);
                if (mysqli_stmt_execute($stmt)) {
                    $message = "Token package updated successfully.";
                    $message_type = 'success';
                } else {
                    $message = "Error updating package: " . mysqli_error($link);
                    $message_type = 'danger';
                    error_log("Error updating token package in token_settings.php: " . mysqli_error($link));
                }
                mysqli_stmt_close($stmt);
            } else {
                $message = "Error preparing statement: " . mysqli_error($link);
                $message_type = 'danger';
                error_log("Error preparing statement in token_settings.php: " . mysqli_error($link));
            }
        } else {
            // Add new package
            $sql = "INSERT INTO token_packages (package_name, quantity, price, status, created_by, created_at) VALUES (?, ?, ?, ?, ?, NOW())";
            if ($stmt = mysqli_prepare($link, $sql)) {
                mysqli_stmt_bind_param($stmt, "sidsi", $package_name, $quantity, $price, $status, $_SESSION['user_id']);
                if (mysqli_stmt_execute($stmt)) {
                    $message = "Token package added successfully.";
                    $message_type = 'success';
                } else {
                    $message = "Error adding package: " . mysqli_error($link);
                    $message_type = 'danger';
                    error_log("Error adding token package in token_settings.php: " . mysqli_error($link));
                }
                mysqli_stmt_close($stmt);
            } else {
                $message = "Error preparing statement: " . mysqli_error($link);
                $message_type = 'danger';
                error_log("Error preparing statement in token_settings.php: " . mysqli_error($link));
            }
        }
    }
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Handle token cost submission
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['save_token_costs']) && isset($_POST['csrf_token']) && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
    $tokens_per_payment = intval($_POST['tokens_per_payment'] ?? 1);
    $tokens_per_expense = intval($_POST['tokens_per_expense'] ?? 1);
    $tokens_per_benefit = intval($_POST['tokens_per_benefit'] ?? 1);
    $deduct_per_member = intval($_POST['deduct_per_member'] ?? 0); // New toggle

    if ($tokens_per_payment < 0 || $tokens_per_expense < 0 || $tokens_per_benefit < 0) {
        $message = "Token costs must be non-negative.";
        $message_type = 'danger';
    } else {
        $settings = [
            'tokens_per_payment' => $tokens_per_payment,
            'tokens_per_expense' => $tokens_per_expense,
            'tokens_per_benefit' => $tokens_per_benefit,
            'deduct_per_member' => $deduct_per_member
        ];
        $save_success = true;
        foreach ($settings as $key => $value) {
            if (!update_setting($link, $key, $value)) {
                $message = "Error saving $key: " . mysqli_error($link);
                $message_type = 'danger';
                error_log("Error saving $key in token_settings.php: " . mysqli_error($link));
                $save_success = false;
                break;
            }
        }
        if ($save_success) {
            $message = "Token costs saved successfully.";
            $message_type = 'success';
        }
    }
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
} elseif ($_SERVER["REQUEST_METHOD"] == "POST" && !isset($_POST['save_package'])) {
    $message = "Invalid CSRF token.";
    $message_type = 'danger';
    error_log("CSRF token validation failed in token_settings.php");
}

// Fetch existing token packages
$token_packages = [];
$sql = "SELECT id, package_name, quantity, price, status FROM token_packages WHERE created_by = ? ORDER BY created_at DESC";
if ($stmt = mysqli_prepare($link, $sql)) {
    mysqli_stmt_bind_param($stmt, "i", $_SESSION['user_id']);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    while ($row = mysqli_fetch_assoc($result)) {
        $token_packages[] = $row;
    }
    mysqli_free_result($result);
    mysqli_stmt_close($stmt);
} else {
    $message = "Error fetching token packages: " . mysqli_error($link);
    $message_type = 'danger';
    error_log("Failed to fetch token packages in token_settings.php: " . mysqli_error($link));
}

// Fetch token costs
$tokens_per_payment = get_setting('tokens_per_payment', $link) ?? 1;
$tokens_per_expense = get_setting('tokens_per_expense', $link) ?? 1;
$tokens_per_benefit = get_setting('tokens_per_benefit', $link) ?? 1;
$deduct_per_member = get_setting('deduct_per_member', $link) ?? 0;

ob_end_clean();
?>

<?php require_once 'includes/header.php'; ?>

<script>
// Set dynamic page title
document.title = 'Token Settings - <?php echo htmlspecialchars(get_setting('system_name', $link) ?? 'Club Portal'); ?>';
</script>

<style>
.main-content {
    padding: 20px;
}
</style>

<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Token Settings</h2>

            <?php if ($message): ?>
                <div class="alert alert-<?php echo htmlspecialchars($message_type); ?>" role="alert">
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>

            <!-- Add/Edit Token Package Form -->
            <div class="card mb-4">
                <div class="card-header">Add/Edit Token Package</div>
                <div class="card-body">
                    <form action="token_settings.php" method="post">
                        <input type="hidden" name="save_package" value="1">
                        <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">
                        <input type="hidden" name="package_id" id="package_id" value="">
                        <div class="mb-3">
                            <label for="package_name" class="form-label">Package Name</label>
                            <input type="text" name="package_name" id="package_name" class="form-control" required>
                        </div>
                        <div class="mb-3">
                            <label for="quantity" class="form-label">Token Quantity</label>
                            <input type="number" name="quantity" id="quantity" class="form-control" min="1" required>
                        </div>
                        <div class="mb-3">
                            <label for="price" class="form-label">Price (<?php echo htmlspecialchars($currency_symbol); ?>)</label>
                            <input type="number" name="price" id="price" class="form-control" min="0" step="0.01" required>
                        </div>
                        <div class="mb-3">
                            <label for="status" class="form-label">Status</label>
                            <select name="status" id="status" class="form-select" required>
                                <option value="active">Active</option>
                                <option value="inactive">Inactive</option>
                            </select>
                        </div>
                        <button type="submit" class="btn btn-primary">Save Package</button>
                    </form>
                </div>
            </div>

            <!-- Token Costs Form -->
            <div class="card mb-4">
                <div class="card-header">Set Token Costs per Record</div>
                <div class="card-body">
                    <form action="token_settings.php" method="post">
                        <input type="hidden" name="save_token_costs" value="1">
                        <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">
                        <div class="mb-3">
                            <label for="tokens_per_payment" class="form-label">Tokens per Payment Record</label>
                            <input type="number" name="tokens_per_payment" id="tokens_per_payment" class="form-control" min="0" value="<?php echo htmlspecialchars($tokens_per_payment); ?>" required>
                        </div>
                        <div class="mb-3">
                            <label for="tokens_per_expense" class="form-label">Tokens per Expense Record</label>
                            <input type="number" name="tokens_per_expense" id="tokens_per_expense" class="form-control" min="0" value="<?php echo htmlspecialchars($tokens_per_expense); ?>" required>
                        </div>
                        <div class="mb-3">
                            <label for="tokens_per_benefit" class="form-label">Tokens per Benefit Record</label>
                            <input type="number" name="tokens_per_benefit" id="tokens_per_benefit" class="form-control" min="0" value="<?php echo htmlspecialchars($tokens_per_benefit); ?>" required>
                        </div>
                        <div class="mb-3">
                            <label for="deduct_per_member" class="form-label">Deduct per Member (0=per log, 1=per member)</label>
                            <input type="number" name="deduct_per_member" id="deduct_per_member" class="form-control" min="0" max="1" value="<?php echo htmlspecialchars($deduct_per_member); ?>" required>
                        </div>
                        <button type="submit" class="btn btn-primary">Save Token Costs</button>
                    </form>
                </div>
            </div>

            <!-- Existing Token Packages -->
            <div class="card">
                <div class="card-header">Existing Token Packages</div>
                <div class="card-body">
                    <?php if (empty($token_packages)): ?>
                        <p class="text-muted">No token packages found.</p>
                    <?php else: ?>
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>Package Name</th>
                                    <th>Quantity</th>
                                    <th>Price (<?php echo htmlspecialchars($currency_symbol); ?>)</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($token_packages as $package): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($package['package_name']); ?></td>
                                        <td><?php echo htmlspecialchars($package['quantity']); ?></td>
                                        <td><?php echo htmlspecialchars(number_format($package['price'], 2)); ?></td>
                                        <td><?php echo htmlspecialchars(ucfirst($package['status'])); ?></td>
                                        <td>
                                            <button class="btn btn-sm btn-primary edit-package" 
                                                    data-id="<?php echo htmlspecialchars($package['id']); ?>"
                                                    data-name="<?php echo htmlspecialchars($package['package_name']); ?>"
                                                    data-quantity="<?php echo htmlspecialchars($package['quantity']); ?>"
                                                    data-price="<?php echo htmlspecialchars($package['price']); ?>"
                                                    data-status="<?php echo htmlspecialchars($package['status']); ?>">
                                                Edit
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function () {
    // Handle edit button clicks
    const editButtons = document.querySelectorAll('.edit-package');
    editButtons.forEach(button => {
        button.addEventListener('click', function () {
            document.getElementById('package_id').value = this.dataset.id;
            document.getElementById('package_name').value = this.dataset.name;
            document.getElementById('quantity').value = this.dataset.quantity;
            document.getElementById('price').value = this.dataset.price;
            document.getElementById('status').value = this.dataset.status;
        });
    });
});
</script>

<?php require_once 'includes/footer.php'; ?>