<?php
// record_expenses.php
ob_start();
session_start();

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Enable mysqli exceptions for better error handling
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

require_once 'includes/functions.php';
require_once 'includes/db_config.php';

// Session timeout
$timeout = 1800; // 30 minutes
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity']) > $timeout) {
    session_unset();
    session_destroy();
    redirect("login.php?timeout=1");
}
$_SESSION['last_activity'] = time();

// Generate CSRF token
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Check if user is logged in and has view_expenses permission
if (!is_logged_in() || !user_has_permission($link, 'view_expenses')) {
    redirect("login.php");
    exit;
}

$message = '';
$message_type = '';
$currency_symbol = get_setting('currency_symbol', $link) ?? '₦';
$token_cost = (int)get_setting('tokens_per_expense', $link) ?: 1;

// NEW: Fetch system available tokens (matches dashboard)
$system_available_tokens = 0;
$sql_available_tokens = "SELECT COALESCE(SUM(balance), 0) AS total FROM token_balances";
$result = mysqli_query($link, $sql_available_tokens);
if ($result) {
    $system_available_tokens = mysqli_fetch_assoc($result)['total'] ?? 0;
    mysqli_free_result($result);
} else {
    error_log("Failed to fetch system available tokens: " . mysqli_error($link));
}

// Handle form submission
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['log_expense']) && isset($_POST['csrf_token']) && hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
    $amount = floatval($_POST['amount'] ?? 0);
    $expense_type_id = intval($_POST['expense_type_id'] ?? 0);
    $description = sanitize_input($_POST['description'] ?? '');
    $current_user_id = $_SESSION['user_id'];

    // Validate inputs
    if ($amount > 0 && $expense_type_id > 0) {
        // Check token balance
        $required_tokens = $token_cost; // Flat for expenses (single log)
        $current_balance = get_token_balance($link, $current_user_id);
        if ($current_balance < $required_tokens) {
            $message = "Insufficient tokens. You need $required_tokens token" . ($required_tokens > 1 ? 's' : '') . ", but you have $current_balance. <a href='token_store.php'>Please purchase more tokens</a>.";
            $message_type = 'danger';
        } else {
            // Verify expense type exists
            $sql_check_type = "SELECT id FROM expense_types WHERE id = ? AND deleted_at IS NULL";
            $stmt_check = mysqli_prepare($link, $sql_check_type);
            if ($stmt_check) {
                mysqli_stmt_bind_param($stmt_check, "i", $expense_type_id);
                mysqli_stmt_execute($stmt_check);
                mysqli_stmt_store_result($stmt_check);
                
                if (mysqli_stmt_num_rows($stmt_check) > 0) {
                    
                    // *** BEGIN TRANSACTION BLOCK ***
                    mysqli_begin_transaction($link);
                    $transaction_success = true;
                    
                    // 1. Deduct tokens
                    $tokens_used = deduct_tokens($link, $current_user_id, 'expense', $required_tokens);
                    if ($tokens_used !== false) {
                        // 2. Insert expense
                        $sql = "INSERT INTO expenses (expense_type_id, amount, description, created_at) VALUES (?, ?, ?, NOW())";
                        $stmt = mysqli_prepare($link, $sql);
                        if ($stmt) {
                            mysqli_stmt_bind_param($stmt, "ids", $expense_type_id, $amount, $description);
                            if (!mysqli_stmt_execute($stmt)) {
                                $transaction_success = false;
                                $message = "Error logging expense: " . mysqli_stmt_error($stmt);
                                $message_type = 'danger';
                                error_log("Expense insert failed: " . mysqli_stmt_error($stmt));
                            }
                            mysqli_stmt_close($stmt);
                        } else {
                            $transaction_success = false;
                            $message = "Database error: " . mysqli_error($link);
                            $message_type = 'danger';
                            error_log("Failed to prepare expense insertion: " . mysqli_error($link));
                        }
                    } else {
                        // Token deduction failed
                        $transaction_success = false;
                        $message = "Failed to deduct tokens. <a href='token_store.php'>Please purchase more</a>.";
                        $message_type = 'danger';
                    }

                    // *** END TRANSACTION BLOCK ***
                    if ($transaction_success) {
                        mysqli_commit($link);
                        $message = "Expense logged successfully! Used $tokens_used token" . ($tokens_used > 1 ? 's' : '') . ".";
                        $message_type = 'success';
                    } else {
                        mysqli_rollback($link);
                        if (empty($message)) {
                            $message = "Transaction failed. No expense was logged and no tokens were deducted.";
                            $message_type = 'danger';
                        }
                    }

                } else {
                    $message = "Invalid expense type selected.";
                    $message_type = 'danger';
                }
                mysqli_stmt_close($stmt_check);
            } else {
                $message = "Database error: " . mysqli_error($link);
                $message_type = 'danger';
                error_log("Failed to prepare expense type check: " . mysqli_error($link));
            }
        }
    } else {
        $message = "Please provide a valid amount and expense type.";
        $message_type = 'danger';
    }
    // Regenerate CSRF token
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
} elseif ($_SERVER["REQUEST_METHOD"] == "POST") {
    $message = "Invalid CSRF token.";
    $message_type = 'danger';
    error_log("CSRF token validation failed in record_expenses.php");
}

// ... rest of the file (fetching expense types, HTML) remains the same ...
// Fetch expense types (using IDs)
$expense_types = [];
$sql_expense_types = "SELECT id, type_name FROM expense_types WHERE deleted_at IS NULL ORDER BY type_name ASC";
$result_expense_types = mysqli_query($link, $sql_expense_types);
if ($result_expense_types) {
    while ($row = mysqli_fetch_assoc($result_expense_types)) {
        $expense_types[] = $row;
    }
    mysqli_free_result($result_expense_types);
}
?>

<?php require_once 'includes/header.php'; ?>

<script>
document.title = 'Record Expenses - <?php echo htmlspecialchars(get_setting('system_name', $link) ?? 'Club Portal'); ?>';
</script>

<style>
.main-content {
    padding: 20px;
}
</style>

<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Record Expenses</h2>

            <?php if ($message): ?>
                <div class="alert alert-<?php echo htmlspecialchars($message_type); ?>" role="alert">
                    <?php echo $message; ?>
                </div>
            <?php endif; ?>

            <div class="card mb-4">
                <div class="card-header">Log New Expense</div>
                <div class="card-body">
                    <p>System Available Tokens: <?php echo htmlspecialchars($system_available_tokens); ?> tokens</p>
                    <form action="record_expenses.php" method="post">
                        <input type="hidden" name="log_expense" value="1">
                        <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">
                        <div class="mb-3">
                            <label for="expense_type_id" class="form-label">Expense Type</label>
                            <select name="expense_type_id" id="expense_type_id" class="form-select" required>
                                <option value="">Select Expense Type</option>
                                <?php foreach ($expense_types as $type): ?>
                                    <option value="<?php echo htmlspecialchars($type['id']); ?>">
                                        <?php echo htmlspecialchars($type['type_name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label for="amount" class="form-label">Amount (<?php echo htmlspecialchars($currency_symbol); ?>)</label>
                            <input type="number" name="amount" id="amount" class="form-control" step="0.01" min="0.01" required>
                        </div>
                        <div class="mb-3">
                            <label for="description" class="form-label">Description (Optional)</label>
                            <textarea name="description" id="description" class="form-control" rows="3"></textarea>
                        </div>
                        <button type="submit" class="btn btn-primary">Log Expense (<?php echo $token_cost; ?> token<?php echo $token_cost > 1 ? 's' : ''; ?>)</button>
                    </form>
                </div>
            </div>
        </main>
    </div>
</div>

<?php require_once 'includes/footer.php'; ?>