<?php
require_once 'includes/functions.php';
require_once 'includes/db_config.php';
session_start();

// Enable error logging
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
ini_set('log_errors', 1);
ini_set('error_log', 'logs/php_errors.log');
error_reporting(E_ALL);

// Check if user is logged in
if (!is_logged_in()) {
    redirect("login.php");
}

// Get current user ID and role from session
$user_id = $_SESSION['user_id'];
$_SESSION['user_role'] = $_SESSION['user_role'] ?? 'member'; // Ensure role is set

// Ensure database connection is open
if (!isset($link) || !mysqli_ping($link)) {
    include 'includes/db_config.php';
    if (!$link) {
        $error_message = "Error reconnecting to database: " . mysqli_connect_error();
        $error_type = 'danger';
    }
}

// Fetch current user data
$sql = "SELECT full_name, email, membership_status, profile_picture, role FROM users WHERE id = ? AND (deleted_at IS NULL OR deleted_at > NOW())";
if ($stmt = mysqli_prepare($link, $sql)) {
    mysqli_stmt_bind_param($stmt, "i", $user_id);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    $user = mysqli_fetch_assoc($result);
    mysqli_stmt_close($stmt);
    // Update session role to match database
    $_SESSION['user_role'] = $user['role'] ?? $_SESSION['user_role'];
} else {
    $error_message = "Error fetching user data: " . mysqli_error($link);
    $error_type = 'danger';
    $user = ['full_name' => '', 'email' => '', 'membership_status' => '', 'profile_picture' => '', 'role' => $_SESSION['user_role']];
}

// Handle profile update form submission
$success_message = '';
$error_message = '';
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $full_name = sanitize_input($_POST['full_name'] ?? '');
    $email = sanitize_input($_POST['email'] ?? '');
    $membership_status = $user['role'] === 'admin' ? sanitize_input($_POST['membership_status'] ?? $user['membership_status']) : $user['membership_status'];

    // Validate inputs
    if (empty($full_name) || strlen($full_name) < 2 || strlen($full_name) > 255) {
        $error_message = "Full name must be between 2 and 255 characters.";
    } elseif (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error_message = "Invalid email format.";
    } else {
        // Handle profile picture upload
        $profile_picture = $user['profile_picture'];
        if (isset($_FILES['profile_picture']) && $_FILES['profile_picture']['error'] === UPLOAD_ERR_OK) {
            $upload_dir = 'assets/images/';
            $allowed_types = ['image/jpeg', 'image/png', 'image/gif'];
            $max_size = 5 * 1024 * 1024; // 5MB
            $file = $_FILES['profile_picture'];

            // Validate file
            $check = getimagesize($file['tmp_name']);
            if ($check === false) {
                $error_message = "File is not an image.";
            } elseif (!in_array($file['type'], $allowed_types)) {
                $error_message = "Only JPG, JPEG, PNG, or GIF files are allowed.";
            } elseif ($file['size'] > $max_size) {
                $error_message = "File is too large. Maximum size is 5MB.";
            } else {
                $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
                $file_name = "user_{$user_id}_profile_" . uniqid() . ".$ext";
                $target_file = $upload_dir . $file_name;

                // Delete old profile picture if it exists
                if ($profile_picture && file_exists($profile_picture)) {
                    unlink($profile_picture);
                }

                if (move_uploaded_file($file['tmp_name'], $target_file)) {
                    $profile_picture = $target_file;
                } else {
                    $error_message = "Error uploading file.";
                }
            }
        } elseif ($_FILES['profile_picture']['error'] !== UPLOAD_ERR_NO_FILE) {
            if ($_FILES['profile_picture']['error'] === UPLOAD_ERR_INI_SIZE) {
                $error_message = "File is too large for the server configuration. Please upload a file smaller than " . ini_get('upload_max_filesize') . ".";
            } else {
                $error_message = "File upload error: " . $_FILES['profile_picture']['error'];
            }
        }

        // Update database if no errors
        if (empty($error_message)) {
            $sql = $profile_picture !== $user['profile_picture'] ?
                "UPDATE users SET full_name = ?, email = ?, membership_status = ?, profile_picture = ? WHERE id = ?" :
                "UPDATE users SET full_name = ?, email = ?, membership_status = ? WHERE id = ?";
            if ($stmt = mysqli_prepare($link, $sql)) {
                if ($profile_picture !== $user['profile_picture']) {
                    mysqli_stmt_bind_param($stmt, "ssssi", $full_name, $email, $membership_status, $profile_picture, $user_id);
                } else {
                    mysqli_stmt_bind_param($stmt, "sssi", $full_name, $email, $membership_status, $user_id);
                }
                if (mysqli_stmt_execute($stmt)) {
                    $success_message = "Profile updated successfully!";
                    // Update session data
                    $_SESSION['full_name'] = $full_name;
                    // Refresh user data
                    $sql = "SELECT full_name, email, membership_status, profile_picture, role FROM users WHERE id = ?";
                    if ($stmt_refresh = mysqli_prepare($link, $sql)) {
                        mysqli_stmt_bind_param($stmt_refresh, "i", $user_id);
                        mysqli_stmt_execute($stmt_refresh);
                        $result = mysqli_stmt_get_result($stmt_refresh);
                        $user = mysqli_fetch_assoc($result);
                        $_SESSION['user_role'] = $user['role'] ?? $_SESSION['user_role'];
                        mysqli_stmt_close($stmt_refresh);
                    }
                } else {
                    $error_message = "Failed to update profile: " . mysqli_error($link);
                }
                mysqli_stmt_close($stmt);
            } else {
                $error_message = "Error preparing update statement: " . mysqli_error($link);
            }
        }
    }
}

// Currency symbol (for consistency)
$currency_symbol = get_setting('currency_symbol', $link) ?? '₦';
?>

<?php require_once 'includes/header.php'; ?>
<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>My Profile</h2>

            <?php if ($success_message): ?>
                <div class="alert alert-success" role="alert"><?php echo htmlspecialchars($success_message); ?></div>
            <?php endif; ?>
            <?php if ($error_message): ?>
                <div class="alert alert-danger" role="alert"><?php echo htmlspecialchars($error_message); ?></div>
            <?php endif; ?>

            <div class="card mb-4">
                <div class="card-header">Profile Details</div>
                <div class="card-body">
                    <form action="profile.php" method="post" enctype="multipart/form-data">
                        <div class="row g-3">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="full_name" class="form-label">Full Name</label>
                                    <input type="text" name="full_name" id="full_name" value="<?php echo htmlspecialchars($user['full_name']); ?>" class="form-control" required>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="email" class="form-label">Email Address</label>
                                    <input type="email" name="email" id="email" value="<?php echo htmlspecialchars($user['email'] ?? ''); ?>" class="form-control">
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="membership_status" class="form-label">Membership Status</label>
                                    <input type="text" name="membership_status" id="membership_status" value="<?php echo htmlspecialchars(ucfirst($user['membership_status'])); ?>" class="form-control" <?php echo $user['role'] !== 'admin' ? 'disabled' : ''; ?>>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="profile_picture" class="form-label">Update Profile Picture</label>
                                    <input type="file" name="profile_picture" id="profile_picture" accept="image/png,image/jpeg,image/gif" class="form-control">
                                    <small class="form-text text-muted">Allowed types: JPG, JPEG, PNG, GIF. Max size: <?php echo ini_get('upload_max_filesize'); ?>.</small>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Current Profile Picture</label><br>
                                    <?php if ($user['profile_picture'] && file_exists($user['profile_picture'])): ?>
                                        <img src="<?php echo htmlspecialchars($user['profile_picture']); ?>" alt="Profile Picture" style="max-width: 150px; max-height: 150px;">
                                    <?php else: ?>
                                        <img src="assets/images/default_profile.jpg" alt="Default Profile Picture" style="max-width: 150px; max-height: 150px;">
                                        <p>No profile picture uploaded.</p>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <div class="col-12">
                                <button type="submit" class="btn btn-primary mt-2">Save Changes</button>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </main>
    </div>
</div>
<?php require_once 'includes/footer.php'; ?>