<?php
// my_payments.php
// Start session to access user data
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Enable error reporting for development (remove in production)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include dependencies
require_once 'includes/functions.php';
require_once 'includes/db_config.php';

// Log file version for debugging
error_log("my_payments.php version: 2025-10-07-v7-obligation-id");

// Check if user is logged in
if (!is_logged_in()) {
    redirect("login.php");
    exit;
}

$is_admin = is_admin();
$user_id = $_SESSION['user_id'];
$currency_symbol = get_setting('currency_symbol', $link) ?? '₦';

// Verify payment_obligations table schema
$schema_check = mysqli_query($link, "DESCRIBE payment_obligations");
$columns = [];
if ($schema_check) {
    while ($row = mysqli_fetch_assoc($schema_check)) {
        $columns[] = $row['Field'];
    }
    error_log("payment_obligations columns: " . implode(', ', $columns));
} else {
    error_log("Failed to check payment_obligations schema: " . mysqli_error($link));
}

// Check for non-NULL required_amount values
$null_check = mysqli_query($link, "SELECT COUNT(*) AS null_count FROM payment_obligations WHERE required_amount IS NULL");
$null_count = $null_check ? mysqli_fetch_assoc($null_check)['null_count'] : 0;
if ($null_count > 0) {
    error_log("Warning: $null_count records in payment_obligations have NULL required_amount");
}

// Filter settings
$record_type = isset($_GET['record_type']) && in_array($_GET['record_type'], ['all', 'payments', 'debts', 'benefits']) ? $_GET['record_type'] : 'all';
$time_period = isset($_GET['time_period']) && in_array($_GET['time_period'], ['date', 'week', 'month', 'year', 'all']) ? $_GET['time_period'] : 'all';
$filter_date = isset($_GET['filter_date']) && !empty($_GET['filter_date']) ? $_GET['filter_date'] : date('Y-m-d');

// Build time filter for queries
$time_filter = '';
$time_filter_debts = '';
$bind_params = [];
$bind_types = '';
if ($time_period !== 'all') {
    switch ($time_period) {
        case 'date':
            $time_filter = " AND DATE(p.created_at) = ?";
            $time_filter_debts = " AND DATE(po.created_at) = ?";
            $bind_params[] = $filter_date;
            $bind_types .= 's';
            break;
        case 'week':
            $time_filter = " AND p.created_at >= DATE_SUB(CURDATE(), INTERVAL 1 WEEK)";
            $time_filter_debts = " AND po.created_at >= DATE_SUB(CURDATE(), INTERVAL 1 WEEK)";
            break;
        case 'month':
            $time_filter = " AND p.created_at >= DATE_SUB(CURDATE(), INTERVAL 1 MONTH)";
            $time_filter_debts = " AND po.created_at >= DATE_SUB(CURDATE(), INTERVAL 1 MONTH)";
            break;
        case 'year':
            $time_filter = " AND p.created_at >= DATE_SUB(CURDATE(), INTERVAL 1 YEAR)";
            $time_filter_debts = " AND po.created_at >= DATE_SUB(CURDATE(), INTERVAL 1 YEAR)";
            break;
    }
}
error_log("Filter settings - record_type: '$record_type', time_period: '$time_period', bind_params_count: " . count($bind_params));

// Pagination settings
$tables = ['payments', 'debts', 'benefits'];
$pagination = [];
foreach ($tables as $table) {
    $pagination[$table] = [
        'per_page' => isset($_GET["per_page_$table"]) && in_array($_GET["per_page_$table"], [5, 10, 25, 50, 100]) ? (int)$_GET["per_page_$table"] : 5,
        'page' => isset($_GET["page_$table"]) && is_numeric($_GET["page_$table"]) && $_GET["page_$table"] > 0 ? (int)$_GET["page_$table"] : 1
    ];
    $pagination[$table]['offset'] = ($pagination[$table]['page'] - 1) * $pagination[$table]['per_page'];
}

// Debug array to store query results (for admin only)
$debug_info = [];

// Total Payments
$sql_total_payments = "SELECT COALESCE(SUM(amount), 0) AS total_payments
                       FROM payments
                       WHERE user_id = ? AND payment_status = 'Paid'" . ($record_type == 'all' || $record_type == 'payments' ? $time_filter : '');
$stmt = mysqli_prepare($link, $sql_total_payments);
if ($stmt) {
    $bind_types_total = 'i' . ($record_type == 'all' || $record_type == 'payments' ? $bind_types : '');
    $bind_params_total = array_merge([&$user_id], ($record_type == 'all' || $record_type == 'payments' ? $bind_params : []));
    error_log("Total payments query - bind_types: '$bind_types_total', params count: " . count($bind_params_total));
    mysqli_stmt_bind_param($stmt, $bind_types_total, ...$bind_params_total);
    if (!mysqli_stmt_execute($stmt)) {
        error_log("Total payments query execution failed: " . mysqli_stmt_error($stmt));
        $total_payments = 0;
    } else {
        $result = mysqli_stmt_get_result($stmt);
        $total_payments = $result ? mysqli_fetch_assoc($result)['total_payments'] ?? 0 : 0;
        $debug_info['total_payments_query'] = $sql_total_payments;
        $debug_info['total_payments'] = $total_payments;
    }
    mysqli_stmt_close($stmt);
} else {
    error_log("Failed to prepare total payments query: " . mysqli_error($link));
    $total_payments = 0;
}

// Total Benefits
$sql_total_benefits = "SELECT COALESCE(SUM(amount), 0) AS total_benefits
                       FROM member_benefits
                       WHERE member_id = ?" . ($record_type == 'all' || $record_type == 'benefits' ? $time_filter : '');
$stmt = mysqli_prepare($link, $sql_total_benefits);
if ($stmt) {
    $bind_types_total = 'i' . ($record_type == 'all' || $record_type == 'benefits' ? $bind_types : '');
    $bind_params_total = array_merge([&$user_id], ($record_type == 'all' || $record_type == 'benefits' ? $bind_params : []));
    error_log("Total benefits query - bind_types: '$bind_types_total', params count: " . count($bind_params_total));
    mysqli_stmt_bind_param($stmt, $bind_types_total, ...$bind_params_total);
    if (!mysqli_stmt_execute($stmt)) {
        error_log("Total benefits query execution failed: " . mysqli_stmt_error($stmt));
        $total_benefits = 0;
    } else {
        $result = mysqli_stmt_get_result($stmt);
        $total_benefits = $result ? mysqli_fetch_assoc($result)['total_benefits'] ?? 0 : 0;
        $debug_info['total_benefits_query'] = $sql_total_benefits;
        $debug_info['total_benefits'] = $total_benefits;
    }
    mysqli_stmt_close($stmt);
} else {
    error_log("Failed to prepare total benefits query: " . mysqli_error($link));
    $total_benefits = 0;
}

// Total Debts
$sql_total_debts = "SELECT COALESCE(SUM(debt), 0) AS total_debt
                    FROM (
                        SELECT po.id, (COALESCE(po.required_amount, 0) - COALESCE(SUM(p.amount), 0)) AS debt
                        FROM payment_obligations po
                        LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status = 'Paid'
                        WHERE po.user_id = ? AND po.deleted_at IS NULL AND po.required_amount IS NOT NULL" . ($record_type == 'all' || $record_type == 'debts' ? $time_filter_debts : '') . "
                        GROUP BY po.id
                        HAVING debt > 0
                    ) AS debtor_summary";
if (in_array('required_amount', $columns)) {
    $stmt = mysqli_prepare($link, $sql_total_debts);
    if ($stmt) {
        $bind_types_total = 'i' . ($record_type == 'all' || $record_type == 'debts' ? $bind_types : '');
        $bind_params_total = array_merge([&$user_id], ($record_type == 'all' || $record_type == 'debts' ? $bind_params : []));
        error_log("Total debts query - bind_types: '$bind_types_total', params count: " . count($bind_params_total));
        mysqli_stmt_bind_param($stmt, $bind_types_total, ...$bind_params_total);
        if (!mysqli_stmt_execute($stmt)) {
            error_log("Total debts query execution failed: " . mysqli_stmt_error($stmt));
            $total_debts = 0;
        } else {
            $result = mysqli_stmt_get_result($stmt);
            $total_debts = $result ? mysqli_fetch_assoc($result)['total_debt'] ?? 0 : 0;
            $debug_info['total_debts_query'] = $sql_total_debts;
            $debug_info['total_debts'] = $total_debts;
        }
        mysqli_stmt_close($stmt);
    } else {
        error_log("Failed to prepare total debts query: " . mysqli_error($link));
        $total_debts = 0;
    }
} else {
    error_log("required_amount column missing in payment_obligations table");
    $total_debts = 0;
    $debug_info['total_debts_error'] = "required_amount column missing in payment_obligations";
}

// Fetch table data
// Payments
$sql_payments = "SELECT 
                    p.payment_type,
                    p.amount,
                    p.description,
                    p.created_at,
                    po.payment_cycle,
                    po.due_date
                 FROM payments p
                 LEFT JOIN payment_obligations po ON p.obligation_id = po.id AND po.deleted_at IS NULL
                 WHERE p.user_id = ? AND p.payment_status = 'Paid'" . ($record_type == 'all' || $record_type == 'payments' ? $time_filter : '') . "
                 ORDER BY p.created_at DESC
                 LIMIT ? OFFSET ?";
$stmt = mysqli_prepare($link, $sql_payments);
if ($stmt) {
    $per_page_payments = (int)$pagination['payments']['per_page'];
    $offset_payments = (int)$pagination['payments']['offset'];
    $bind_types_payments = 'iii';
    $bind_params_payments = [&$user_id, &$per_page_payments, &$offset_payments];
    if ($record_type == 'all' || $record_type == 'payments') {
        $bind_types_payments .= $bind_types;
        $bind_params_payments = array_merge($bind_params_payments, $bind_params);
    }
    error_log("Payments query - bind_types: '$bind_types_payments', params count: " . count($bind_params_payments));
    mysqli_stmt_bind_param($stmt, $bind_types_payments, ...$bind_params_payments);
    if (!mysqli_stmt_execute($stmt)) {
        error_log("Payments query execution failed: " . mysqli_stmt_error($stmt));
        $payments = [];
    } else {
        $result_payments = mysqli_stmt_get_result($stmt);
        $payments = [];
        while ($row = mysqli_fetch_assoc($result_payments)) {
            $payments[] = $row;
        }
        error_log("Payments query returned " . count($payments) . " records");
    }
    mysqli_stmt_close($stmt);
} else {
    error_log("Failed to prepare payments query: " . mysqli_error($link));
    $payments = [];
}

// Payments count
$sql_count_payments = "SELECT COUNT(*) AS total
                       FROM payments p
                       WHERE p.user_id = ? AND p.payment_status = 'Paid'" . ($record_type == 'all' || $record_type == 'payments' ? $time_filter : '');
$stmt = mysqli_prepare($link, $sql_count_payments);
if ($stmt) {
    $bind_types_count = 'i' . ($record_type == 'all' || $record_type == 'payments' ? $bind_types : '');
    $bind_params_count = array_merge([&$user_id], ($record_type == 'all' || $record_type == 'payments' ? $bind_params : []));
    error_log("Payments count query - bind_types: '$bind_types_count', params count: " . count($bind_params_count));
    mysqli_stmt_bind_param($stmt, $bind_types_count, ...$bind_params_count);
    if (!mysqli_stmt_execute($stmt)) {
        error_log("Payments count query execution failed: " . mysqli_stmt_error($stmt));
        $total_records_payments = 0;
        $total_pages_payments = 1;
    } else {
        $result = mysqli_stmt_get_result($stmt);
        $total_records_payments = $result ? mysqli_fetch_assoc($result)['total'] : 0;
        $total_pages_payments = ceil($total_records_payments / $pagination['payments']['per_page']);
    }
    mysqli_stmt_close($stmt);
} else {
    error_log("Failed to prepare payments count query: " . mysqli_error($link));
    $total_records_payments = 0;
    $total_pages_payments = 1;
}

// Benefits
$sql_benefits = "SELECT 
                    bt.type_name AS benefit_type,
                    b.amount,
                    b.created_at,
                    b.benefit_date,
                    b.benefit_status
                 FROM member_benefits b
                 JOIN benefit_types bt ON b.benefit_type_id = bt.id
                 WHERE b.member_id = ?" . ($record_type == 'all' || $record_type == 'benefits' ? $time_filter : '') . "
                 ORDER BY b.created_at DESC
                 LIMIT ? OFFSET ?";
$stmt = mysqli_prepare($link, $sql_benefits);
if ($stmt) {
    $per_page_benefits = (int)$pagination['benefits']['per_page'];
    $offset_benefits = (int)$pagination['benefits']['offset'];
    $bind_types_benefits = 'iii';
    $bind_params_benefits = [&$user_id, &$per_page_benefits, &$offset_benefits];
    if ($record_type == 'all' || $record_type == 'benefits') {
        $bind_types_benefits .= $bind_types;
        $bind_params_benefits = array_merge($bind_params_benefits, $bind_params);
    }
    error_log("Benefits query - bind_types: '$bind_types_benefits', params count: " . count($bind_params_benefits));
    mysqli_stmt_bind_param($stmt, $bind_types_benefits, ...$bind_params_benefits);
    if (!mysqli_stmt_execute($stmt)) {
        error_log("Benefits query execution failed: " . mysqli_stmt_error($stmt));
        $benefits = [];
    } else {
        $result_benefits = mysqli_stmt_get_result($stmt);
        $benefits = [];
        while ($row = mysqli_fetch_assoc($result_benefits)) {
            $benefits[] = $row;
        }
        error_log("Benefits query returned " . count($benefits) . " records");
    }
    mysqli_stmt_close($stmt);
} else {
    error_log("Failed to prepare benefits query: " . mysqli_error($link));
    $benefits = [];
}

// Benefits count
$sql_count_benefits = "SELECT COUNT(*) AS total
                       FROM member_benefits b
                       WHERE b.member_id = ?" . ($record_type == 'all' || $record_type == 'benefits' ? $time_filter : '');
$stmt = mysqli_prepare($link, $sql_count_benefits);
if ($stmt) {
    $bind_types_count = 'i' . ($record_type == 'all' || $record_type == 'benefits' ? $bind_types : '');
    $bind_params_count = array_merge([&$user_id], ($record_type == 'all' || $record_type == 'benefits' ? $bind_params : []));
    error_log("Benefits count query - bind_types: '$bind_types_count', params count: " . count($bind_params_count));
    mysqli_stmt_bind_param($stmt, $bind_types_count, ...$bind_params_count);
    if (!mysqli_stmt_execute($stmt)) {
        error_log("Benefits count query execution failed: " . mysqli_stmt_error($stmt));
        $total_records_benefits = 0;
        $total_pages_benefits = 1;
    } else {
        $result = mysqli_stmt_get_result($stmt);
        $total_records_benefits = $result ? mysqli_fetch_assoc($result)['total'] : 0;
        $total_pages_benefits = ceil($total_records_benefits / $pagination['benefits']['per_page']);
    }
    mysqli_stmt_close($stmt);
} else {
    error_log("Failed to prepare benefits count query: " . mysqli_error($link));
    $total_records_benefits = 0;
    $total_pages_benefits = 1;
}

// Debts
$sql_debts = "SELECT 
                 po.payment_type,
                 COALESCE(po.required_amount, 0) AS required_amount,
                 COALESCE(SUM(p.amount), 0) AS total_paid,
                 (COALESCE(po.required_amount, 0) - COALESCE(SUM(p.amount), 0)) AS debt,
                 po.payment_cycle,
                 po.due_date
              FROM payment_obligations po
              LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status = 'Paid'
              WHERE po.user_id = ? AND po.deleted_at IS NULL" . ($record_type == 'all' || $record_type == 'debts' ? $time_filter_debts : '') . "
              GROUP BY po.id, po.payment_type, po.required_amount, po.payment_cycle, po.due_date
              HAVING debt > 0
              ORDER BY po.created_at DESC
              LIMIT ? OFFSET ?";
if (in_array('required_amount', $columns)) {
    $stmt = mysqli_prepare($link, $sql_debts);
    if ($stmt) {
        $per_page_debts = (int)$pagination['debts']['per_page'];
        $offset_debts = (int)$pagination['debts']['offset'];
        $bind_types_debts = 'iii';
        $bind_params_debts = [&$user_id, &$per_page_debts, &$offset_debts];
        if ($record_type == 'all' || $record_type == 'debts') {
            $bind_types_debts .= $bind_types;
            $bind_params_debts = array_merge($bind_params_debts, $bind_params);
        }
        error_log("Debts query - bind_types: '$bind_types_debts', params count: " . count($bind_params_debts));
        mysqli_stmt_bind_param($stmt, $bind_types_debts, ...$bind_params_debts);
        if (!mysqli_stmt_execute($stmt)) {
            error_log("Debts query execution failed: " . mysqli_stmt_error($stmt));
            $debts = [];
        } else {
            $result_debts = mysqli_stmt_get_result($stmt);
            $debts = [];
            while ($row = mysqli_fetch_assoc($result_debts)) {
                $debts[] = $row;
            }
            error_log("Debts query returned " . count($debts) . " records");
        }
        mysqli_stmt_close($stmt);
    } else {
        error_log("Failed to prepare debts query: " . mysqli_error($link));
        $debts = [];
    }
} else {
    error_log("required_amount column missing in payment_obligations table for debts query");
    $debts = [];
    $debug_info['debts_error'] = "required_amount column missing in payment_obligations";
}

// Debts count
$sql_count_debts = "SELECT COUNT(*) AS total
                    FROM (
                        SELECT po.id, (COALESCE(po.required_amount, 0) - COALESCE(SUM(p.amount), 0)) AS debt
                        FROM payment_obligations po
                        LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status = 'Paid'
                        WHERE po.user_id = ? AND po.deleted_at IS NULL" . ($record_type == 'all' || $record_type == 'debts' ? $time_filter_debts : '') . "
                        GROUP BY po.id
                        HAVING debt > 0
                    ) AS debtor_count";
if (in_array('required_amount', $columns)) {
    $stmt = mysqli_prepare($link, $sql_count_debts);
    if ($stmt) {
        $bind_types_count = 'i' . ($record_type == 'all' || $record_type == 'debts' ? $bind_types : '');
        $bind_params_count = array_merge([&$user_id], ($record_type == 'all' || $record_type == 'debts' ? $bind_params : []));
        error_log("Debts count query - bind_types: '$bind_types_count', params count: " . count($bind_params_count));
        mysqli_stmt_bind_param($stmt, $bind_types_count, ...$bind_params_count);
        if (!mysqli_stmt_execute($stmt)) {
            error_log("Debts count query execution failed: " . mysqli_stmt_error($stmt));
            $total_records_debts = 0;
            $total_pages_debts = 1;
        } else {
            $result = mysqli_stmt_get_result($stmt);
            $total_records_debts = $result ? mysqli_fetch_assoc($result)['total'] : 0;
            $total_pages_debts = ceil($total_records_debts / $pagination['debts']['per_page']);
            $debug_info['debts_count_query'] = $sql_count_debts;
            $debug_info['debts_count'] = $total_records_debts;
        }
        mysqli_stmt_close($stmt);
    } else {
        error_log("Failed to prepare debts count query: " . mysqli_error($link));
        $total_records_debts = 0;
        $total_pages_debts = 1;
    }
} else {
    error_log("required_amount column missing in payment_obligations table for debts count query");
    $total_records_debts = 0;
    $total_pages_debts = 1;
    $debug_info['debts_count_error'] = "required_amount column missing in payment_obligations";
}
?>

<?php require_once 'includes/header.php'; ?>

<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>My Financial Records</h2>

            <!-- Display schema or data issue warnings -->
            <?php if (!in_array('required_amount', $columns)): ?>
                <div class="alert alert-danger" role="alert">
                    Error: The 'required_amount' column is missing in the payment_obligations table. Please contact the administrator or apply the necessary database migration.
                </div>
            <?php elseif ($null_count > 0): ?>
                <div class="alert alert-warning" role="alert">
                    Warning: <?php echo $null_count; ?> payment obligation(s) have NULL required_amount values, which may affect debt calculations. Please contact the administrator to update the database.
                </div>
            <?php endif; ?>

            <!-- Filter Form -->
            <form method="GET" class="mb-4">
                <div class="row g-3">
                    <div class="col-md-4">
                        <label for="record_type" class="form-label">Record Type</label>
                        <select name="record_type" id="record_type" class="form-select">
                            <option value="all" <?php echo $record_type == 'all' ? 'selected' : ''; ?>>All</option>
                            <option value="payments" <?php echo $record_type == 'payments' ? 'selected' : ''; ?>>Payments</option>
                            <option value="debts" <?php echo $record_type == 'debts' ? 'selected' : ''; ?>>Debts</option>
                            <option value="benefits" <?php echo $record_type == 'benefits' ? 'selected' : ''; ?>>Benefits</option>
                        </select>
                    </div>
                    <div class="col-md-4">
                        <label for="time_period" class="form-label">Time Period</label>
                        <select name="time_period" id="time_period" class="form-select" onchange="toggleDateField()">
                            <option value="all" <?php echo $time_period == 'all' ? 'selected' : ''; ?>>All Time</option>
                            <option value="date" <?php echo $time_period == 'date' ? 'selected' : ''; ?>>Specific Date</option>
                            <option value="week" <?php echo $time_period == 'week' ? 'selected' : ''; ?>>Last Week</option>
                            <option value="month" <?php echo $time_period == 'month' ? 'selected' : ''; ?>>Last Month</option>
                            <option value="year" <?php echo $time_period == 'year' ? 'selected' : ''; ?>>Last Year</option>
                        </select>
                    </div>
                    <div class="col-md-4" id="filter_date_container" style="display: <?php echo $time_period == 'date' ? 'block' : 'none'; ?>;">
                        <label for="filter_date" class="form-label">Select Date</label>
                        <input type="date" name="filter_date" id="filter_date" class="form-control" value="<?php echo htmlspecialchars($filter_date); ?>">
                    </div>
                    <div class="col-12">
                        <button type="submit" class="btn btn-primary mt-2">Apply Filters</button>
                    </div>
                </div>
            </form>

            <!-- Summary Cards -->
            <div class="row g-3 mb-4">
                <div class="col-md-4">
                    <div class="card text-white bg-success">
                        <div class="card-body">
                            <h5 class="card-title">Total Payments</h5>
                            <p class="card-text"><?php echo htmlspecialchars($currency_symbol); ?><?php echo number_format($total_payments, 2); ?></p>
                        </div>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="card text-white bg-danger">
                        <div class="card-body">
                            <h5 class="card-title">Total Debts</h5>
                            <p class="card-text"><?php echo htmlspecialchars($currency_symbol); ?><?php echo number_format($total_debts, 2); ?></p>
                        </div>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="card text-white bg-info">
                        <div class="card-body">
                            <h5 class="card-title">Total Benefits</h5>
                            <p class="card-text"><?php echo htmlspecialchars($currency_symbol); ?><?php echo number_format($total_benefits, 2); ?></p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Tabs for Payments, Debts, Benefits -->
            <ul class="nav nav-tabs mb-4" id="financeTabs" role="tablist">
                <?php if ($record_type == 'all' || $record_type == 'payments'): ?>
                <li class="nav-item">
                    <a class="nav-link <?php echo $record_type == 'payments' ? 'active' : ''; ?>" id="payments-tab" data-bs-toggle="tab" href="#payments" role="tab" aria-controls="payments" aria-selected="<?php echo $record_type == 'payments' ? 'true' : 'false'; ?>">Payments</a>
                </li>
                <?php endif; ?>
                <?php if ($record_type == 'all' || $record_type == 'debts'): ?>
                <li class="nav-item">
                    <a class="nav-link <?php echo $record_type == 'debts' ? 'active' : ''; ?>" id="debts-tab" data-bs-toggle="tab" href="#debts" role="tab" aria-controls="debts" aria-selected="<?php echo $record_type == 'debts' ? 'true' : 'false'; ?>">Debts</a>
                </li>
                <?php endif; ?>
                <?php if ($record_type == 'all' || $record_type == 'benefits'): ?>
                <li class="nav-item">
                    <a class="nav-link <?php echo $record_type == 'benefits' ? 'active' : ''; ?>" id="benefits-tab" data-bs-toggle="tab" href="#benefits" role="tab" aria-controls="benefits" aria-selected="<?php echo $record_type == 'benefits' ? 'true' : 'false'; ?>">Benefits</a>
                </li>
                <?php endif; ?>
            </ul>

            <div class="tab-content" id="financeTabsContent">
                <?php if ($record_type == 'all' || $record_type == 'payments'): ?>
                <div class="tab-pane fade <?php echo $record_type == 'payments' ? 'show active' : ''; ?>" id="payments" role="tabpanel" aria-labelledby="payments-tab">
                    <div class="card mb-4">
                        <div class="card-header">Payments</div>
                        <div class="card-body">
                            <form method="GET" class="mb-3">
                                <input type="hidden" name="record_type" value="<?php echo htmlspecialchars($record_type); ?>">
                                <input type="hidden" name="time_period" value="<?php echo htmlspecialchars($time_period); ?>">
                                <?php if ($time_period == 'date'): ?>
                                    <input type="hidden" name="filter_date" value="<?php echo htmlspecialchars($filter_date); ?>">
                                <?php endif; ?>
                                <div class="row g-3 align-items-center">
                                    <div class="col-auto">
                                        <label for="per_page_payments" class="col-form-label">Show</label>
                                    </div>
                                    <div class="col-auto">
                                        <select name="per_page_payments" id="per_page_payments" class="form-select" onchange="this.form.submit()">
                                            <option value="5" <?php echo $pagination['payments']['per_page'] == 5 ? 'selected' : ''; ?>>5</option>
                                            <option value="10" <?php echo $pagination['payments']['per_page'] == 10 ? 'selected' : ''; ?>>10</option>
                                            <option value="25" <?php echo $pagination['payments']['per_page'] == 25 ? 'selected' : ''; ?>>25</option>
                                            <option value="50" <?php echo $pagination['payments']['per_page'] == 50 ? 'selected' : ''; ?>>50</option>
                                            <option value="100" <?php echo $pagination['payments']['per_page'] == 100 ? 'selected' : ''; ?>>100</option>
                                        </select>
                                    </div>
                                    <div class="col-auto">
                                        <span>entries</span>
                                    </div>
                                </div>
                            </form>
                            <div class="table-responsive">
                                <table class="table table-striped table-hover">
                                    <thead>
                                        <tr>
                                            <th>Payment Type</th>
                                            <th>Amount</th>
                                            <th>Description</th>
                                            <th>Payment Cycle</th>
                                            <th>Due Date</th>
                                            <th>Date</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php if (empty($payments)): ?>
                                            <tr><td colspan="6" class="text-center">No payments found.</td></tr>
                                        <?php else: ?>
                                            <?php foreach ($payments as $payment): ?>
                                                <tr>
                                                    <td><?php echo htmlspecialchars($payment['payment_type']); ?></td>
                                                    <td><?php echo htmlspecialchars($currency_symbol . number_format($payment['amount'], 2)); ?></td>
                                                    <td><?php echo htmlspecialchars($payment['description'] ?: 'N/A'); ?></td>
                                                    <td><?php echo htmlspecialchars($payment['payment_cycle'] ?: 'N/A'); ?></td>
                                                    <td><?php echo htmlspecialchars($payment['due_date'] ? date('Y-m-d H:i', strtotime($payment['due_date'])) : 'N/A'); ?></td>
                                                    <td><?php echo htmlspecialchars(date('Y-m-d H:i', strtotime($payment['created_at']))); ?></td>
                                                </tr>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                    </tbody>
                                </table>
                            </div>
                            <!-- Pagination -->
                            <?php if ($total_pages_payments > 1): ?>
                                <nav aria-label="Payments pagination">
                                    <ul class="pagination justify-content-center">
                                        <li class="page-item <?php echo $pagination['payments']['page'] <= 1 ? 'disabled' : ''; ?>">
                                            <a class="page-link" href="?record_type=<?php echo urlencode($record_type); ?>&time_period=<?php echo urlencode($time_period); ?>&filter_date=<?php echo urlencode($filter_date); ?>&page_payments=<?php echo $pagination['payments']['page'] - 1; ?>&per_page_payments=<?php echo $pagination['payments']['per_page']; ?>">Previous</a>
                                        </li>
                                        <?php for ($i = 1; $i <= $total_pages_payments; $i++): ?>
                                            <li class="page-item <?php echo $pagination['payments']['page'] == $i ? 'active' : ''; ?>">
                                                <a class="page-link" href="?record_type=<?php echo urlencode($record_type); ?>&time_period=<?php echo urlencode($time_period); ?>&filter_date=<?php echo urlencode($filter_date); ?>&page_payments=<?php echo $i; ?>&per_page_payments=<?php echo $pagination['payments']['per_page']; ?>"><?php echo $i; ?></a>
                                            </li>
                                        <?php endfor; ?>
                                        <li class="page-item <?php echo $pagination['payments']['page'] >= $total_pages_payments ? 'disabled' : ''; ?>">
                                            <a class="page-link" href="?record_type=<?php echo urlencode($record_type); ?>&time_period=<?php echo urlencode($time_period); ?>&filter_date=<?php echo urlencode($filter_date); ?>&page_payments=<?php echo $pagination['payments']['page'] + 1; ?>&per_page_payments=<?php echo $pagination['payments']['per_page']; ?>">Next</a>
                                        </li>
                                    </ul>
                                </nav>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
                <?php if ($record_type == 'all' || $record_type == 'debts'): ?>
                <div class="tab-pane fade <?php echo $record_type == 'debts' ? 'show active' : ''; ?>" id="debts" role="tabpanel" aria-labelledby="debts-tab">
                    <div class="card mb-4">
                        <div class="card-header">Debts</div>
                        <div class="card-body">
                            <form method="GET" class="mb-3">
                                <input type="hidden" name="record_type" value="<?php echo htmlspecialchars($record_type); ?>">
                                <input type="hidden" name="time_period" value="<?php echo htmlspecialchars($time_period); ?>">
                                <?php if ($time_period == 'date'): ?>
                                    <input type="hidden" name="filter_date" value="<?php echo htmlspecialchars($filter_date); ?>">
                                <?php endif; ?>
                                <div class="row g-3 align-items-center">
                                    <div class="col-auto">
                                        <label for="per_page_debts" class="col-form-label">Show</label>
                                    </div>
                                    <div class="col-auto">
                                        <select name="per_page_debts" id="per_page_debts" class="form-select" onchange="this.form.submit()">
                                            <option value="5" <?php echo $pagination['debts']['per_page'] == 5 ? 'selected' : ''; ?>>5</option>
                                            <option value="10" <?php echo $pagination['debts']['per_page'] == 10 ? 'selected' : ''; ?>>10</option>
                                            <option value="25" <?php echo $pagination['debts']['per_page'] == 25 ? 'selected' : ''; ?>>25</option>
                                            <option value="50" <?php echo $pagination['debts']['per_page'] == 50 ? 'selected' : ''; ?>>50</option>
                                            <option value="100" <?php echo $pagination['debts']['per_page'] == 100 ? 'selected' : ''; ?>>100</option>
                                        </select>
                                    </div>
                                    <div class="col-auto">
                                        <span>entries</span>
                                    </div>
                                </div>
                            </form>
                            <div class="table-responsive">
                                <table class="table table-striped table-hover">
                                    <thead>
                                        <tr>
                                            <th>Payment Type</th>
                                            <th>Required Amount</th>
                                            <th>Paid Amount</th>
                                            <th>Debt</th>
                                            <th>Payment Cycle</th>
                                            <th>Due Date</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php if (empty($debts)): ?>
                                            <tr><td colspan="6" class="text-center">No debts found.</td></tr>
                                        <?php else: ?>
                                            <?php foreach ($debts as $debt): ?>
                                                <tr>
                                                    <td><?php echo htmlspecialchars($debt['payment_type']); ?></td>
                                                    <td><?php echo htmlspecialchars($currency_symbol . number_format($debt['required_amount'], 2)); ?></td>
                                                    <td><?php echo htmlspecialchars($currency_symbol . number_format($debt['total_paid'], 2)); ?></td>
                                                    <td><?php echo htmlspecialchars($currency_symbol . number_format($debt['debt'], 2)); ?></td>
                                                    <td><?php echo htmlspecialchars($debt['payment_cycle'] ?: 'N/A'); ?></td>
                                                    <td><?php echo htmlspecialchars($debt['due_date'] ? date('Y-m-d H:i', strtotime($debt['due_date'])) : 'N/A'); ?></td>
                                                </tr>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                    </tbody>
                                </table>
                            </div>
                            <!-- Pagination -->
                            <?php if ($total_pages_debts > 1): ?>
                                <nav aria-label="Debts pagination">
                                    <ul class="pagination justify-content-center">
                                        <li class="page-item <?php echo $pagination['debts']['page'] <= 1 ? 'disabled' : ''; ?>">
                                            <a class="page-link" href="?record_type=<?php echo urlencode($record_type); ?>&time_period=<?php echo urlencode($time_period); ?>&filter_date=<?php echo urlencode($filter_date); ?>&page_debts=<?php echo $pagination['debts']['page'] - 1; ?>&per_page_debts=<?php echo $pagination['debts']['per_page']; ?>">Previous</a>
                                        </li>
                                        <?php for ($i = 1; $i <= $total_pages_debts; $i++): ?>
                                            <li class="page-item <?php echo $pagination['debts']['page'] == $i ? 'active' : ''; ?>">
                                                <a class="page-link" href="?record_type=<?php echo urlencode($record_type); ?>&time_period=<?php echo urlencode($time_period); ?>&filter_date=<?php echo urlencode($filter_date); ?>&page_debts=<?php echo $i; ?>&per_page_debts=<?php echo $pagination['debts']['per_page']; ?>"><?php echo $i; ?></a>
                                            </li>
                                        <?php endfor; ?>
                                        <li class="page-item <?php echo $pagination['debts']['page'] >= $total_pages_debts ? 'disabled' : ''; ?>">
                                            <a class="page-link" href="?record_type=<?php echo urlencode($record_type); ?>&time_period=<?php echo urlencode($time_period); ?>&filter_date=<?php echo urlencode($filter_date); ?>&page_debts=<?php echo $pagination['debts']['page'] + 1; ?>&per_page_debts=<?php echo $pagination['debts']['per_page']; ?>">Next</a>
                                        </li>
                                    </ul>
                                </nav>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
                <?php if ($record_type == 'all' || $record_type == 'benefits'): ?>
                <div class="tab-pane fade <?php echo $record_type == 'benefits' ? 'show active' : ''; ?>" id="benefits" role="tabpanel" aria-labelledby="benefits-tab">
                    <div class="card mb-4">
                        <div class="card-header">Benefits</div>
                        <div class="card-body">
                            <form method="GET" class="mb-3">
                                <input type="hidden" name="record_type" value="<?php echo htmlspecialchars($record_type); ?>">
                                <input type="hidden" name="time_period" value="<?php echo htmlspecialchars($time_period); ?>">
                                <?php if ($time_period == 'date'): ?>
                                    <input type="hidden" name="filter_date" value="<?php echo htmlspecialchars($filter_date); ?>">
                                <?php endif; ?>
                                <div class="row g-3 align-items-center">
                                    <div class="col-auto">
                                        <label for="per_page_benefits" class="col-form-label">Show</label>
                                    </div>
                                    <div class="col-auto">
                                        <select name="per_page_benefits" id="per_page_benefits" class="form-select" onchange="this.form.submit()">
                                            <option value="5" <?php echo $pagination['benefits']['per_page'] == 5 ? 'selected' : ''; ?>>5</option>
                                            <option value="10" <?php echo $pagination['benefits']['per_page'] == 10 ? 'selected' : ''; ?>>10</option>
                                            <option value="25" <?php echo $pagination['benefits']['per_page'] == 25 ? 'selected' : ''; ?>>25</option>
                                            <option value="50" <?php echo $pagination['benefits']['per_page'] == 50 ? 'selected' : ''; ?>>50</option>
                                            <option value="100" <?php echo $pagination['benefits']['per_page'] == 100 ? 'selected' : ''; ?>>100</option>
                                        </select>
                                    </div>
                                    <div class="col-auto">
                                        <span>entries</span>
                                    </div>
                                </div>
                            </form>
                            <div class="table-responsive">
                                <table class="table table-striped table-hover">
                                    <thead>
                                        <tr>
                                            <th>Benefit Type</th>
                                            <th>Amount</th>
                                            <th>Date</th>
                                            <th>Status</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php if (empty($benefits)): ?>
                                            <tr><td colspan="4" class="text-center">No benefits found.</td></tr>
                                        <?php else: ?>
                                            <?php foreach ($benefits as $benefit): ?>
                                                <tr>
                                                    <td><?php echo htmlspecialchars($benefit['benefit_type']); ?></td>
                                                    <td><?php echo htmlspecialchars($currency_symbol . number_format($benefit['amount'], 2)); ?></td>
                                                    <td><?php echo htmlspecialchars(date('Y-m-d H:i', strtotime($benefit['created_at']))); ?></td>
                                                    <td><?php echo htmlspecialchars($benefit['benefit_status']); ?></td>
                                                </tr>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                    </tbody>
                                </table>
                            </div>
                            <!-- Pagination -->
                            <?php if ($total_pages_benefits > 1): ?>
                                <nav aria-label="Benefits pagination">
                                    <ul class="pagination justify-content-center">
                                        <li class="page-item <?php echo $pagination['benefits']['page'] <= 1 ? 'disabled' : ''; ?>">
                                            <a class="page-link" href="?record_type=<?php echo urlencode($record_type); ?>&time_period=<?php echo urlencode($time_period); ?>&filter_date=<?php echo urlencode($filter_date); ?>&page_benefits=<?php echo $pagination['benefits']['page'] - 1; ?>&per_page_benefits=<?php echo $pagination['benefits']['per_page']; ?>">Previous</a>
                                        </li>
                                        <?php for ($i = 1; $i <= $total_pages_benefits; $i++): ?>
                                            <li class="page-item <?php echo $pagination['benefits']['page'] == $i ? 'active' : ''; ?>">
                                                <a class="page-link" href="?record_type=<?php echo urlencode($record_type); ?>&time_period=<?php echo urlencode($time_period); ?>&filter_date=<?php echo urlencode($filter_date); ?>&page_benefits=<?php echo $i; ?>&per_page_benefits=<?php echo $pagination['benefits']['per_page']; ?>"><?php echo $i; ?></a>
                                            </li>
                                        <?php endfor; ?>
                                        <li class="page-item <?php echo $pagination['benefits']['page'] >= $total_pages_benefits ? 'disabled' : ''; ?>">
                                            <a class="page-link" href="?record_type=<?php echo urlencode($record_type); ?>&time_period=<?php echo urlencode($time_period); ?>&filter_date=<?php echo urlencode($filter_date); ?>&page_benefits=<?php echo $pagination['benefits']['page'] + 1; ?>&per_page_benefits=<?php echo $pagination['benefits']['per_page']; ?>">Next</a>
                                        </li>
                                    </ul>
                                </nav>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            </div>

            <!-- Debug Info for Admins -->
            <?php if ($is_admin): ?>
                <div class="card mt-4">
                    <div class="card-header">Debug Information (Admin Only)</div>
                    <div class="card-body">
                        <pre><?php echo htmlspecialchars(json_encode($debug_info, JSON_PRETTY_PRINT)); ?></pre>
                    </div>
                </div>
            <?php endif; ?>
        </main>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js" integrity="sha384-YvpcrYf0tY3lHB60NNkmXc5s9fDVZLESaAA55NDzOxhy9GkcIdslK1eN7N6jIeHz" crossorigin="anonymous"></script>
<script>
function toggleDateField() {
    const timePeriod = document.getElementById('time_period').value;
    const filterDateContainer = document.getElementById('filter_date_container');
    filterDateContainer.style.display = timePeriod === 'date' ? 'block' : 'none';
}
</script>

<?php require_once 'includes/footer.php'; ?>