<?php
// Ensure session is started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once 'includes/functions.php';
require_once 'includes/db_config.php';

// Debug session data
error_log("Session Data in messages_admin.php: " . json_encode([
    'is_logged_in' => is_logged_in(),
    'user_id' => $_SESSION['user_id'] ?? 'none',
    'user_role' => $_SESSION['user_role'] ?? 'none',
    'session_id' => session_id(),
    'session_vars' => $_SESSION
]));

// Check if user is logged in and has manage_messages permission
if (!is_logged_in() || !user_has_permission($link, 'manage_messages')) {
    error_log("Access denied to messages_admin.php. Role: " . ($_SESSION['user_role'] ?? 'none') . ", Permissions: " . json_encode(get_role_permissions($link, $_SESSION['user_role'] ?? '')));
    redirect("admin_dashboard.php");
}

$admin_id = $_SESSION['user_id'];
$selected_member_id = isset($_GET['member_id']) ? intval($_GET['member_id']) : null;
$success_message = '';
$error_message = '';

// Handle message reply
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['reply_message'])) {
    $message_content = sanitize_input($_POST['message_content']);
    $member_id = intval($_POST['member_id']);
    
    if (!empty($message_content) && $member_id) {
        $sql = "INSERT INTO messages (sender_id, recipient_id, message_content, timestamp, is_read) VALUES (?, ?, ?, NOW(), 0)";
        if ($stmt = mysqli_prepare($link, $sql)) {
            mysqli_stmt_bind_param($stmt, "iis", $admin_id, $member_id, $message_content);
            if (mysqli_stmt_execute($stmt)) {
                $success_message = "Reply sent successfully!";
            } else {
                $error_message = "Failed to send reply.";
            }
            mysqli_stmt_close($stmt);
        }
    } else {
        $error_message = "Reply cannot be empty.";
    }
}

// Fetch members who have sent messages
$sql_members = "SELECT DISTINCT users.id, users.full_name FROM users JOIN messages ON users.id = messages.sender_id WHERE users.role = 'member' ORDER BY users.full_name ASC";
$result_members = mysqli_query($link, $sql_members);
$member_list = [];
if ($result_members) {
    while ($row = mysqli_fetch_assoc($result_members)) {
        $member_list[] = $row;
    }
}

// Fetch conversation history for the selected member
$conversation = [];
if ($selected_member_id) {
    $sql_conversation = "SELECT sender_id, message_content, timestamp FROM messages WHERE (sender_id = ? AND recipient_id = ?) OR (sender_id = ? AND recipient_id = ?) ORDER BY timestamp ASC";
    if ($stmt_conv = mysqli_prepare($link, $sql_conversation)) {
        mysqli_stmt_bind_param($stmt_conv, "iiii", $selected_member_id, $admin_id, $admin_id, $selected_member_id);
        mysqli_stmt_execute($stmt_conv);
        $result_conv = mysqli_stmt_get_result($stmt_conv);
        while ($row = mysqli_fetch_assoc($result_conv)) {
            $conversation[] = $row;
        }
        mysqli_stmt_close($stmt_conv);

        // Mark messages from this member as read
        $sql_mark_read = "UPDATE messages SET is_read = 1 WHERE sender_id = ? AND recipient_id = ?";
        $stmt_read = mysqli_prepare($link, $sql_mark_read);
        mysqli_stmt_bind_param($stmt_read, "ii", $selected_member_id, $admin_id);
        mysqli_stmt_execute($stmt_read);
        mysqli_stmt_close($stmt_read);
    }
}
?>

<?php require_once 'includes/header.php'; ?>
<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Messages Dashboard</h2>

            <?php if (!empty($success_message)): ?>
                <div class="alert alert-success"><?php echo $success_message; ?></div>
            <?php endif; ?>
            <?php if (!empty($error_message)): ?>
                <div class="alert alert-danger"><?php echo $error_message; ?></div>
            <?php endif; ?>
            
            <div class="card">
                <div class="card-header">Messages</div>
                <div class="card-body">
                    <div class="admin-message-container d-flex">
                        <div class="member-list-sidebar col-md-3">
                            <h3>Members with Messages</h3>
                            <ul class="list-group">
                                <?php if (empty($member_list)): ?>
                                    <li class="list-group-item">No members have sent messages yet.</li>
                                <?php else: ?>
                                    <?php foreach ($member_list as $member): ?>
                                        <li class="list-group-item">
                                            <a href="messages_admin.php?member_id=<?php echo $member['id']; ?>" class="<?php echo ($selected_member_id == $member['id']) ? 'active' : ''; ?>">
                                                <?php echo htmlspecialchars($member['full_name']); ?>
                                            </a>
                                        </li>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </ul>
                        </div>
                        <div class="conversation-panel col-md-9">
                            <?php if ($selected_member_id): ?>
                                <h3>Conversation with <?php echo htmlspecialchars(current(array_filter($member_list, function($m) use ($selected_member_id) { return $m['id'] == $selected_member_id; }))['full_name']); ?></h3>
                                <div class="conversation-history">
                                    <?php if (empty($conversation)): ?>
                                        <p>No messages in this conversation.</p>
                                    <?php else: ?>
                                        <?php foreach ($conversation as $msg): ?>
                                            <div class="message-bubble <?php echo ($msg['sender_id'] === $admin_id) ? 'sent-by-admin' : 'received-by-admin'; ?>">
                                                <p class="message-content"><?php echo htmlspecialchars($msg['message_content']); ?></p>
                                                <span class="message-timestamp"><?php echo date('M j, Y, h:i a', strtotime($msg['timestamp'])); ?></span>
                                            </div>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </div>
                                <div class="message-reply-form">
                                    <form action="messages_admin.php?member_id=<?php echo $selected_member_id; ?>" method="post">
                                        <input type="hidden" name="member_id" value="<?php echo $selected_member_id; ?>">
                                        <textarea name="message_content" class="form-control" rows="3" placeholder="Type your reply here..." required></textarea>
                                        <button type="submit" name="reply_message" class="btn btn-primary mt-2">Reply</button>
                                    </form>
                                </div>
                            <?php else: ?>
                                <p class="select-member-prompt">Select a member from the sidebar to view their messages.</p>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>
</div>
<?php require_once 'includes/footer.php'; ?>