<?php
require_once 'includes/functions.php';
require_once 'includes/db_config.php';
session_start();

// Check if user is logged in
if (!is_logged_in()) {
    redirect("login.php");
}

$is_admin = is_admin();
$currency_symbol = get_setting('currency_symbol', $link) ?? '₦';

// Pagination settings for each table
$tables = ['transactions', 'recent_debtors', 'all_debtors', 'expenses', 'benefits', 'payment_summary'];
$pagination = [];
foreach ($tables as $table) {
    $pagination[$table] = [
        'per_page' => isset($_GET["per_page_$table"]) && in_array($_GET["per_page_$table"], [10, 25, 50, 100]) ? (int)$_GET["per_page_$table"] : 10,
        'page' => isset($_GET["page_$table"]) && is_numeric($_GET["page_$table"]) && $_GET["page_$table"] > 0 ? (int)$_GET["page_$table"] : 1
    ];
    $pagination[$table]['offset'] = ($pagination[$table]['page'] - 1) * $pagination[$table]['per_page'];
}

// Fetch total records for pagination
// Financial Transactions
$sql_count_transactions = "SELECT COUNT(*) AS total FROM payments p JOIN users u ON p.user_id = u.id WHERE u.role = 'member'";
$result_count_transactions = mysqli_query($link, $sql_count_transactions);
if (!$result_count_transactions) {
    die("Error fetching transaction count: " . mysqli_error($link));
}
$total_records_transactions = mysqli_fetch_assoc($result_count_transactions)['total'];
$total_pages_transactions = ceil($total_records_transactions / $pagination['transactions']['per_page']);

// Debtors (Recent and All Debtors use the same query, different limits)
$sql_count_debtors = "SELECT COUNT(*) AS total
                     FROM users u
                     CROSS JOIN payment_types pt
                     LEFT JOIN payments p ON u.id = p.user_id AND p.payment_type = pt.type_name
                     WHERE u.role = 'member'
                     GROUP BY u.id, pt.type_name
                     HAVING COALESCE(SUM(p.amount), 0) < 1000"; // Assume 1000 as default required amount
$result_count_debtors = mysqli_query($link, $sql_count_debtors);
if (!$result_count_debtors) {
    die("Error fetching debtors count: " . mysqli_error($link));
}
$total_records_debtors = mysqli_num_rows($result_count_debtors);
$total_pages_debtors = ceil($total_records_debtors / $pagination['all_debtors']['per_page']);
$total_pages_recent_debtors = ceil($total_records_debtors / $pagination['recent_debtors']['per_page']);

// Expense Summary
$sql_count_expenses = "SELECT COUNT(*) AS total
                       FROM expenses e
                       JOIN expense_types et ON e.expense_type_id = et.id
                       GROUP BY et.type_name";
$result_count_expenses = mysqli_query($link, $sql_count_expenses);
if (!$result_count_expenses) {
    die("Error fetching expenses count: " . mysqli_error($link));
}
$total_records_expenses = mysqli_num_rows($result_count_expenses);
$total_pages_expenses = ceil($total_records_expenses / $pagination['expenses']['per_page']);

// Benefit Types Summary
$sql_count_benefits = "SELECT COUNT(*) AS total
                       FROM member_benefits mb
                       JOIN benefit_types bt ON mb.benefit_type_id = bt.id
                       GROUP BY bt.type_name";
$result_count_benefits = mysqli_query($link, $sql_count_benefits);
if (!$result_count_benefits) {
    die("Error fetching benefits count: " . mysqli_error($link));
}
$total_records_benefits = mysqli_num_rows($result_count_benefits);
$total_pages_benefits = ceil($total_records_benefits / $pagination['benefits']['per_page']);

// Payment Types Summary (admin-only)
$sql_count_payment_summary = "SELECT COUNT(*) AS total
                              FROM payments p
                              JOIN users u ON p.user_id = u.id
                              WHERE u.role = 'member'
                              GROUP BY p.payment_type";
$result_count_payment_summary = mysqli_query($link, $sql_count_payment_summary);
if (!$result_count_payment_summary) {
    die("Error fetching payment summary count: " . mysqli_error($link));
}
$total_records_payment_summary = mysqli_num_rows($result_count_payment_summary);
$total_pages_payment_summary = ceil($total_records_payment_summary / $pagination['payment_summary']['per_page']);

// Fetch payment transactions
$sql_payments = "SELECT 
                    p.id, p.payment_type, p.amount, p.payment_status, p.created_at,
                    u.id AS user_id, u.member_id, u.full_name,
                    fd.data AS form_data,
                    f.form_data AS form_structure
                 FROM payments p
                 JOIN users u ON p.user_id = u.id
                 LEFT JOIN form_data fd ON u.id = fd.user_id
                 LEFT JOIN forms f ON fd.form_id = f.id
                 WHERE u.role = 'member'
                 ORDER BY p.created_at DESC
                 LIMIT ? OFFSET ?";
$stmt = mysqli_prepare($link, $sql_payments);
mysqli_stmt_bind_param($stmt, "ii", $pagination['transactions']['per_page'], $pagination['transactions']['offset']);
mysqli_stmt_execute($stmt);
$result_payments = mysqli_stmt_get_result($stmt);
$payments = [];
if ($result_payments) {
    while ($row = mysqli_fetch_assoc($result_payments)) {
        $surname = '';
        $other_names = '';
        $member_full_name = $row['full_name'] ?? 'Unknown';

        if (isset($row['form_data']) && is_string($row['form_data']) && isset($row['form_structure']) && is_string($row['form_structure'])) {
            $form_data_array = json_decode($row['form_data'], true);
            $form_structure_array = json_decode($row['form_structure'], true);

            if (is_array($form_data_array) && is_array($form_structure_array) && isset($form_structure_array['fields'])) {
                foreach ($form_structure_array['fields'] as $field) {
                    if (isset($field['label']) && strpos(strtolower($field['label']), 'surname') !== false) {
                        if (isset($field['name']) && isset($form_data_array[$field['name']])) {
                            $surname = $form_data_array[$field['name']];
                        }
                    } elseif (isset($field['label']) && strpos(strtolower($field['label']), 'other names') !== false) {
                        if (isset($field['name']) && isset($form_data_array[$field['name']])) {
                            $other_names = $form_data_array[$field['name']];
                        }
                    }
                }
                $constructed_name = trim($surname . ' ' . $other_names);
                if (!empty($constructed_name)) {
                    $member_full_name = $constructed_name;
                }
            }
        }

        $payments[] = [
            'id' => $row['id'],
            'member_id' => $row['member_id'],
            'full_name' => $member_full_name,
            'payment_type' => $row['payment_type'] ?? 'Unknown',
            'amount' => $row['amount'] ?? 0,
            'payment_status' => $row['payment_status'] ?? 'Unknown',
            'membership_status' => 'active',
            'created_at' => $row['created_at'],
            'user_id' => $row['user_id']
        ];
    }
}
mysqli_stmt_close($stmt);

// Calculate totals for cards
$total_paid = array_sum(array_column($payments, 'amount'));
$paid_this_month = array_sum(array_filter($payments, function($p) {
    return date('Y-m', strtotime($p['created_at'])) == date('Y-m');
}, ARRAY_FILTER_USE_BOTH));

// Fetch debtors
$sql_debtors = "SELECT 
                    u.id AS user_id, u.member_id, u.full_name,
                    pt.type_name AS payment_type,
                    COALESCE(SUM(p.amount), 0) AS total_paid,
                    (1000 - COALESCE(SUM(p.amount), 0)) AS debt
                 FROM users u
                 CROSS JOIN payment_types pt
                 LEFT JOIN payments p ON u.id = p.user_id AND p.payment_type = pt.type_name
                 WHERE u.role = 'member'
                 GROUP BY u.id, pt.type_name
                 HAVING debt > 0
                 ORDER BY debt DESC
                 LIMIT ? OFFSET ?";
$stmt = mysqli_prepare($link, $sql_debtors);
mysqli_stmt_bind_param($stmt, "ii", $pagination['all_debtors']['per_page'], $pagination['all_debtors']['offset']);
mysqli_stmt_execute($stmt);
$result_debtors = mysqli_stmt_get_result($stmt);
$debtors = [];
$total_debt = 0;
if ($result_debtors) {
    while ($row = mysqli_fetch_assoc($result_debtors)) {
        $surname = '';
        $other_names = '';
        $member_full_name = $row['full_name'] ?? 'Unknown';

        $sql_form = "SELECT fd.data, f.form_data
                     FROM form_data fd
                     JOIN forms f ON fd.form_id = f.id
                     WHERE fd.user_id = ?";
        if ($stmt_form = mysqli_prepare($link, $sql_form)) {
            mysqli_stmt_bind_param($stmt_form, "i", $row['user_id']);
            mysqli_stmt_execute($stmt_form);
            $form_result = mysqli_stmt_get_result($stmt_form);
            if ($form_row = mysqli_fetch_assoc($form_result)) {
                if (isset($form_row['data']) && is_string($form_row['data']) && isset($form_row['form_data']) && is_string($form_row['form_data'])) {
                    $form_data_array = json_decode($form_row['data'], true);
                    $form_structure_array = json_decode($form_row['form_data'], true);
                    if (is_array($form_data_array) && is_array($form_structure_array) && isset($form_structure_array['fields'])) {
                        foreach ($form_structure_array['fields'] as $field) {
                            if (isset($field['label']) && strpos(strtolower($field['label']), 'surname') !== false) {
                                if (isset($field['name']) && isset($form_data_array[$field['name']])) {
                                    $surname = $form_data_array[$field['name']];
                                }
                            } elseif (isset($field['label']) && strpos(strtolower($field['label']), 'other names') !== false) {
                                if (isset($field['name']) && isset($form_data_array[$field['name']])) {
                                    $other_names = $form_data_array[$field['name']];
                                }
                            }
                        }
                        $constructed_name = trim($surname . ' ' . $other_names);
                        if (!empty($constructed_name)) {
                            $member_full_name = $constructed_name;
                        }
                    }
                }
            }
            mysqli_stmt_close($stmt_form);
        }

        $debtors[] = [
            'member_id' => $row['member_id'],
            'full_name' => $member_full_name,
            'payment_type' => $row['payment_type'],
            'required_amount' => 1000, // Hard-coded for now
            'total_paid' => $row['total_paid'],
            'debt' => $row['debt'],
            'user_id' => $row['user_id']
        ];
        $total_debt += $row['debt'];
    }
}
mysqli_stmt_close($stmt);

// Fetch recent debtors (same query, different pagination)
$stmt = mysqli_prepare($link, $sql_debtors);
mysqli_stmt_bind_param($stmt, "ii", $pagination['recent_debtors']['per_page'], $pagination['recent_debtors']['offset']);
mysqli_stmt_execute($stmt);
$result_recent_debtors = mysqli_stmt_get_result($stmt);
$recent_debtors = [];
if ($result_recent_debtors) {
    while ($row = mysqli_fetch_assoc($result_recent_debtors)) {
        $surname = '';
        $other_names = '';
        $member_full_name = $row['full_name'] ?? 'Unknown';

        $sql_form = "SELECT fd.data, f.form_data
                     FROM form_data fd
                     JOIN forms f ON fd.form_id = f.id
                     WHERE fd.user_id = ?";
        if ($stmt_form = mysqli_prepare($link, $sql_form)) {
            mysqli_stmt_bind_param($stmt_form, "i", $row['user_id']);
            mysqli_stmt_execute($stmt_form);
            $form_result = mysqli_stmt_get_result($stmt_form);
            if ($form_row = mysqli_fetch_assoc($form_result)) {
                if (isset($form_row['data']) && is_string($form_row['data']) && isset($form_row['form_data']) && is_string($form_row['form_data'])) {
                    $form_data_array = json_decode($form_row['data'], true);
                    $form_structure_array = json_decode($form_row['form_data'], true);
                    if (is_array($form_data_array) && is_array($form_structure_array) && isset($form_structure_array['fields'])) {
                        foreach ($form_structure_array['fields'] as $field) {
                            if (isset($field['label']) && strpos(strtolower($field['label']), 'surname') !== false) {
                                if (isset($field['name']) && isset($form_data_array[$field['name']])) {
                                    $surname = $form_data_array[$field['name']];
                                }
                            } elseif (isset($field['label']) && strpos(strtolower($field['label']), 'other names') !== false) {
                                if (isset($field['name']) && isset($form_data_array[$field['name']])) {
                                    $other_names = $form_data_array[$field['name']];
                                }
                            }
                        }
                        $constructed_name = trim($surname . ' ' . $other_names);
                        if (!empty($constructed_name)) {
                            $member_full_name = $constructed_name;
                        }
                    }
                }
            }
            mysqli_stmt_close($stmt_form);
        }

        $recent_debtors[] = [
            'member_id' => $row['member_id'],
            'full_name' => $member_full_name,
            'payment_type' => $row['payment_type'],
            'required_amount' => 1000, // Hard-coded for now
            'total_paid' => $row['total_paid'],
            'debt' => $row['debt'],
            'user_id' => $row['user_id']
        ];
    }
}
mysqli_stmt_close($stmt);

// Fetch total expenses for card
$sql_total_expenses = "SELECT COALESCE(SUM(e.amount), 0) AS total_expenses
                       FROM expenses e";
$result_total_expenses = mysqli_query($link, $sql_total_expenses);
if (!$result_total_expenses) {
    die("Error fetching total expenses: " . mysqli_error($link));
}
$total_expenses = mysqli_fetch_assoc($result_total_expenses)['total_expenses'];

// Fetch total benefits for card
$sql_total_benefits = "SELECT COALESCE(SUM(mb.amount), 0) AS total_benefits
                       FROM member_benefits mb";
$result_total_benefits = mysqli_query($link, $sql_total_benefits);
if (!$result_total_benefits) {
    die("Error fetching total benefits: " . mysqli_error($link));
}
$total_benefits = mysqli_fetch_assoc($result_total_benefits)['total_benefits'];

// Fetch expense summary
$sql_expenses = "SELECT 
                    et.type_name AS expense_type,
                    SUM(e.amount) AS total_amount,
                    COUNT(e.id) AS count
                 FROM expenses e
                 JOIN expense_types et ON e.expense_type_id = et.id
                 GROUP BY et.type_name
                 ORDER BY total_amount DESC
                 LIMIT ? OFFSET ?";
$stmt = mysqli_prepare($link, $sql_expenses);
mysqli_stmt_bind_param($stmt, "ii", $pagination['expenses']['per_page'], $pagination['expenses']['offset']);
mysqli_stmt_execute($stmt);
$result_expenses = mysqli_stmt_get_result($stmt);
$expenses_summary = [];
if ($result_expenses) {
    while ($row = mysqli_fetch_assoc($result_expenses)) {
        $expenses_summary[] = $row;
    }
}
mysqli_stmt_close($stmt);

// Fetch benefit types summary
$sql_benefits = "SELECT 
                    bt.type_name AS benefit_type,
                    SUM(mb.amount) AS total_amount,
                    COUNT(mb.id) AS count
                 FROM member_benefits mb
                 JOIN benefit_types bt ON mb.benefit_type_id = bt.id
                 GROUP BY bt.type_name
                 ORDER BY total_amount DESC
                 LIMIT ? OFFSET ?";
$stmt = mysqli_prepare($link, $sql_benefits);
mysqli_stmt_bind_param($stmt, "ii", $pagination['benefits']['per_page'], $pagination['benefits']['offset']);
mysqli_stmt_execute($stmt);
$result_benefits = mysqli_stmt_get_result($stmt);
$benefits_summary = [];
if ($result_benefits) {
    while ($row = mysqli_fetch_assoc($result_benefits)) {
        $benefits_summary[] = $row;
    }
}
mysqli_stmt_close($stmt);

// Fetch payment types summary (admin-only)
$sql_payment_summary = "SELECT 
                          p.payment_type,
                          SUM(p.amount) AS total_amount,
                          COUNT(p.id) AS count
                       FROM payments p
                       JOIN users u ON p.user_id = u.id
                       WHERE u.role = 'member'
                       GROUP BY p.payment_type
                       ORDER BY total_amount DESC
                       LIMIT ? OFFSET ?";
$stmt = mysqli_prepare($link, $sql_payment_summary);
mysqli_stmt_bind_param($stmt, "ii", $pagination['payment_summary']['per_page'], $pagination['payment_summary']['offset']);
mysqli_stmt_execute($stmt);
$result_payment_summary = mysqli_stmt_get_result($stmt);
$payment_summary = [];
if ($result_payment_summary) {
    while ($row = mysqli_fetch_assoc($result_payment_summary)) {
        $payment_summary[] = $row;
    }
}
mysqli_stmt_close($stmt);
?>

<?php require_once 'includes/header.php'; ?>
<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 col-lg-10 main-content">
            <h2>Club Financials</h2>
            <hr>

            <!-- Cards: 2 rows, 3 columns -->
            <div class="row mb-4">
                <div class="col-md-4">
                    <div class="card text-white bg-success mb-3">
                        <div class="card-header">Total Paid (All Time)</div>
                        <div class="card-body">
                            <h5 class="card-title"><?php echo htmlspecialchars($currency_symbol . ' ' . number_format($total_paid, 2)); ?></h5>
                        </div>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="card text-white bg-info mb-3">
                        <div class="card-header">Payments This Month</div>
                        <div class="card-body">
                            <h5 class="card-title"><?php echo htmlspecialchars($currency_symbol . ' ' . number_format($paid_this_month, 2)); ?></h5>
                        </div>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="card text-white bg-warning mb-3">
                        <div class="card-header">Outstanding Debtors</div>
                        <div class="card-body">
                            <h5 class="card-title"><?php echo htmlspecialchars($currency_symbol . ' ' . number_format($total_debt, 2)); ?></h5>
                        </div>
                    </div>
                </div>
            </div>
            <div class="row mb-4">
                <div class="col-md-4">
                    <div class="card text-white bg-danger mb-3">
                        <div class="card-header">Total Expenses</div>
                        <div class="card-body">
                            <h5 class="card-title"><?php echo htmlspecialchars($currency_symbol . ' ' . number_format($total_expenses, 2)); ?></h5>
                        </div>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="card text-white bg-primary mb-3">
                        <div class="card-header">Total Benefits Given Out</div>
                        <div class="card-body">
                            <h5 class="card-title"><?php echo htmlspecialchars($currency_symbol . ' ' . number_format($total_benefits, 2)); ?></h5>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Financial Transactions -->
            <div class="card mb-4">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <span>Financial Transactions</span>
                    <?php if ($is_admin): ?>
                        <div>
                            <button class="btn btn-sm btn-outline-secondary" onclick="printTable('financialsTable')">Print</button>
                            <a href="api/export_financials.php" class="btn btn-sm btn-outline-success">Export All to CSV</a>
                        </div>
                    <?php endif; ?>
                </div>
                <div class="card-body">
                    <div class="mb-3">
                        <label for="recordsPerPageTransactions" class="form-label">Records per page:</label>
                        <select id="recordsPerPageTransactions" class="form-select w-auto d-inline-block" onchange="changeRecordsPerPage(this, 'transactions')">
                            <option value="10" <?php echo $pagination['transactions']['per_page'] == 10 ? 'selected' : ''; ?>>10</option>
                            <option value="25" <?php echo $pagination['transactions']['per_page'] == 25 ? 'selected' : ''; ?>>25</option>
                            <option value="50" <?php echo $pagination['transactions']['per_page'] == 50 ? 'selected' : ''; ?>>50</option>
                            <option value="100" <?php echo $pagination['transactions']['per_page'] == 100 ? 'selected' : ''; ?>>100</option>
                        </select>
                    </div>
                    <div class="table-responsive">
                        <table class="table table-striped table-hover" id="financialsTable">
                            <thead>
                                <tr>
                                    <th>Member ID</th>
                                    <th>Member Name</th>
                                    <th>Payment Type</th>
                                    <th>Amount Paid</th>
                                    <th>Payment Status</th>
                                    <th>Membership Status</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($payments)): ?>
                                    <tr><td colspan="6" class="text-center">No financial data found.</td></tr>
                                <?php else: ?>
                                    <?php foreach ($payments as $payment): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($payment['member_id']); ?></td>
                                            <td><?php echo htmlspecialchars($payment['full_name']); ?></td>
                                            <td><?php echo htmlspecialchars($payment['payment_type']); ?></td>
                                            <td><?php echo htmlspecialchars($currency_symbol . ' ' . number_format($payment['amount'], 2)); ?></td>
                                            <td><span class="badge bg-<?php echo $payment['payment_status'] === 'Paid' ? 'success' : 'danger'; ?>"><?php echo htmlspecialchars($payment['payment_status']); ?></span></td>
                                            <td><?php echo htmlspecialchars(ucfirst($payment['membership_status'])); ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                    <?php if ($total_pages_transactions > 1): ?>
                        <nav aria-label="Transactions page navigation">
                            <ul class="pagination justify-content-center mt-3">
                                <li class="page-item <?php echo $pagination['transactions']['page'] <= 1 ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page_transactions=<?php echo $pagination['transactions']['page'] - 1; ?>&per_page_transactions=<?php echo $pagination['transactions']['per_page']; ?>&<?php echo http_build_query(array_diff_key($_GET, array_flip(['page_transactions', 'per_page_transactions']))); ?>">Previous</a>
                                </li>
                                <?php
                                $start_page = max(1, $pagination['transactions']['page'] - 2);
                                $end_page = min($total_pages_transactions, $pagination['transactions']['page'] + 2);
                                if ($start_page > 1) {
                                    echo '<li class="page-item"><a class="page-link" href="?page_transactions=1&per_page_transactions=' . $pagination['transactions']['per_page'] . '&' . http_build_query(array_diff_key($_GET, array_flip(['page_transactions', 'per_page_transactions']))) . '">1</a></li>';
                                    if ($start_page > 2) {
                                        echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                                    }
                                }
                                for ($i = $start_page; $i <= $end_page; $i++) {
                                    echo '<li class="page-item ' . ($i == $pagination['transactions']['page'] ? 'active' : '') . '">';
                                    echo '<a class="page-link" href="?page_transactions=' . $i . '&per_page_transactions=' . $pagination['transactions']['per_page'] . '&' . http_build_query(array_diff_key($_GET, array_flip(['page_transactions', 'per_page_transactions']))) . '">' . $i . '</a>';
                                    echo '</li>';
                                }
                                if ($end_page < $total_pages_transactions) {
                                    if ($end_page < $total_pages_transactions - 1) {
                                        echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                                    }
                                    echo '<li class="page-item"><a class="page-link" href="?page_transactions=' . $total_pages_transactions . '&per_page_transactions=' . $pagination['transactions']['per_page'] . '&' . http_build_query(array_diff_key($_GET, array_flip(['page_transactions', 'per_page_transactions']))) . '">' . $total_pages_transactions . '</a></li>';
                                }
                                ?>
                                <li class="page-item <?php echo $pagination['transactions']['page'] >= $total_pages_transactions ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page_transactions=<?php echo $pagination['transactions']['page'] + 1; ?>&per_page_transactions=<?php echo $pagination['transactions']['per_page']; ?>&<?php echo http_build_query(array_diff_key($_GET, array_flip(['page_transactions', 'per_page_transactions']))); ?>">Next</a>
                                </li>
                            </ul>
                        </nav>
                    <?php endif; ?>
                    <?php if (!$is_admin): ?>
                        <p class="text-muted mt-2">Note: Print and export functionality is available to admins only.</p>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Recent Debtors -->
            <div class="card mb-4">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <span>Recent Debtors</span>
                    <?php if ($is_admin): ?>
                        <button class="btn btn-sm btn-outline-secondary" onclick="printTable('recentDebtorsTable')">Print</button>
                    <?php endif; ?>
                </div>
                <div class="card-body">
                    <div class="mb-3">
                        <label for="recordsPerPageRecentDebtors" class="form-label">Records per page:</label>
                        <select id="recordsPerPageRecentDebtors" class="form-select w-auto d-inline-block" onchange="changeRecordsPerPage(this, 'recent_debtors')">
                            <option value="10" <?php echo $pagination['recent_debtors']['per_page'] == 10 ? 'selected' : ''; ?>>10</option>
                            <option value="25" <?php echo $pagination['recent_debtors']['per_page'] == 25 ? 'selected' : ''; ?>>25</option>
                            <option value="50" <?php echo $pagination['recent_debtors']['per_page'] == 50 ? 'selected' : ''; ?>>50</option>
                            <option value="100" <?php echo $pagination['recent_debtors']['per_page'] == 100 ? 'selected' : ''; ?>>100</option>
                        </select>
                    </div>
                    <div class="table-responsive">
                        <table class="table table-striped table-hover" id="recentDebtorsTable">
                            <thead>
                                <tr>
                                    <th>Member ID</th>
                                    <th>Member Name</th>
                                    <th>Payment Type</th>
                                    <th>Required Amount</th>
                                    <th>Total Paid</th>
                                    <th>Debt</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($recent_debtors)): ?>
                                    <tr><td colspan="6" class="text-center">No debtors found.</td></tr>
                                <?php else: ?>
                                    <?php foreach ($recent_debtors as $debtor): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($debtor['member_id']); ?></td>
                                            <td><?php echo htmlspecialchars($debtor['full_name']); ?></td>
                                            <td><?php echo htmlspecialchars($debtor['payment_type']); ?></td>
                                            <td><?php echo htmlspecialchars($currency_symbol . ' ' . number_format($debtor['required_amount'], 2)); ?></td>
                                            <td><?php echo htmlspecialchars($currency_symbol . ' ' . number_format($debtor['total_paid'], 2)); ?></td>
                                            <td><?php echo htmlspecialchars($currency_symbol . ' ' . number_format($debtor['debt'], 2)); ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                    <?php if ($total_pages_recent_debtors > 1): ?>
                        <nav aria-label="Recent Debtors page navigation">
                            <ul class="pagination justify-content-center mt-3">
                                <li class="page-item <?php echo $pagination['recent_debtors']['page'] <= 1 ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page_recent_debtors=<?php echo $pagination['recent_debtors']['page'] - 1; ?>&per_page_recent_debtors=<?php echo $pagination['recent_debtors']['per_page']; ?>&<?php echo http_build_query(array_diff_key($_GET, array_flip(['page_recent_debtors', 'per_page_recent_debtors']))); ?>">Previous</a>
                                </li>
                                <?php
                                $start_page = max(1, $pagination['recent_debtors']['page'] - 2);
                                $end_page = min($total_pages_recent_debtors, $pagination['recent_debtors']['page'] + 2);
                                if ($start_page > 1) {
                                    echo '<li class="page-item"><a class="page-link" href="?page_recent_debtors=1&per_page_recent_debtors=' . $pagination['recent_debtors']['per_page'] . '&' . http_build_query(array_diff_key($_GET, array_flip(['page_recent_debtors', 'per_page_recent_debtors']))) . '">1</a></li>';
                                    if ($start_page > 2) {
                                        echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                                    }
                                }
                                for ($i = $start_page; $i <= $end_page; $i++) {
                                    echo '<li class="page-item ' . ($i == $pagination['recent_debtors']['page'] ? 'active' : '') . '">';
                                    echo '<a class="page-link" href="?page_recent_debtors=' . $i . '&per_page_recent_debtors=' . $pagination['recent_debtors']['per_page'] . '&' . http_build_query(array_diff_key($_GET, array_flip(['page_recent_debtors', 'per_page_recent_debtors']))) . '">' . $i . '</a>';
                                    echo '</li>';
                                }
                                if ($end_page < $total_pages_recent_debtors) {
                                    if ($end_page < $total_pages_recent_debtors - 1) {
                                        echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                                    }
                                    echo '<li class="page-item"><a class="page-link" href="?page_recent_debtors=' . $total_pages_recent_debtors . '&per_page_recent_debtors=' . $pagination['recent_debtors']['per_page'] . '&' . http_build_query(array_diff_key($_GET, array_flip(['page_recent_debtors', 'per_page_recent_debtors']))) . '">' . $total_pages_recent_debtors . '</a></li>';
                                }
                                ?>
                                <li class="page-item <?php echo $pagination['recent_debtors']['page'] >= $total_pages_recent_debtors ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page_recent_debtors=<?php echo $pagination['recent_debtors']['page'] + 1; ?>&per_page_recent_debtors=<?php echo $pagination['recent_debtors']['per_page']; ?>&<?php echo http_build_query(array_diff_key($_GET, array_flip(['page_recent_debtors', 'per_page_recent_debtors']))); ?>">Next</a>
                                </li>
                            </ul>
                        </nav>
                    <?php endif; ?>
                    <?php if (!$is_admin): ?>
                        <p class="text-muted mt-2">Note: Print functionality is available to admins only.</p>
                    <?php endif; ?>
                </div>
            </div>

            <!-- All Debtors -->
            <div class="card mb-4">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <span>All Debtors</span>
                    <?php if ($is_admin): ?>
                        <button class="btn btn-sm btn-outline-secondary" onclick="printTable('debtorsTable')">Print</button>
                    <?php endif; ?>
                </div>
                <div class="card-body">
                    <div class="mb-3">
                        <label for="recordsPerPageAllDebtors" class="form-label">Records per page:</label>
                        <select id="recordsPerPageAllDebtors" class="form-select w-auto d-inline-block" onchange="changeRecordsPerPage(this, 'all_debtors')">
                            <option value="10" <?php echo $pagination['all_debtors']['per_page'] == 10 ? 'selected' : ''; ?>>10</option>
                            <option value="25" <?php echo $pagination['all_debtors']['per_page'] == 25 ? 'selected' : ''; ?>>25</option>
                            <option value="50" <?php echo $pagination['all_debtors']['per_page'] == 50 ? 'selected' : ''; ?>>50</option>
                            <option value="100" <?php echo $pagination['all_debtors']['per_page'] == 100 ? 'selected' : ''; ?>>100</option>
                        </select>
                    </div>
                    <div class="table-responsive">
                        <table class="table table-striped table-hover" id="debtorsTable">
                            <thead>
                                <tr>
                                    <th>Member ID</th>
                                    <th>Member Name</th>
                                    <th>Payment Type</th>
                                    <th>Required Amount</th>
                                    <th>Total Paid</th>
                                    <th>Debt</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($debtors)): ?>
                                    <tr><td colspan="6" class="text-center">No debtors found.</td></tr>
                                <?php else: ?>
                                    <?php foreach ($debtors as $debtor): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($debtor['member_id']); ?></td>
                                            <td><?php echo htmlspecialchars($debtor['full_name']); ?></td>
                                            <td><?php echo htmlspecialchars($debtor['payment_type']); ?></td>
                                            <td><?php echo htmlspecialchars($currency_symbol . ' ' . number_format($debtor['required_amount'], 2)); ?></td>
                                            <td><?php echo htmlspecialchars($currency_symbol . ' ' . number_format($debtor['total_paid'], 2)); ?></td>
                                            <td><?php echo htmlspecialchars($currency_symbol . ' ' . number_format($debtor['debt'], 2)); ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                    <?php if ($total_pages_debtors > 1): ?>
                        <nav aria-label="All Debtors page navigation">
                            <ul class="pagination justify-content-center mt-3">
                                <li class="page-item <?php echo $pagination['all_debtors']['page'] <= 1 ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page_all_debtors=<?php echo $pagination['all_debtors']['page'] - 1; ?>&per_page_all_debtors=<?php echo $pagination['all_debtors']['per_page']; ?>&<?php echo http_build_query(array_diff_key($_GET, array_flip(['page_all_debtors', 'per_page_all_debtors']))); ?>">Previous</a>
                                </li>
                                <?php
                                $start_page = max(1, $pagination['all_debtors']['page'] - 2);
                                $end_page = min($total_pages_debtors, $pagination['all_debtors']['page'] + 2);
                                if ($start_page > 1) {
                                    echo '<li class="page-item"><a class="page-link" href="?page_all_debtors=1&per_page_all_debtors=' . $pagination['all_debtors']['per_page'] . '&' . http_build_query(array_diff_key($_GET, array_flip(['page_all_debtors', 'per_page_all_debtors']))) . '">1</a></li>';
                                    if ($start_page > 2) {
                                        echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                                    }
                                }
                                for ($i = $start_page; $i <= $end_page; $i++) {
                                    echo '<li class="page-item ' . ($i == $pagination['all_debtors']['page'] ? 'active' : '') . '">';
                                    echo '<a class="page-link" href="?page_all_debtors=' . $i . '&per_page_all_debtors=' . $pagination['all_debtors']['per_page'] . '&' . http_build_query(array_diff_key($_GET, array_flip(['page_all_debtors', 'per_page_all_debtors']))) . '">' . $i . '</a>';
                                    echo '</li>';
                                }
                                if ($end_page < $total_pages_debtors) {
                                    if ($end_page < $total_pages_debtors - 1) {
                                        echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                                    }
                                    echo '<li class="page-item"><a class="page-link" href="?page_all_debtors=' . $total_pages_debtors . '&per_page_all_debtors=' . $pagination['all_debtors']['per_page'] . '&' . http_build_query(array_diff_key($_GET, array_flip(['page_all_debtors', 'per_page_all_debtors']))) . '">' . $total_pages_debtors . '</a></li>';
                                }
                                ?>
                                <li class="page-item <?php echo $pagination['all_debtors']['page'] >= $total_pages_debtors ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page_all_debtors=<?php echo $pagination['all_debtors']['page'] + 1; ?>&per_page_all_debtors=<?php echo $pagination['all_debtors']['per_page']; ?>&<?php echo http_build_query(array_diff_key($_GET, array_flip(['page_all_debtors', 'per_page_all_debtors']))); ?>">Next</a>
                                </li>
                            </ul>
                        </nav>
                    <?php endif; ?>
                    <?php if (!$is_admin): ?>
                        <p class="text-muted mt-2">Note: Print functionality is available to admins only.</p>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Expense Summary -->
            <div class="card mb-4">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <span>Expense Summary</span>
                    <?php if ($is_admin): ?>
                        <button class="btn btn-sm btn-outline-secondary" onclick="printTable('expenseSummaryTable')">Print</button>
                    <?php endif; ?>
                </div>
                <div class="card-body">
                    <div class="mb-3">
                        <label for="recordsPerPageExpenses" class="form-label">Records per page:</label>
                        <select id="recordsPerPageExpenses" class="form-select w-auto d-inline-block" onchange="changeRecordsPerPage(this, 'expenses')">
                            <option value="10" <?php echo $pagination['expenses']['per_page'] == 10 ? 'selected' : ''; ?>>10</option>
                            <option value="25" <?php echo $pagination['expenses']['per_page'] == 25 ? 'selected' : ''; ?>>25</option>
                            <option value="50" <?php echo $pagination['expenses']['per_page'] == 50 ? 'selected' : ''; ?>>50</option>
                            <option value="100" <?php echo $pagination['expenses']['per_page'] == 100 ? 'selected' : ''; ?>>100</option>
                        </select>
                    </div>
                    <div class="table-responsive">
                        <table class="table table-striped table-hover" id="expenseSummaryTable">
                            <thead>
                                <tr>
                                    <th>Expense Type</th>
                                    <th>Total Amount</th>
                                    <th>Count</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($expenses_summary)): ?>
                                    <tr><td colspan="3" class="text-center">No expense summary found.</td></tr>
                                <?php else: ?>
                                    <?php foreach ($expenses_summary as $summary): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($summary['expense_type']); ?></td>
                                            <td><?php echo htmlspecialchars($currency_symbol . ' ' . number_format($summary['total_amount'], 2)); ?></td>
                                            <td><?php echo htmlspecialchars($summary['count']); ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                    <?php if ($total_pages_expenses > 1): ?>
                        <nav aria-label="Expense Summary page navigation">
                            <ul class="pagination justify-content-center mt-3">
                                <li class="page-item <?php echo $pagination['expenses']['page'] <= 1 ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page_expenses=<?php echo $pagination['expenses']['page'] - 1; ?>&per_page_expenses=<?php echo $pagination['expenses']['per_page']; ?>&<?php echo http_build_query(array_diff_key($_GET, array_flip(['page_expenses', 'per_page_expenses']))); ?>">Previous</a>
                                </li>
                                <?php
                                $start_page = max(1, $pagination['expenses']['page'] - 2);
                                $end_page = min($total_pages_expenses, $pagination['expenses']['page'] + 2);
                                if ($start_page > 1) {
                                    echo '<li class="page-item"><a class="page-link" href="?page_expenses=1&per_page_expenses=' . $pagination['expenses']['per_page'] . '&' . http_build_query(array_diff_key($_GET, array_flip(['page_expenses', 'per_page_expenses']))) . '">1</a></li>';
                                    if ($start_page > 2) {
                                        echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                                    }
                                }
                                for ($i = $start_page; $i <= $end_page; $i++) {
                                    echo '<li class="page-item ' . ($i == $pagination['expenses']['page'] ? 'active' : '') . '">';
                                    echo '<a class="page-link" href="?page_expenses=' . $i . '&per_page_expenses=' . $pagination['expenses']['per_page'] . '&' . http_build_query(array_diff_key($_GET, array_flip(['page_expenses', 'per_page_expenses']))) . '">' . $i . '</a>';
                                    echo '</li>';
                                }
                                if ($end_page < $total_pages_expenses) {
                                    if ($end_page < $total_pages_expenses - 1) {
                                        echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                                    }
                                    echo '<li class="page-item"><a class="page-link" href="?page_expenses=' . $total_pages_expenses . '&per_page_expenses=' . $pagination['expenses']['per_page'] . '&' . http_build_query(array_diff_key($_GET, array_flip(['page_expenses', 'per_page_expenses']))) . '">' . $total_pages_expenses . '</a></li>';
                                }
                                ?>
                                <li class="page-item <?php echo $pagination['expenses']['page'] >= $total_pages_expenses ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page_expenses=<?php echo $pagination['expenses']['page'] + 1; ?>&per_page_expenses=<?php echo $pagination['expenses']['per_page']; ?>&<?php echo http_build_query(array_diff_key($_GET, array_flip(['page_expenses', 'per_page_expenses']))); ?>">Next</a>
                                </li>
                            </ul>
                        </nav>
                    <?php endif; ?>
                    <?php if (!$is_admin): ?>
                        <p class="text-muted mt-2">Note: Print functionality is available to admins only.</p>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Benefit Types Summary -->
            <div class="card mb-4">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <span>Benefit Types Summary</span>
                    <?php if ($is_admin): ?>
                        <button class="btn btn-sm btn-outline-secondary" onclick="printTable('benefitSummaryTable')">Print</button>
                    <?php endif; ?>
                </div>
                <div class="card-body">
                    <div class="mb-3">
                        <label for="recordsPerPageBenefits" class="form-label">Records per page:</label>
                        <select id="recordsPerPageBenefits" class="form-select w-auto d-inline-block" onchange="changeRecordsPerPage(this, 'benefits')">
                            <option value="10" <?php echo $pagination['benefits']['per_page'] == 10 ? 'selected' : ''; ?>>10</option>
                            <option value="25" <?php echo $pagination['benefits']['per_page'] == 25 ? 'selected' : ''; ?>>25</option>
                            <option value="50" <?php echo $pagination['benefits']['per_page'] == 50 ? 'selected' : ''; ?>>50</option>
                            <option value="100" <?php echo $pagination['benefits']['per_page'] == 100 ? 'selected' : ''; ?>>100</option>
                        </select>
                    </div>
                    <div class="table-responsive">
                        <table class="table table-striped table-hover" id="benefitSummaryTable">
                            <thead>
                                <tr>
                                    <th>Benefit Type</th>
                                    <th>Total Amount</th>
                                    <th>Count</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($benefits_summary)): ?>
                                    <tr><td colspan="3" class="text-center">No benefit type summary found.</td></tr>
                                <?php else: ?>
                                    <?php foreach ($benefits_summary as $summary): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($summary['benefit_type']); ?></td>
                                            <td><?php echo htmlspecialchars($currency_symbol . ' ' . number_format($summary['total_amount'], 2)); ?></td>
                                            <td><?php echo htmlspecialchars($summary['count']); ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                    <?php if ($total_pages_benefits > 1): ?>
                        <nav aria-label="Benefit Types Summary page navigation">
                            <ul class="pagination justify-content-center mt-3">
                                <li class="page-item <?php echo $pagination['benefits']['page'] <= 1 ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page_benefits=<?php echo $pagination['benefits']['page'] - 1; ?>&per_page_benefits=<?php echo $pagination['benefits']['per_page']; ?>&<?php echo http_build_query(array_diff_key($_GET, array_flip(['page_benefits', 'per_page_benefits']))); ?>">Previous</a>
                                </li>
                                <?php
                                $start_page = max(1, $pagination['benefits']['page'] - 2);
                                $end_page = min($total_pages_benefits, $pagination['benefits']['page'] + 2);
                                if ($start_page > 1) {
                                    echo '<li class="page-item"><a class="page-link" href="?page_benefits=1&per_page_benefits=' . $pagination['benefits']['per_page'] . '&' . http_build_query(array_diff_key($_GET, array_flip(['page_benefits', 'per_page_benefits']))) . '">1</a></li>';
                                    if ($start_page > 2) {
                                        echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                                    }
                                }
                                for ($i = $start_page; $i <= $end_page; $i++) {
                                    echo '<li class="page-item ' . ($i == $pagination['benefits']['page'] ? 'active' : '') . '">';
                                    echo '<a class="page-link" href="?page_benefits=' . $i . '&per_page_benefits=' . $pagination['benefits']['per_page'] . '&' . http_build_query(array_diff_key($_GET, array_flip(['page_benefits', 'per_page_benefits']))) . '">' . $i . '</a>';
                                    echo '</li>';
                                }
                                if ($end_page < $total_pages_benefits) {
                                    if ($end_page < $total_pages_benefits - 1) {
                                        echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                                    }
                                    echo '<li class="page-item"><a class="page-link" href="?page_benefits=' . $total_pages_benefits . '&per_page_benefits=' . $pagination['benefits']['per_page'] . '&' . http_build_query(array_diff_key($_GET, array_flip(['page_benefits', 'per_page_benefits']))) . '">' . $total_pages_benefits . '</a></li>';
                                }
                                ?>
                                <li class="page-item <?php echo $pagination['benefits']['page'] >= $total_pages_benefits ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page_benefits=<?php echo $pagination['benefits']['page'] + 1; ?>&per_page_benefits=<?php echo $pagination['benefits']['per_page']; ?>&<?php echo http_build_query(array_diff_key($_GET, array_flip(['page_benefits', 'per_page_benefits']))); ?>">Next</a>
                                </li>
                            </ul>
                        </nav>
                    <?php endif; ?>
                    <?php if (!$is_admin): ?>
                        <p class="text-muted mt-2">Note: Print functionality is available to admins only.</p>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Payment Types Summary (admin-only) -->
            <?php if ($is_admin): ?>
                <div class="card mb-4">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <span>Payment Types Summary</span>
                        <button class="btn btn-sm btn-outline-secondary" onclick="printTable('paymentSummaryTable')">Print</button>
                    </div>
                    <div class="card-body">
                        <div class="mb-3">
                            <label for="recordsPerPagePaymentSummary" class="form-label">Records per page:</label>
                            <select id="recordsPerPagePaymentSummary" class="form-select w-auto d-inline-block" onchange="changeRecordsPerPage(this, 'payment_summary')">
                                <option value="10" <?php echo $pagination['payment_summary']['per_page'] == 10 ? 'selected' : ''; ?>>10</option>
                                <option value="25" <?php echo $pagination['payment_summary']['per_page'] == 25 ? 'selected' : ''; ?>>25</option>
                                <option value="50" <?php echo $pagination['payment_summary']['per_page'] == 50 ? 'selected' : ''; ?>>50</option>
                                <option value="100" <?php echo $pagination['payment_summary']['per_page'] == 100 ? 'selected' : ''; ?>>100</option>
                            </select>
                        </div>
                        <div class="table-responsive">
                            <table class="table table-striped table-hover" id="paymentSummaryTable">
                                <thead>
                                    <tr>
                                        <th>Payment Type</th>
                                        <th>Total Amount</th>
                                        <th>Count</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (empty($payment_summary)): ?>
                                        <tr><td colspan="3" class="text-center">No payment type summary found.</td></tr>
                                    <?php else: ?>
                                        <?php foreach ($payment_summary as $summary): ?>
                                            <tr>
                                                <td><?php echo htmlspecialchars($summary['payment_type']); ?></td>
                                                <td><?php echo htmlspecialchars($currency_symbol . ' ' . number_format($summary['total_amount'], 2)); ?></td>
                                                <td><?php echo htmlspecialchars($summary['count']); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                        <?php if ($total_pages_payment_summary > 1): ?>
                            <nav aria-label="Payment Types Summary page navigation">
                                <ul class="pagination justify-content-center mt-3">
                                    <li class="page-item <?php echo $pagination['payment_summary']['page'] <= 1 ? 'disabled' : ''; ?>">
                                        <a class="page-link" href="?page_payment_summary=<?php echo $pagination['payment_summary']['page'] - 1; ?>&per_page_payment_summary=<?php echo $pagination['payment_summary']['per_page']; ?>&<?php echo http_build_query(array_diff_key($_GET, array_flip(['page_payment_summary', 'per_page_payment_summary']))); ?>">Previous</a>
                                    </li>
                                    <?php
                                    $start_page = max(1, $pagination['payment_summary']['page'] - 2);
                                    $end_page = min($total_pages_payment_summary, $pagination['payment_summary']['page'] + 2);
                                    if ($start_page > 1) {
                                        echo '<li class="page-item"><a class="page-link" href="?page_payment_summary=1&per_page_payment_summary=' . $pagination['payment_summary']['per_page'] . '&' . http_build_query(array_diff_key($_GET, array_flip(['page_payment_summary', 'per_page_payment_summary']))) . '">1</a></li>';
                                    if ($start_page > 2) {
                                        echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                                    }
                                }
                                for ($i = $start_page; $i <= $end_page; $i++) {
                                    echo '<li class="page-item ' . ($i == $pagination['payment_summary']['page'] ? 'active' : '') . '">';
                                    echo '<a class="page-link" href="?page_payment_summary=' . $i . '&per_page_payment_summary=' . $pagination['payment_summary']['per_page'] . '&' . http_build_query(array_diff_key($_GET, array_flip(['page_payment_summary', 'per_page_payment_summary']))) . '">' . $i . '</a>';
                                    echo '</li>';
                                }
                                if ($end_page < $total_pages_payment_summary) {
                                    if ($end_page < $total_pages_payment_summary - 1) {
                                        echo '<li class="page-item disabled"><span class="page-link">...</span></li>';
                                    }
                                    echo '<li class="page-item"><a class="page-link" href="?page_payment_summary=' . $total_pages_payment_summary . '&per_page_payment_summary=' . $pagination['payment_summary']['per_page'] . '&' . http_build_query(array_diff_key($_GET, array_flip(['page_payment_summary', 'per_page_payment_summary']))) . '">' . $total_pages_payment_summary . '</a></li>';
                                }
                                ?>
                                <li class="page-item <?php echo $pagination['payment_summary']['page'] >= $total_pages_payment_summary ? 'disabled' : ''; ?>">
                                    <a class="page-link" href="?page_payment_summary=<?php echo $pagination['payment_summary']['page'] + 1; ?>&per_page_payment_summary=<?php echo $pagination['payment_summary']['per_page']; ?>&<?php echo http_build_query(array_diff_key($_GET, array_flip(['page_payment_summary', 'per_page_payment_summary']))); ?>">Next</a>
                                </li>
                            </ul>
                        </nav>
                    <?php endif; ?>
                </div>
            <?php else: ?>
                <p class="text-muted mt-2">Note: Payment Types Summary is available to admins only.</p>
            <?php endif; ?>
        </main>
    </div>
</div>

<script>
function printTable(tableId) {
    const table = document.getElementById(tableId);
    const originalContent = document.body.innerHTML;
    const printContent = `<style>
        body { font-family: sans-serif; }
        table { width: 100%; border-collapse: collapse; margin-bottom: 20px; }
        th, td { border: 1px solid #ccc; padding: 8px; text-align: left; }
        th { background-color: #f2f2f2; }
        h2, h3 { text-align: center; }
        .table-primary th { background-color: #b8daff; }
        @media print {
            .no-print { display: none; }
        }
    </style>
    <div class="no-print"><h2>Club Financials</h2></div>
    <h3>${tableId === 'financialsTable' ? 'Financial Transactions' : tableId === 'recentDebtorsTable' ? 'Recent Debtors' : tableId === 'debtorsTable' ? 'All Debtors' : tableId === 'paymentSummaryTable' ? 'Payment Types Summary' : tableId === 'benefitSummaryTable' ? 'Benefit Types Summary' : 'Expense Summary'}</h3>
    ${table.outerHTML}`;

    document.body.innerHTML = printContent;
    window.print();
    document.body.innerHTML = originalContent;
    window.location.reload();
}

function changeRecordsPerPage(select, table) {
    const perPage = select.value;
    const params = new URLSearchParams(window.location.search);
    params.set(`per_page_${table}`, perPage);
    params.set(`page_${table}`, '1');
    window.location.href = `?${params.toString()}`;
}
</script>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js" integrity="sha384-YvpcrYf0tY3lHB60NNkmXc5s9fDVZLESaAA55NDzOxhy9GkcIdslK1eN7N6jIeHz" crossorigin="anonymous"></script>
<?php require_once 'includes/footer.php'; ?>