<?php
function get_total_counts($link, $record_type, $time_period, $filter_date, $type_filter, $search, $date_field, $pagination) {
    $tables = ['payments', 'expenses', 'debts', 'benefits'];
    $total_counts = [];
    $type_filter_condition = validate_type_filter($type_filter, $record_type);
    $time_filter = build_time_filter($time_period, $date_field, $record_type);
    $time_filter_debts = str_replace('created_at', ($date_field === 'due_date' ? 'po.due_date' : 'po.created_at'), $time_filter);

    foreach ($tables as $table) {
        $sql_count = '';
        $count_bind_types = '';
        $count_bind_params = [];
        $include_filters = ($record_type === 'all' || $record_type === $table);
        $search_condition = '';

        if ($table === 'payments') {
            $sql_count = "SELECT COUNT(*) as total FROM payments p JOIN users u ON p.user_id = u.id WHERE payment_status = 'Paid'";
            if ($include_filters) {
                $sql_count .= str_replace('created_at', 'p.created_at', $time_filter);
                if ($time_period === 'date') {
                    $count_bind_types .= 's';
                    $count_bind_params[] = $filter_date;
                }
                if ($type_filter && strpos($type_filter, 'payment_') === 0) {
                    $sql_count .= $type_filter_condition;
                    $count_bind_types .= 's';
                    $count_bind_params[] = explode('_', $type_filter, 2)[1];
                }
            } else {
                $sql_count .= ' AND FALSE';
            }
            $search_condition = " AND (u.full_name LIKE ? OR p.payment_type LIKE ? OR CAST(p.id AS CHAR) LIKE ?)";
        } elseif ($table === 'expenses') {
            $sql_count = "SELECT COUNT(*) as total FROM expenses e LEFT JOIN expense_types et ON e.expense_type_id = et.id WHERE 1=1";
            if ($include_filters) {
                $sql_count .= str_replace('created_at', 'e.created_at', $time_filter);
                if ($time_period === 'date') {
                    $count_bind_types .= 's';
                    $count_bind_params[] = $filter_date;
                }
                if ($type_filter && strpos($type_filter, 'expense_') === 0) {
                    $sql_count .= $type_filter_condition;
                    $count_bind_types .= 's';
                    $count_bind_params[] = explode('_', $type_filter, 2)[1];
                }
            } else {
                $sql_count .= ' AND FALSE';
            }
            $search_condition = " AND (et.type_name LIKE ? OR e.description LIKE ? OR CAST(e.id AS CHAR) LIKE ?)";
        } elseif ($table === 'benefits') {
            $sql_count = "SELECT COUNT(*) as total FROM member_benefits b JOIN users u ON b.member_id = u.id JOIN benefit_types bt ON b.benefit_type_id = bt.id WHERE 1=1";
            if ($include_filters) {
                $sql_count .= str_replace('created_at', 'b.created_at', $time_filter);
                if ($time_period === 'date') {
                    $count_bind_types .= 's';
                    $count_bind_params[] = $filter_date;
                }
                if ($type_filter && strpos($type_filter, 'benefit_') === 0) {
                    $sql_count .= $type_filter_condition;
                    $count_bind_types .= 's';
                    $count_bind_params[] = explode('_', $type_filter, 2)[1];
                }
            } else {
                $sql_count .= ' AND FALSE';
            }
            $search_condition = " AND (u.full_name LIKE ? OR bt.type_name LIKE ? OR CAST(b.id AS CHAR) LIKE ?)";
        } elseif ($table === 'debts') {
            $sql_count = "SELECT COUNT(*) as total 
                          FROM payment_obligations po 
                          JOIN users u ON po.user_id = u.id 
                          JOIN payment_types pt ON po.payment_type = pt.type_name 
                          LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status = 'Paid'
                          WHERE po.deleted_at IS NULL 
                              AND u.role = 'member' 
                              AND (u.deleted_at IS NULL OR u.deleted_at > NOW())
                              AND COALESCE(po.required_amount, 0) > 0";
            if ($include_filters) {
                $sql_count .= $time_filter_debts;
                if ($time_period === 'date') {
                    $count_bind_types .= 's';
                    $count_bind_params[] = $filter_date;
                }
                if ($type_filter && strpos($type_filter, 'payment_') === 0) {
                    $adjusted_condition = str_replace('p.payment_type', 'pt.type_name', $type_filter_condition);
                    $sql_count .= $adjusted_condition;
                    $count_bind_types .= 's';
                    $count_bind_params[] = explode('_', $type_filter, 2)[1];
                }
            } else {
                $sql_count .= ' AND FALSE';
            }
            $search_condition = " AND (u.full_name LIKE ? OR pt.type_name LIKE ? OR CAST(po.id AS CHAR) LIKE ?)";
        }

        if (!empty($search)) {
            $sql_count .= $search_condition;
            $count_bind_types .= 'sss';
            $search_param = "%$search%";
            $count_bind_params[] = $search_param;
            $count_bind_params[] = $search_param;
            $count_bind_params[] = $search_param;
        }

        if ($table === 'debts') {
            $sql_count .= " GROUP BY po.id, u.id, po.required_amount, po.due_date HAVING (COALESCE(po.required_amount, 0) - COALESCE(SUM(p.amount), 0)) > 0";
        }

        if (!$stmt = mysqli_prepare($link, $sql_count)) {
            error_log("Failed to prepare statement for $table count: " . mysqli_error($link) . "; SQL: $sql_count");
            $total_counts[$table] = 0;
            continue;
        }
        if ($count_bind_types && !mysqli_stmt_bind_param($stmt, $count_bind_types, ...$count_bind_params)) {
            error_log("Failed to bind parameters for $table count: " . mysqli_stmt_error($stmt) . "; SQL: $sql_count");
            $total_counts[$table] = 0;
            mysqli_stmt_close($stmt);
            continue;
        }
        if (!mysqli_stmt_execute($stmt)) {
            error_log("Failed to execute $table count query: " . mysqli_stmt_error($stmt) . "; SQL: $sql_count");
            $total_counts[$table] = 0;
            mysqli_stmt_close($stmt);
            continue;
        }
        $result = mysqli_stmt_get_result($stmt);
        if ($table === 'debts') {
            $total_counts[$table] = mysqli_num_rows($result);
        } else {
            $total_counts[$table] = mysqli_fetch_assoc($result)['total'] ?? 0;
        }
        mysqli_stmt_close($stmt);
    }
    return $total_counts;
}

function get_total_payments($link, $record_type, $time_period, $filter_date, $type_filter) {
    $sql = "SELECT COALESCE(SUM(amount), 0) AS total_payments FROM payments p WHERE payment_status = 'Paid'";
    $bind_types = '';
    $bind_params = [];
    $type_filter_condition = validate_type_filter($type_filter, $record_type);
    if ($record_type === 'all' || $record_type === 'payments') {
        $sql .= str_replace('created_at', 'p.created_at', build_time_filter($time_period, 'created_at', $record_type));
        if ($time_period === 'date') {
            $bind_types .= 's';
            $bind_params[] = $filter_date;
        }
        if ($type_filter && strpos($type_filter, 'payment_') === 0) {
            $sql .= $type_filter_condition;
            $bind_types .= 's';
            $bind_params[] = explode('_', $type_filter, 2)[1];
        }
    } else {
        $sql .= ' AND FALSE';
    }
    if ($stmt = mysqli_prepare($link, $sql)) {
        if ($bind_types) {
            mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
        }
        if (mysqli_stmt_execute($stmt)) {
            $total = mysqli_fetch_assoc(mysqli_stmt_get_result($stmt))['total_payments'] ?? 0;
            mysqli_stmt_close($stmt);
            return $total;
        }
        error_log("Failed to execute total_payments query: " . mysqli_stmt_error($stmt) . "; SQL: $sql");
        mysqli_stmt_close($stmt);
    }
    error_log("Failed to prepare total_payments query: " . mysqli_error($link) . "; SQL: $sql");
    return 0;
}

function get_total_expenses($link, $record_type, $time_period, $filter_date, $type_filter) {
    $sql = "SELECT COALESCE(SUM(amount), 0) AS total_expenses FROM expenses e LEFT JOIN expense_types et ON e.expense_type_id = et.id WHERE 1=1";
    $bind_types = '';
    $bind_params = [];
    $type_filter_condition = validate_type_filter($type_filter, $record_type);
    if ($record_type === 'all' || $record_type === 'expenses') {
        $sql .= str_replace('created_at', 'e.created_at', build_time_filter($time_period, 'created_at', $record_type));
        if ($time_period === 'date') {
            $bind_types .= 's';
            $bind_params[] = $filter_date;
        }
        if ($type_filter && strpos($type_filter, 'expense_') === 0) {
            $sql .= $type_filter_condition;
            $bind_types .= 's';
            $bind_params[] = explode('_', $type_filter, 2)[1];
        }
    } else {
        $sql .= ' AND FALSE';
    }
    if ($stmt = mysqli_prepare($link, $sql)) {
        if ($bind_types) {
            mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
        }
        if (mysqli_stmt_execute($stmt)) {
            $total = mysqli_fetch_assoc(mysqli_stmt_get_result($stmt))['total_expenses'] ?? 0;
            mysqli_stmt_close($stmt);
            return $total;
        }
        error_log("Failed to execute total_expenses query: " . mysqli_stmt_error($stmt) . "; SQL: $sql");
        mysqli_stmt_close($stmt);
    }
    error_log("Failed to prepare total_expenses query: " . mysqli_error($link) . "; SQL: $sql");
    return 0;
}

function get_total_benefits($link, $record_type, $time_period, $filter_date, $type_filter) {
    $sql = "SELECT COALESCE(SUM(amount), 0) AS total_benefits FROM member_benefits b JOIN benefit_types bt ON b.benefit_type_id = bt.id WHERE 1=1";
    $bind_types = '';
    $bind_params = [];
    $type_filter_condition = validate_type_filter($type_filter, $record_type);
    if ($record_type === 'all' || $record_type === 'benefits') {
        $sql .= str_replace('created_at', 'b.created_at', build_time_filter($time_period, 'created_at', $record_type));
        if ($time_period === 'date') {
            $bind_types .= 's';
            $bind_params[] = $filter_date;
        }
        if ($type_filter && strpos($type_filter, 'benefit_') === 0) {
            $sql .= $type_filter_condition;
            $bind_types .= 's';
            $bind_params[] = explode('_', $type_filter, 2)[1];
        }
    } else {
        $sql .= ' AND FALSE';
    }
    if ($stmt = mysqli_prepare($link, $sql)) {
        if ($bind_types) {
            mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
        }
        if (mysqli_stmt_execute($stmt)) {
            $total = mysqli_fetch_assoc(mysqli_stmt_get_result($stmt))['total_benefits'] ?? 0;
            mysqli_stmt_close($stmt);
            return $total;
        }
        error_log("Failed to execute total_benefits query: " . mysqli_stmt_error($stmt) . "; SQL: $sql");
        mysqli_stmt_close($stmt);
    }
    error_log("Failed to prepare total_benefits query: " . mysqli_error($link) . "; SQL: $sql");
    return 0;
}

function get_total_debts($link, $record_type, $time_period, $filter_date, $type_filter, $date_field) {
    $sql = "SELECT COALESCE(SUM(debt), 0) AS total_debt 
            FROM (
                SELECT 
                    po.id,
                    (COALESCE(po.required_amount, 0) - COALESCE(SUM(p.amount), 0)) AS debt
                FROM payment_obligations po
                JOIN users u ON po.user_id = u.id
                JOIN payment_types pt ON po.payment_type = pt.type_name
                LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status = 'Paid'
                WHERE po.deleted_at IS NULL
                    AND u.role = 'member'
                    AND (u.deleted_at IS NULL OR u.deleted_at > NOW())
                    AND COALESCE(po.required_amount, 0) > 0";
    $bind_types = '';
    $bind_params = [];
    $type_filter_condition = validate_type_filter($type_filter, $record_type);
    if ($record_type === 'all' || $record_type === 'debts') {
        $sql .= str_replace('created_at', ($date_field === 'due_date' ? 'po.due_date' : 'po.created_at'), build_time_filter($time_period, $date_field, $record_type));
        if ($time_period === 'date') {
            $bind_types .= 's';
            $bind_params[] = $filter_date;
        }
        if ($type_filter && strpos($type_filter, 'payment_') === 0) {
            $adjusted_condition = str_replace('p.payment_type', 'pt.type_name', $type_filter_condition);
            $sql .= $adjusted_condition;
            $bind_types .= 's';
            $bind_params[] = explode('_', $type_filter, 2)[1];
        }
    } else {
        $sql .= ' AND FALSE';
    }
    $sql .= " GROUP BY po.id, po.required_amount 
              HAVING debt > 0
            ) AS subquery";
    if ($stmt = mysqli_prepare($link, $sql)) {
        if ($bind_types) {
            mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
        }
        if (mysqli_stmt_execute($stmt)) {
            $result = mysqli_stmt_get_result($stmt);
            $total = mysqli_fetch_assoc($result)['total_debt'] ?? 0;
            mysqli_free_result($result);
            mysqli_stmt_close($stmt);
            return $total;
        }
        error_log("Failed to execute total_debts query: " . mysqli_stmt_error($stmt) . "; SQL: $sql");
        mysqli_stmt_close($stmt);
    }
    error_log("Failed to prepare total_debts query: " . mysqli_error($link) . "; SQL: $sql");
    return 0;
}

function get_total_members($link) {
    if (user_has_permission($link, 'view_member_directory')) {
        $sql = "SELECT COUNT(*) AS total FROM users WHERE role IN ('member', 'pending') AND (deleted_at IS NULL OR deleted_at > NOW())";
        if ($result = mysqli_query($link, $sql)) {
            $total = mysqli_fetch_assoc($result)['total'] ?? 0;
            mysqli_free_result($result);
            return $total;
        }
        error_log("Failed to fetch members: " . mysqli_error($link) . "; SQL: $sql");
    }
    return 0;
}

function get_total_payment_types($link) {
    $sql = "SELECT COUNT(*) AS total_payment_types FROM payment_types WHERE deleted_at IS NULL";
    if ($result = mysqli_query($link, $sql)) {
        $total = mysqli_fetch_assoc($result)['total_payment_types'] ?? 0;
        mysqli_free_result($result);
        return $total;
    }
    error_log("Failed to execute total_payment_types query: " . mysqli_error($link) . "; SQL: $sql");
    return 0;
}

function get_previous_balances($link, $filter_date, $date_field) {
    $prev_month_start = date('Y-m-01', strtotime('-1 month', strtotime($filter_date)));
    $prev_month_end = date('Y-m-t', strtotime('-1 month', strtotime($filter_date)));
    $prev_year_start = date('Y-01-01', strtotime('-1 year', strtotime($filter_date)));
    $prev_year_end = date('Y-12-31', strtotime('-1 year', strtotime($filter_date)));
    $balances = ['month' => 0, 'year' => 0];

    foreach (['month' => [$prev_month_start, $prev_month_end], 'year' => [$prev_year_start, $prev_year_end]] as $period => $dates) {
        $start_date = $dates[0];
        $end_date = $dates[1];

        $sql_payments = "SELECT COALESCE(SUM(amount), 0) FROM payments WHERE payment_status = 'Paid' AND DATE(created_at) BETWEEN ? AND ?";
        if ($stmt = mysqli_prepare($link, $sql_payments)) {
            mysqli_stmt_bind_param($stmt, 'ss', $start_date, $end_date);
            mysqli_stmt_execute($stmt);
            $prev_payments = mysqli_fetch_array(mysqli_stmt_get_result($stmt))[0] ?? 0;
            mysqli_stmt_close($stmt);
        } else {
            error_log("Failed to prepare prev_payments query for $period: " . mysqli_error($link) . "; SQL: $sql_payments");
            $prev_payments = 0;
        }

        $sql_expenses = "SELECT COALESCE(SUM(amount), 0) FROM expenses WHERE DATE(created_at) BETWEEN ? AND ?";
        if ($stmt = mysqli_prepare($link, $sql_expenses)) {
            mysqli_stmt_bind_param($stmt, 'ss', $start_date, $end_date);
            mysqli_stmt_execute($stmt);
            $prev_expenses = mysqli_fetch_array(mysqli_stmt_get_result($stmt))[0] ?? 0;
            mysqli_stmt_close($stmt);
        } else {
            error_log("Failed to prepare prev_expenses query for $period: " . mysqli_error($link) . "; SQL: $sql_expenses");
            $prev_expenses = 0;
        }

        $sql_benefits = "SELECT COALESCE(SUM(amount), 0) FROM member_benefits WHERE DATE(created_at) BETWEEN ? AND ?";
        if ($stmt = mysqli_prepare($link, $sql_benefits)) {
            mysqli_stmt_bind_param($stmt, 'ss', $start_date, $end_date);
            mysqli_stmt_execute($stmt);
            $prev_benefits = mysqli_fetch_array(mysqli_stmt_get_result($stmt))[0] ?? 0;
            mysqli_stmt_close($stmt);
        } else {
            error_log("Failed to prepare prev_benefits query for $period: " . mysqli_error($link) . "; SQL: $sql_benefits");
            $prev_benefits = 0;
        }

        $sql_debts = "SELECT COALESCE(SUM(debt), 0) AS total_debt 
                      FROM (
                          SELECT 
                              po.id,
                              (COALESCE(po.required_amount, 0) - COALESCE(SUM(p.amount), 0)) AS debt
                          FROM payment_obligations po
                          JOIN users u ON po.user_id = u.id
                          JOIN payment_types pt ON po.payment_type = pt.type_name
                          LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status = 'Paid'
                          WHERE po.deleted_at IS NULL
                              AND u.role = 'member'
                              AND (u.deleted_at IS NULL OR u.deleted_at > NOW())
                              AND COALESCE(po.required_amount, 0) > 0
                              AND DATE(" . ($date_field === 'due_date' ? 'po.due_date' : 'po.created_at') . ") BETWEEN ? AND ?
                          GROUP BY po.id, po.required_amount
                          HAVING debt > 0
                      ) AS subquery";
        if ($stmt = mysqli_prepare($link, $sql_debts)) {
            mysqli_stmt_bind_param($stmt, 'ss', $start_date, $end_date);
            mysqli_stmt_execute($stmt);
            $prev_debts = mysqli_fetch_array(mysqli_stmt_get_result($stmt))[0] ?? 0;
            mysqli_stmt_close($stmt);
        } else {
            error_log("Failed to prepare prev_debts query for $period: " . mysqli_error($link) . "; SQL: $sql_debts");
            $prev_debts = 0;
        }

        $balances[$period] = $prev_payments - $prev_expenses - $prev_benefits - $prev_debts;
    }
    return $balances;
}

function get_type_options($link) {
    $type_options = ['payment' => [], 'expense' => [], 'benefit' => []];
    $sql_payment_types = "SELECT type_name FROM payment_types WHERE deleted_at IS NULL";
    if ($result = mysqli_query($link, $sql_payment_types)) {
        while ($row = mysqli_fetch_assoc($result)) {
            $type_options['payment'][] = $row['type_name'];
        }
        mysqli_free_result($result);
    } else {
        error_log("Failed to execute get_type_options payment_types query: " . mysqli_error($link) . "; SQL: $sql_payment_types");
    }
    $sql_expense_types = "SELECT type_name FROM expense_types WHERE deleted_at IS NULL";
    if ($result = mysqli_query($link, $sql_expense_types)) {
        while ($row = mysqli_fetch_assoc($result)) {
            $type_options['expense'][] = $row['type_name'];
        }
        mysqli_free_result($result);
    } else {
        error_log("Failed to execute get_type_options expense_types query: " . mysqli_error($link) . "; SQL: $sql_expense_types");
    }
    $sql_benefit_types = "SELECT type_name FROM benefit_types WHERE deleted_at IS NULL";
    if ($result = mysqli_query($link, $sql_benefit_types)) {
        while ($row = mysqli_fetch_assoc($result)) {
            $type_options['benefit'][] = $row['type_name'];
        }
        mysqli_free_result($result);
    } else {
        error_log("Failed to execute get_type_options benefit_types query: " . mysqli_error($link) . "; SQL: $sql_benefit_types");
    }
    return $type_options;
}

function get_payments($link, $record_type, $time_period, $filter_date, $type_filter, $search, $pagination) {
    $sql = "SELECT 
                p.id,
                u.member_id,
                IFNULL(u.full_name, 'Unknown') AS full_name,
                p.payment_type,
                p.amount,
                p.payment_status,
                p.created_at,
                po.payment_cycle,
                pt.event_date
             FROM payments p
             JOIN users u ON p.user_id = u.id
             LEFT JOIN payment_obligations po ON p.obligation_id = po.id
             LEFT JOIN payment_types pt ON p.payment_type = pt.type_name
             WHERE payment_status = 'Paid'";
    $bind_types = '';
    $bind_params = [];
    $type_filter_condition = validate_type_filter($type_filter, $record_type);
    if ($record_type === 'all' || $record_type === 'payments') {
        $sql .= str_replace('created_at', 'p.created_at', build_time_filter($time_period, 'created_at', $record_type));
        if ($time_period === 'date') {
            $bind_types .= 's';
            $bind_params[] = $filter_date;
        }
        if ($type_filter && strpos($type_filter, 'payment_') === 0) {
            $sql .= $type_filter_condition;
            $bind_types .= 's';
            $bind_params[] = explode('_', $type_filter, 2)[1];
        }
    } else {
        $sql .= ' AND FALSE';
    }
    if (!empty($search)) {
        $sql .= " AND (u.full_name LIKE ? OR p.payment_type LIKE ? OR CAST(p.id AS CHAR) LIKE ?)";
        $bind_types .= 'sss';
        $search_param = "%$search%";
        $bind_params[] = $search_param;
        $bind_params[] = $search_param;
        $bind_params[] = $search_param;
    }
    $sql .= " ORDER BY p.created_at DESC LIMIT ? OFFSET ?";
    $bind_types .= 'ii';
    $bind_params[] = $pagination['per_page'];
    $bind_params[] = $pagination['offset'];
    $payments = [];
    if ($stmt = mysqli_prepare($link, $sql)) {
        if ($bind_types) {
            mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
        }
        if (mysqli_stmt_execute($stmt)) {
            $result = mysqli_stmt_get_result($stmt);
            while ($row = mysqli_fetch_assoc($result)) {
                $payments[] = $row;
            }
            mysqli_free_result($result);
        } else {
            error_log("Failed to execute get_payments query: " . mysqli_stmt_error($stmt) . "; SQL: $sql");
        }
        mysqli_stmt_close($stmt);
    } else {
        error_log("Failed to prepare get_payments query: " . mysqli_error($link) . "; SQL: $sql");
    }
    return $payments;
}

function get_expenses($link, $record_type, $time_period, $filter_date, $type_filter, $search, $pagination) {
    $sql = "SELECT 
                e.id,
                et.type_name AS expense_type,
                e.amount,
                IFNULL(e.description, 'N/A') AS description,
                e.created_at
             FROM expenses e
             LEFT JOIN expense_types et ON e.expense_type_id = et.id
             WHERE 1=1";
    $bind_types = '';
    $bind_params = [];
    $type_filter_condition = validate_type_filter($type_filter, $record_type);
    if ($record_type === 'all' || $record_type === 'expenses') {
        $sql .= str_replace('created_at', 'e.created_at', build_time_filter($time_period, 'created_at', $record_type));
        if ($time_period === 'date') {
            $bind_types .= 's';
            $bind_params[] = $filter_date;
        }
        if ($type_filter && strpos($type_filter, 'expense_') === 0) {
            $sql .= $type_filter_condition;
            $bind_types .= 's';
            $bind_params[] = explode('_', $type_filter, 2)[1];
        }
    } else {
        $sql .= ' AND FALSE';
    }
    if (!empty($search)) {
        $sql .= " AND (et.type_name LIKE ? OR e.description LIKE ? OR CAST(e.id AS CHAR) LIKE ?)";
        $bind_types .= 'sss';
        $search_param = "%$search%";
        $bind_params[] = $search_param;
        $bind_params[] = $search_param;
        $bind_params[] = $search_param;
    }
    $sql .= " ORDER BY e.created_at DESC LIMIT ? OFFSET ?";
    $bind_types .= 'ii';
    $bind_params[] = $pagination['per_page'];
    $bind_params[] = $pagination['offset'];
    $expenses = [];
    if ($stmt = mysqli_prepare($link, $sql)) {
        if ($bind_types) {
            mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
        }
        if (mysqli_stmt_execute($stmt)) {
            $result = mysqli_stmt_get_result($stmt);
            while ($row = mysqli_fetch_assoc($result)) {
                $expenses[] = $row;
            }
            mysqli_free_result($result);
        } else {
            error_log("Failed to execute get_expenses query: " . mysqli_stmt_error($stmt) . "; SQL: $sql");
        }
        mysqli_stmt_close($stmt);
    } else {
        error_log("Failed to prepare get_expenses query: " . mysqli_error($link) . "; SQL: $sql");
    }
    return $expenses;
}

function get_benefits($link, $record_type, $time_period, $filter_date, $type_filter, $search, $pagination) {
    $sql = "SELECT 
                b.id,
                u.member_id,
                IFNULL(u.full_name, 'Unknown') AS full_name,
                bt.type_name AS benefit_type,
                b.amount,
                b.created_at
             FROM member_benefits b
             JOIN users u ON b.member_id = u.id
             JOIN benefit_types bt ON b.benefit_type_id = bt.id
             WHERE 1=1";
    $bind_types = '';
    $bind_params = [];
    $type_filter_condition = validate_type_filter($type_filter, $record_type);
    if ($record_type === 'all' || $record_type === 'benefits') {
        $sql .= str_replace('created_at', 'b.created_at', build_time_filter($time_period, 'created_at', $record_type));
        if ($time_period === 'date') {
            $bind_types .= 's';
            $bind_params[] = $filter_date;
        }
        if ($type_filter && strpos($type_filter, 'benefit_') === 0) {
            $sql .= $type_filter_condition;
            $bind_types .= 's';
            $bind_params[] = explode('_', $type_filter, 2)[1];
        }
    } else {
        $sql .= ' AND FALSE';
    }
    if (!empty($search)) {
        $sql .= " AND (u.full_name LIKE ? OR bt.type_name LIKE ? OR CAST(b.id AS CHAR) LIKE ?)";
        $bind_types .= 'sss';
        $search_param = "%$search%";
        $bind_params[] = $search_param;
        $bind_params[] = $search_param;
        $bind_params[] = $search_param;
    }
    $sql .= " ORDER BY b.created_at DESC LIMIT ? OFFSET ?";
    $bind_types .= 'ii';
    $bind_params[] = $pagination['per_page'];
    $bind_params[] = $pagination['offset'];
    $benefits = [];
    if ($stmt = mysqli_prepare($link, $sql)) {
        if ($bind_types) {
            mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
        }
        if (mysqli_stmt_execute($stmt)) {
            $result = mysqli_stmt_get_result($stmt);
            while ($row = mysqli_fetch_assoc($result)) {
                $benefits[] = $row;
            }
            mysqli_free_result($result);
        } else {
            error_log("Failed to execute get_benefits query: " . mysqli_stmt_error($stmt) . "; SQL: $sql");
        }
        mysqli_stmt_close($stmt);
    } else {
        error_log("Failed to prepare get_benefits query: " . mysqli_error($link) . "; SQL: $sql");
    }
    return $benefits;
}

function get_debts($link, $record_type, $time_period, $filter_date, $type_filter, $search, $date_field, $pagination) {
    $sql = "SELECT 
                u.id AS user_id,
                u.member_id,
                IFNULL(u.full_name, 'Unknown') AS full_name,
                pt.type_name AS payment_type,
                COALESCE(po.required_amount, 0) AS required_amount,
                COALESCE(SUM(p.amount), 0) AS total_paid,
                (COALESCE(po.required_amount, 0) - COALESCE(SUM(p.amount), 0)) AS debt,
                po.payment_cycle,
                po.due_date
             FROM payment_obligations po
             JOIN users u ON po.user_id = u.id
             JOIN payment_types pt ON po.payment_type = pt.type_name
             LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status = 'Paid'
             WHERE po.deleted_at IS NULL
                 AND u.role = 'member'
                 AND (u.deleted_at IS NULL OR u.deleted_at > NOW())
                 AND COALESCE(po.required_amount, 0) > 0";
    $bind_types = '';
    $bind_params = [];
    $type_filter_condition = validate_type_filter($type_filter, $record_type);
    if ($record_type === 'all' || $record_type === 'debts') {
        $sql .= str_replace('created_at', ($date_field === 'due_date' ? 'po.due_date' : 'po.created_at'), build_time_filter($time_period, $date_field, $record_type));
        if ($time_period === 'date') {
            $bind_types .= 's';
            $bind_params[] = $filter_date;
        }
        if ($type_filter && strpos($type_filter, 'payment_') === 0) {
            $adjusted_condition = str_replace('p.payment_type', 'pt.type_name', $type_filter_condition);
            $sql .= $adjusted_condition;
            $bind_types .= 's';
            $bind_params[] = explode('_', $type_filter, 2)[1];
        }
    } else {
        $sql .= ' AND FALSE';
    }
    if (!empty($search)) {
        $sql .= " AND (u.full_name LIKE ? OR pt.type_name LIKE ? OR CAST(po.id AS CHAR) LIKE ?)";
        $bind_types .= 'sss';
        $search_param = "%$search%";
        $bind_params[] = $search_param;
        $bind_params[] = $search_param;
        $bind_params[] = $search_param;
    }
    $sql .= " GROUP BY po.id, u.id, po.required_amount, po.payment_cycle, po.due_date
             HAVING debt > 0
             ORDER BY debt DESC
             LIMIT ? OFFSET ?";
    $bind_types .= 'ii';
    $bind_params[] = $pagination['per_page'];
    $bind_params[] = $pagination['offset'];
    $debts = [];
    if ($stmt = mysqli_prepare($link, $sql)) {
        if ($bind_types) {
            mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
        }
        if (mysqli_stmt_execute($stmt)) {
            $result = mysqli_stmt_get_result($stmt);
            while ($row = mysqli_fetch_assoc($result)) {
                $debts[] = $row;
            }
            mysqli_free_result($result);
        } else {
            error_log("Failed to execute get_debts query: " . mysqli_stmt_error($stmt) . "; SQL: $sql");
        }
        mysqli_stmt_close($stmt);
    } else {
        error_log("Failed to prepare get_debts query: " . mysqli_error($link) . "; SQL: $sql");
    }
    return $debts;
}

function get_payment_types_summary($link, $record_type, $time_period, $filter_date, $type_filter, $date_field) {
    $sql = "SELECT 
                pt.type_name,
                SUM(COALESCE(po.required_amount, 0)) AS required_amount,
                COALESCE(SUM(p.amount), 0) AS total_paid,
                SUM(COALESCE(po.required_amount, 0) - COALESCE(p.amount, 0)) AS total_debt
            FROM payment_obligations po
            JOIN users u ON po.user_id = u.id
            JOIN payment_types pt ON po.payment_type = pt.type_name
            LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status = 'Paid'
            WHERE po.deleted_at IS NULL
                AND u.role = 'member'
                AND (u.deleted_at IS NULL OR u.deleted_at > NOW())
                AND COALESCE(po.required_amount, 0) > 0";
    $bind_types = '';
    $bind_params = [];
    $type_filter_condition = validate_type_filter($type_filter, $record_type);
    if ($record_type === 'all' || $record_type === 'payments') {
        $sql .= str_replace('created_at', ($date_field === 'due_date' ? 'po.due_date' : 'po.created_at'), build_time_filter($time_period, $date_field, $record_type));
        if ($time_period === 'date') {
            $bind_types .= 's';
            $bind_params[] = $filter_date;
        }
        if ($type_filter && strpos($type_filter, 'payment_') === 0) {
            $adjusted_condition = str_replace('p.payment_type', 'pt.type_name', $type_filter_condition);
            $sql .= $adjusted_condition;
            $bind_types .= 's';
            $bind_params[] = explode('_', $type_filter, 2)[1];
        }
    } else {
        $sql .= ' AND FALSE';
    }
    $sql .= " GROUP BY pt.type_name
              HAVING total_debt > 0";
    $summary = [];
    if ($stmt = mysqli_prepare($link, $sql)) {
        if ($bind_types) {
            mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
        }
        if (mysqli_stmt_execute($stmt)) {
            $result = mysqli_stmt_get_result($stmt);
            while ($row = mysqli_fetch_assoc($result)) {
                $summary[] = $row;
            }
            mysqli_free_result($result);
        } else {
            error_log("Failed to execute get_payment_types_summary query: " . mysqli_stmt_error($stmt) . "; SQL: $sql");
        }
        mysqli_stmt_close($stmt);
    } else {
        error_log("Failed to prepare get_payment_types_summary query: " . mysqli_error($link) . "; SQL: $sql");
    }
    return $summary;
}

function get_expense_types_summary($link, $record_type, $time_period, $filter_date, $type_filter) {
    $sql = "SELECT et.type_name, COALESCE(SUM(e.amount), 0) AS total
            FROM expense_types et
            LEFT JOIN expenses e ON e.expense_type_id = et.id";
    $bind_types = '';
    $bind_params = [];
    $type_filter_condition = validate_type_filter($type_filter, $record_type);
    $where = ($record_type === 'all' || $record_type === 'expenses') ? str_replace('created_at', 'e.created_at', build_time_filter($time_period, 'created_at', $record_type)) : ' AND FALSE';
    if ($time_period === 'date') {
        $bind_types .= 's';
        $bind_params[] = $filter_date;
    }
    if ($type_filter && strpos($type_filter, 'expense_') === 0) {
        $where .= $type_filter_condition;
        $bind_types .= 's';
        $bind_params[] = explode('_', $type_filter, 2)[1];
    }
    $sql .= $where . " GROUP BY et.type_name";
    $summary = [];
    if ($stmt = mysqli_prepare($link, $sql)) {
        if ($bind_types) {
            mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
        }
        if (mysqli_stmt_execute($stmt)) {
            $result = mysqli_stmt_get_result($stmt);
            while ($row = mysqli_fetch_assoc($result)) {
                $summary[] = $row;
            }
            mysqli_free_result($result);
        } else {
            error_log("Failed to execute get_expense_types_summary query: " . mysqli_stmt_error($stmt) . "; SQL: $sql");
        }
        mysqli_stmt_close($stmt);
    } else {
        error_log("Failed to prepare get_expense_types_summary query: " . mysqli_error($link) . "; SQL: $sql");
    }
    return $summary;
}

function get_benefit_types_summary($link, $record_type, $time_period, $filter_date, $type_filter) {
    $sql = "SELECT bt.type_name, COALESCE(SUM(b.amount), 0) AS total
            FROM benefit_types bt
            LEFT JOIN member_benefits b ON b.benefit_type_id = bt.id";
    $bind_types = '';
    $bind_params = [];
    $type_filter_condition = validate_type_filter($type_filter, $record_type);
    $where = ($record_type === 'all' || $record_type === 'benefits') ? str_replace('created_at', 'b.created_at', build_time_filter($time_period, 'created_at', $record_type)) : ' AND FALSE';
    if ($time_period === 'date') {
        $bind_types .= 's';
        $bind_params[] = $filter_date;
    }
    if ($type_filter && strpos($type_filter, 'benefit_') === 0) {
        $where .= $type_filter_condition;
        $bind_types .= 's';
        $bind_params[] = explode('_', $type_filter, 2)[1];
    }
    $sql .= $where . " GROUP BY bt.type_name";
    $summary = [];
    if ($stmt = mysqli_prepare($link, $sql)) {
        if ($bind_types) {
            mysqli_stmt_bind_param($stmt, $bind_types, ...$bind_params);
        }
        if (mysqli_stmt_execute($stmt)) {
            $result = mysqli_stmt_get_result($stmt);
            while ($row = mysqli_fetch_assoc($result)) {
                $summary[] = $row;
            }
            mysqli_free_result($result);
        } else {
            error_log("Failed to execute get_benefit_types_summary query: " . mysqli_stmt_error($stmt) . "; SQL: $sql");
        }
        mysqli_stmt_close($stmt);
    } else {
        error_log("Failed to prepare get_benefit_types_summary query: " . mysqli_error($link) . "; SQL: $sql");
    }
    return $summary;
}

function get_line_chart_data($link, $filter_date, $date_field) {
    $data = ['labels' => [], 'payments' => [], 'expenses' => [], 'benefits' => [], 'debts' => []];
    $current = new DateTime(date('Y-m-01', strtotime('-11 months', strtotime($filter_date))));
    $end = new DateTime(date('Y-m-t', strtotime($filter_date)));
    while ($current <= $end) {
        $month_start = $current->format('Y-m-01');
        $month_end = $current->format('Y-m-t');
        $data['labels'][] = $current->format('M Y');

        $sql = "SELECT COALESCE(SUM(amount), 0) FROM payments WHERE payment_status = 'Paid' AND DATE(created_at) BETWEEN ? AND ?";
        if ($stmt = mysqli_prepare($link, $sql)) {
            mysqli_stmt_bind_param($stmt, 'ss', $month_start, $month_end);
            mysqli_stmt_execute($stmt);
            $data['payments'][] = mysqli_fetch_array(mysqli_stmt_get_result($stmt))[0] ?? 0;
            mysqli_stmt_close($stmt);
        } else {
            error_log("Failed to prepare get_line_chart_data payments query: " . mysqli_error($link) . "; SQL: $sql");
        }

        $sql = "SELECT COALESCE(SUM(amount), 0) FROM expenses WHERE DATE(created_at) BETWEEN ? AND ?";
        if ($stmt = mysqli_prepare($link, $sql)) {
            mysqli_stmt_bind_param($stmt, 'ss', $month_start, $month_end);
            mysqli_stmt_execute($stmt);
            $data['expenses'][] = mysqli_fetch_array(mysqli_stmt_get_result($stmt))[0] ?? 0;
            mysqli_stmt_close($stmt);
        } else {
            error_log("Failed to prepare get_line_chart_data expenses query: " . mysqli_error($link) . "; SQL: $sql");
        }

        $sql = "SELECT COALESCE(SUM(amount), 0) FROM member_benefits WHERE DATE(created_at) BETWEEN ? AND ?";
        if ($stmt = mysqli_prepare($link, $sql)) {
            mysqli_stmt_bind_param($stmt, 'ss', $month_start, $month_end);
            mysqli_stmt_execute($stmt);
            $data['benefits'][] = mysqli_fetch_array(mysqli_stmt_get_result($stmt))[0] ?? 0;
            mysqli_stmt_close($stmt);
        } else {
            error_log("Failed to prepare get_line_chart_data benefits query: " . mysqli_error($link) . "; SQL: $sql");
        }

        $sql = "SELECT COALESCE(SUM(debt), 0) AS total_debt 
                FROM (
                    SELECT 
                        po.id,
                        (COALESCE(po.required_amount, 0) - COALESCE(SUM(p.amount), 0)) AS debt
                    FROM payment_obligations po
                    JOIN users u ON po.user_id = u.id
                    JOIN payment_types pt ON po.payment_type = pt.type_name
                    LEFT JOIN payments p ON p.obligation_id = po.id AND p.payment_status = 'Paid'
                    WHERE po.deleted_at IS NULL
                        AND u.role = 'member'
                        AND (u.deleted_at IS NULL OR u.deleted_at > NOW())
                        AND COALESCE(po.required_amount, 0) > 0
                        AND DATE(" . ($date_field === 'due_date' ? 'po.due_date' : 'po.created_at') . ") BETWEEN ? AND ?
                    GROUP BY po.id, po.required_amount
                    HAVING debt > 0
                ) AS subquery";
        if ($stmt = mysqli_prepare($link, $sql)) {
            mysqli_stmt_bind_param($stmt, 'ss', $month_start, $month_end);
            mysqli_stmt_execute($stmt);
            $data['debts'][] = mysqli_fetch_array(mysqli_stmt_get_result($stmt))[0] ?? 0;
            mysqli_stmt_close($stmt);
        } else {
            error_log("Failed to prepare get_line_chart_data debts query: " . mysqli_error($link) . "; SQL: $sql");
        }

        $current->modify('+1 month');
    }
    return $data;
}

function test_payment_type_column($link) {
    $sql = "SELECT po.payment_type FROM payment_obligations po LIMIT 1";
    if ($stmt = mysqli_prepare($link, $sql)) {
        if (mysqli_stmt_execute($stmt)) {
            $result = mysqli_stmt_get_result($stmt);
            $row = mysqli_fetch_assoc($result);
            mysqli_free_result($result);
            mysqli_stmt_close($stmt);
            return $row['payment_type'] ?? 'No data';
        } else {
            error_log("Test query execution failed: " . mysqli_stmt_error($stmt) . "; SQL: $sql");
            mysqli_stmt_close($stmt);
            return 'Execution failed';
        }
    } else {
        error_log("Test query preparation failed: " . mysqli_error($link) . "; SQL: $sql");
        return 'Preparation failed';
    }
}
?>