<?php
// This API fetches detailed member financial data for the main table.
require_once 'includes/functions.php';
require_once 'includes/db_config.php';
session_start();

// Check if user is logged in and is an admin
if (!is_logged_in() || !is_admin()) {
    redirect("login.php");
}

$currency_symbol = get_setting('currency_symbol', $link);
?>

<?php require_once 'includes/header.php'; ?>
<div class="container mt-4">
    <h2>Member Financials</h2>
    <hr>
    
    <div class="row mb-4">
        <div class="col-md-4">
            <div class="card text-white bg-success mb-3">
                <div class="card-header">Total Paid (All Time)</div>
                <div class="card-body">
                    <h5 class="card-title"><span id="total_paid">...</span></h5>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="card text-white bg-info mb-3">
                <div class="card-header">Payments This Month</div>
                <div class="card-body">
                    <h5 class="card-title"><span id="paid_this_month">...</span></h5>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="card text-white bg-warning mb-3">
                <div class="card-header">Outstanding Debtors</div>
                <div class="card-body">
                    <h5 class="card-title"><span id="outstanding_debtors">...</span></h5>
                </div>
            </div>
        </div>
    </div>

    <div class="card mb-4">
        <div class="card-header">Search & Filter Financials</div>
        <div class="card-body">
            <div class="row">
                <div class="col-md-6 mb-3">
                    <input type="text" id="searchInput" class="form-control" placeholder="Search by Member ID or Name...">
                </div>
                <div class="col-md-6 mb-3">
                    <select id="periodFilter" class="form-control">
                        <option value="all">All Time</option>
                        <option value="today">Today</option>
                        <option value="this_week">This Week</option>
                        <option value="this_month">This Month</option>
                        <option value="this_year">This Year</option>
                    </select>
                </div>
            </div>
        </div>
    </div>
    
    <div class="card mb-4">
        <div class="card-header d-flex justify-content-between align-items-center">
            <span>Financial Transactions</span>
            <div>
                <button class="btn btn-sm btn-outline-secondary" onclick="printTable('financialsTable')">Print</button>
                <a href="api/export_financials.php" class="btn btn-sm btn-outline-success">Export All to CSV</a>
            </div>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-striped table-hover" id="financialsTable">
                    <thead>
                        <tr>
                            <th>Member ID</th>
                            <th>Member Name</th>
                            <th>Payment Type</th>
                            <th>Amount Paid</th>
                            <th>Payment Status</th>
                            <th>Membership Status</th>
                        </tr>
                    </thead>
                    <tbody id="financialsTableBody">
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    
    <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
            <span>Payment Types Summary</span>
            <button class="btn btn-sm btn-outline-secondary" onclick="printTable('paymentSummaryTable')">Print</button>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-striped table-hover" id="paymentSummaryTable">
                    <tbody id="paymentSummaryTableBody">
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', () => {
    // Initial data load
    fetchFinancialData();

    // Event listeners for search and filters
    document.getElementById('searchInput').addEventListener('input', () => {
        fetchFinancialData();
    });
    document.getElementById('periodFilter').addEventListener('change', () => {
        fetchFinancialData();
    });
});

async function fetchFinancialData() {
    const currencySymbol = '<?php echo $currency_symbol; ?>';
    const search = document.getElementById('searchInput').value;
    const period = document.getElementById('periodFilter').value;
    const url = `/club_portal/api/get_member_financials.php?search=${search}&period=${period}`;
    
    try {
        const response = await fetch(url);
        const data = await response.json();

        if (data.success) {
            // Populate the summary cards
            document.getElementById('total_paid').textContent = currencySymbol + ' ' + data.summary.total_paid;
            document.getElementById('paid_this_month').textContent = currencySymbol + ' ' + data.summary.paid_this_month;
            document.getElementById('outstanding_debtors').textContent = data.summary.outstanding_debtors;

            // Populate the financial transactions table
            const financialsTableBody = document.getElementById('financialsTableBody');
            financialsTableBody.innerHTML = ''; // Clear existing rows
            if (data.transactions.length > 0) {
                data.transactions.forEach(member => {
                    const row = `
                        <tr>
                            <td>${member.member_id}</td>
                            <td>${member.full_name}</td>
                            <td>${member.payment_type}</td>
                            <td>${currencySymbol} ${member.amount_paid}</td>
                            <td><span class="badge bg-${member.payment_status === 'Paid' ? 'success' : 'danger'}">${member.payment_status}</span></td>
                            <td>${member.membership_status}</td>
                        </tr>
                    `;
                    financialsTableBody.insertAdjacentHTML('beforeend', row);
                });
            } else {
                const row = `<tr><td colspan="6" class="text-center">No financial data found.</td></tr>`;
                financialsTableBody.insertAdjacentHTML('beforeend', row);
            }

            // Populate the payment types summary table
            const paymentSummaryTableBody = document.getElementById('paymentSummaryTableBody');
            paymentSummaryTableBody.innerHTML = ''; // Clear existing rows
            const summaryByType = data.payment_types_summary;
            if (Object.keys(summaryByType).length > 0) {
                for (const type in summaryByType) {
                    const summary = summaryByType[type];
                    
                    // Group header row
                    const headerRow = `
                        <tr class="table-primary">
                            <th colspan="4">${type} <span class="text-muted">(Total Paid: ${currencySymbol} ${summary.total_paid})</span></th>
                        </tr>
                    `;
                    paymentSummaryTableBody.insertAdjacentHTML('beforeend', headerRow);

                    // Members list
                    if (summary.members.length > 0) {
                        summary.members.forEach(member => {
                            const memberRow = `
                                <tr>
                                    <td>${member.member_id}</td>
                                    <td>${member.full_name}</td>
                                    <td>${member.amount !== 'N/A' ? `${currencySymbol} ${member.amount}` : 'N/A'}</td>
                                    <td></td>
                                </tr>
                            `;
                            paymentSummaryTableBody.insertAdjacentHTML('beforeend', memberRow);
                        });
                    } else {
                        const noMembersRow = `<tr><td colspan="4" class="text-muted text-center">No members in this category.</td></tr>`;
                        paymentSummaryTableBody.insertAdjacentHTML('beforeend', noMembersRow);
                    }
                }
            } else {
                paymentSummaryTableBody.innerHTML = `<tr><td colspan="4" class="text-center">No payment type summary found.</td></tr>`;
            }
        } else {
            // Handle API failure
            console.error('API Error:', data.message);
            document.getElementById('financialsTableBody').innerHTML = '<tr><td colspan="6" class="text-center">Failed to load data.</td></tr>';
            document.getElementById('paymentSummaryTableBody').innerHTML = `<tr><td colspan="4" class="text-center">Failed to load summary.</td></tr>`;
        }
    } catch (error) {
        console.error('Error fetching financial data:', error);
        document.getElementById('financialsTableBody').innerHTML = '<tr><td colspan="6" class="text-center">Failed to load data.</td></tr>';
        document.getElementById('paymentSummaryTableBody').innerHTML = `<tr><td colspan="4" class="text-center">Failed to load summary.</td></tr>`;
    }
}

function printTable(tableId) {
    const table = document.getElementById(tableId);
    const originalContent = document.body.innerHTML;
    const printContent = `<style>
        body { font-family: sans-serif; }
        table { width: 100%; border-collapse: collapse; margin-bottom: 20px; }
        th, td { border: 1px solid #ccc; padding: 8px; text-align: left; }
        th { background-color: #f2f2f2; }
        h2, h3 { text-align: center; }
        .table-primary th { background-color: #b8daff; }
        @media print {
            .no-print { display: none; }
        }
    </style>
    <div class="no-print"><h2>Financial Report</h2></div>
    <h3>${tableId === 'financialsTable' ? 'Financial Transactions' : 'Payment Types Summary'}</h3>
    ${table.outerHTML}`;

    document.body.innerHTML = printContent;
    window.print();
    document.body.innerHTML = originalContent;
    window.location.reload(); // Reload to restore page state
}
</script>

<?php require_once 'includes/footer.php'; ?>
