<?php
require_once 'includes/db_config.php';
session_start();

if (!is_logged_in() || !is_admin()) {
    redirect("login.php");
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $user_id = $_POST['user_id'];
    $payment_type = $_POST['payment_type'];
    $amount = $_POST['amount'];
    $payment_status = 'Paid';

    // Fetch payment_category and recurrence_interval
    $sql = "SELECT payment_category, recurrence_interval, event_date FROM payment_types WHERE type_name = ? AND deleted_at IS NULL";
    $stmt = mysqli_prepare($link, $sql);
    mysqli_stmt_bind_param($stmt, "s", $payment_type);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    $type = mysqli_fetch_assoc($result);
    mysqli_stmt_close($stmt);

    if (!$type) {
        die("Invalid payment type.");
    }

    $payment_category = $type['payment_category'];
    $recurrence_interval = $type['recurrence_interval'];
    $event_date = $type['event_date'];

    // Determine payment_cycle or event_date
    $payment_cycle = null;
    $payment_event_date = null;
    if ($payment_category === 'recurring') {
        $payment_cycle = match ($recurrence_interval) {
            'weekly' => date('Y-m-d', strtotime('monday this week')),
            'monthly' => date('Y-m'),
            'yearly' => date('Y'),
            default => null,
        };
    } elseif ($payment_category === 'event') {
        $payment_event_date = $event_date ?? $_POST['event_date'];
    } elseif ($payment_category === 'freewill') {
        $payment_cycle = $_POST['payment_cycle'] ?? null; // Optional cycle for freewill
        $payment_event_date = $_POST['event_date'] ?? null; // Optional event date
    }

    // Insert payment
    $sql = "INSERT INTO payments (user_id, payment_type, amount, payment_status, created_at, payment_cycle, event_date)
            VALUES (?, ?, ?, ?, NOW(), ?, ?)";
    $stmt = mysqli_prepare($link, $sql);
    mysqli_stmt_bind_param($stmt, "isdsss", $user_id, $payment_type, $amount, $payment_status, $payment_cycle, $payment_event_date);
    mysqli_stmt_execute($stmt);
    mysqli_stmt_close($stmt);

    redirect("financial_summary.php");
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Add Payment</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body>
<div class="container mt-5">
    <h2>Add Payment</h2>
    <form method="POST" id="paymentForm">
        <div class="mb-3">
            <label for="user_id" class="form-label">Member</label>
            <select name="user_id" id="user_id" class="form-select" required>
                <?php
                $sql = "SELECT id, full_name FROM users WHERE role = 'member' AND (deleted_at IS NULL OR deleted_at > NOW())";
                $result = mysqli_query($link, $sql);
                while ($user = mysqli_fetch_assoc($result)) {
                    echo "<option value='{$user['id']}'>" . htmlspecialchars($user['full_name']) . "</option>";
                }
                ?>
            </select>
        </div>
        <div class="mb-3">
            <label for="payment_type" class="form-label">Payment Type</label>
            <select name="payment_type" id="payment_type" class="form-select" required onchange="updateFields()">
                <?php
                $sql = "SELECT type_name, payment_category, event_date FROM payment_types WHERE deleted_at IS NULL";
                $result = mysqli_query($link, $sql);
                while ($type = mysqli_fetch_assoc($result)) {
                    echo "<option value='{$type['type_name']}' data-category='{$type['payment_category']}' data-event-date='{$type['event_date']}'>" . htmlspecialchars($type['type_name']) . "</option>";
                }
                ?>
            </select>
        </div>
        <div class="mb-3">
            <label for="amount" class="form-label">Amount</label>
            <input type="number" name="amount" id="amount" class="form-control" step="0.01" required>
        </div>
        <div class="mb-3" id="payment_cycle_container" style="display: none;">
            <label for="payment_cycle" class="form-label">Payment Cycle</label>
            <input type="text" name="payment_cycle" id="payment_cycle" class="form-control">
        </div>
        <div class="mb-3" id="event_date_container" style="display: none;">
            <label for="event_date" class="form-label">Event Date</label>
            <input type="date" name="event_date" id="event_date" class="form-control">
        </div>
        <button type="submit" class="btn btn-primary">Submit Payment</button>
    </form>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
function updateFields() {
    const paymentTypeSelect = document.getElementById('payment_type');
    const selectedOption = paymentTypeSelect.options[paymentTypeSelect.selectedIndex];
    const category = selectedOption.getAttribute('data-category');
    const eventDate = selectedOption.getAttribute('data-event-date');

    const cycleContainer = document.getElementById('payment_cycle_container');
    const eventContainer = document.getElementById('event_date_container');
    const cycleInput = document.getElementById('payment_cycle');
    const eventInput = document.getElementById('event_date');

    cycleContainer.style.display = 'none';
    eventContainer.style.display = 'none';
    cycleInput.value = '';
    eventInput.value = '';

    if (category === 'recurring') {
        cycleContainer.style.display = 'block';
        const today = new Date();
        cycleInput.value = '<?php echo date('Y-m'); ?>'; // Default to current cycle
    } else if (category === 'event' && eventDate) {
        eventContainer.style.display = 'block';
        eventInput.value = eventDate.split(' ')[0]; // Pre-fill event date
        eventInput.readOnly = true;
    } else if (category === 'freewill') {
        cycleContainer.style.display = 'block';
        eventContainer.style.display = 'block';
    }
}
</script>
</body>
</html>
<?php mysqli_close($link); ?>